
import sys
import os
import numpy as np

from osgeo import ogr, osr, gdal, gdalnumeric

from shapely.geometry import Point
from shapely.geometry import Polygon
from shapely import wkb
from shapely import wkt

import pandas as pd

READ_ONLY =0
WRITE = 1

# create a dictionary of WKB Geometry Types names for OGR
# WKBGeometryTypes are a enumerated value. that starts with the
# prefix 'wkb'. This dictionary allows us to use the numeric value
# returned by .GetGeomType() to find the type name, with the 'wkb' removed
__WKBGeometryTypes__ = {}
_ignore = ['wkb25DBit', 'wkb25Bit', 'wkbXDR', 'wkbNDR']
for c in filter(lambda x: x.startswith('wkb'), dir(ogr)):
    if c not in _ignore:
        __WKBGeometryTypes__[ogr.__dict__[c]] = c[3:]

__PhrLayerCapabilitiesConst__ = {}

class GDALHelper():
    @staticmethod
    def __get_layer_capabilities__(layer):
        capabilities = [
            ogr.OLCRandomRead,
            ogr.OLCSequentialWrite,
            ogr.OLCRandomWrite,
            ogr.OLCFastSpatialFilter,
            ogr.OLCFastFeatureCount,
            ogr.OLCFastGetExtent,
            ogr.OLCCreateField,
            ogr.OLCDeleteField,
            ogr.OLCReorderFields,
            ogr.OLCAlterFieldDefn,
            ogr.OLCTransactions,
            ogr.OLCDeleteFeature,
            ogr.OLCFastSetNextByIndex,
            ogr.OLCStringsAsUTF8,
            ogr.OLCIgnoreFields
        ]
        lyr_capabilities = {}
        for cap in capabilities:
            lyr_capabilities[cap] =  layer.TestCapability(cap)
        return lyr_capabilities

    @staticmethod
    def __geom_str__(geom):
        """Get a geometry string for printing attributes."""
        if geom.GetGeometryType() == ogr.wkbPoint:
            return 'POINT ({:.3f}, {:.3f})'.format(geom.GetX(), geom.GetY())
        else:
            return geom.GetGeometryName()

    @staticmethod
    def __get_atts__(feature, fields, geom):
        """Get attribute values from a feature."""
        data = [feature.GetFID()]
        geometry = feature.geometry()
        if geom and geometry:
            data.append(GDALHelper.__geom_str__(geometry))
        values = feature.items()
        data += [values[field] for field in fields]
        return data

    @staticmethod
    def __get_df__atts__(feature, fields, geom):
        """Get attribute values from a feature."""
        data = [feature.GetFID()]
        geometry = feature.geometry()
        if geom and geometry:
            data.append(geometry.ExportToWkt())
        values = feature.items()
        data += [values[field] for field in fields]
        return data

    @staticmethod
    def __get_layers__(lyr_or_fn):
        """Get the data source and layer from a filename."""
        if not isinstance(lyr_or_fn, ogr.Layer):
            if type(lyr_or_fn) is str:
                ds = ogr.Open(lyr_or_fn)
                if ds is None:
                    raise OSError('Could not open {0}.'.format(lyr_or_fn))
            return ds.GetLayer(), ds
        else:
            return lyr_or_fn, None

    @staticmethod
    def __get_capabilities__(item, name, prefix):
        """Print capabilities for a driver, data source, or layer.

        item   - item to test
        name   - name of the type of item
        prefix - prefix of the ogr constants to use for testing
        """
        capability_dict = {}
        for c in filter(lambda x: x.startswith(prefix), dir(ogr)):
            capability_dict[c] =  item.TestCapability(ogr.__dict__[c])
        return capability_dict

    @staticmethod
    def get_ogr_drivers():
        """
        Returns a sorted list of ogr drivers. The list consist of tuples of driver name,
        whether driver is 'writable': True is writable, False is read only
        and whether the driver can delete the source: True is deletable, False not deletable
        """
        cnt = ogr.GetDriverCount()
        formatsList = []  # Empty List

        for i in range(cnt):
            driver = ogr.GetDriver(i)
            driverName = driver.GetName()
            writable = driver.TestCapability(ogr.ODrCCreateDataSource)
            deletable = driver.TestCapability(ogr.ODrCDeleteDataSource)
            if  len( [item for item in formatsList if item[0] == driverName]) == 0:
                formatsList.append((driverName, writable, deletable))

        formatsList.sort(key=lambda tup: tup[0].lower()) # Sorting the messy list of gdal drivers
        return formatsList

    @staticmethod
    def get_gdal_drivers():
        """
        Returns a sorted list of gdal raster drivers. The list consist of tuples of driver name,
        whether driver is 'writable': True is writable, False is read only
        and whether the driver can delete the source: True is deletable, False not deletable
        """
        cnt = gdal.GetDriverCount()
        formatsList = []  # Empty List

        for i in range(cnt):
            driver = gdal.GetDriver(i)
            driverName = driver.ShortName
            """
            Unlike ogr drivers which have to deal with proprietary formats and
            license agreements, the gdal raster drivers all have the ability
            to create and delete
            """
            writable = True
            deletable = True

            if  len( [item for item in formatsList if item[0] == driverName]) == 0:
                formatsList.append((driverName, writable, deletable))

        formatsList.sort(key=lambda tup: tup[0].lower()) # Sorting the messy list of drivers
        return formatsList

    @staticmethod
    def get_drivers():
        """
        Returns a sorted list of gogr and gdal raster drivers. The list consist of tuples of driver name, whether driver is and ogr or gdal driver ('ogr', 'gdal'), whether the driver 'writable': True is writable, False is read only and whether the driver can delete the source: True is deletable, False not deletable
        """

        formatsList = []  # Empty List

        cnt = ogr.GetDriverCount()
        driverType='ogr'
        for i in range(cnt):
            driver = ogr.GetDriver(i)
            driverName = driver.GetName()
            writable = driver.TestCapability(ogr.ODrCCreateDataSource)
            deletable = driver.TestCapability(ogr.ODrCDeleteDataSource)
            if  len( [item for item in formatsList if item[0] == driverName]) == 0:
                formatsList.append((driverName, driverType,writable, deletable))

        cnt = gdal.GetDriverCount()
        driverType='gdal'
        for i in range(cnt):
            driver = gdal.GetDriver(i)
            driverName = driver.ShortName
            writable = True
            deletable = True

            if  len( [item for item in formatsList if item[0] == driverName]) == 0:
                formatsList.append((driverName, driverType,writable, deletable))

        formatsList.sort(key=lambda tup: tup[0].lower()) # Sorting the messy list of ogr drivers
        return formatsList

    @staticmethod
    def check_driver_availability(driver_name, dvr_type = 'ogr'):
        rtn = False
        driver_list = None
        if dvr_type == 'ogr':
            driver_list = GDALHelper.get_ogr_drivers()
        elif dvr_type == 'gdal':
            driver_list = GDALHelper.get_gdal_drivers()
        else:
            raise ValueError('Invalid driver type {}, should be ogr or gdal'.format(dvr_type))

        if len([item for item in driver_list if item[0] == driver_name]) > 0:
            rtn = True
        return rtn

    @staticmethod
    def get_vfile_name(filename, vsf=None):
        """
        GDAL supports a virtual file system:
        GDAL can access files located on “standard” file systems. But most GDAL raster and vector drivers use a GDAL-specific abstraction to access files. GDAL vfs makes it possible to access less standard types of files, such as in-memory files, compressed files (.zip, .gz, .tar, .tar.gz archives), encrypted files, files stored on network (either publicly accessible, or in private buckets of commercial cloud storage services), etc.

        Each special file system has a prefix, and the general syntax to name a file is /vsfPREFIX/…

        This routine uses a give file name and generates a virtual file based on the vsf type. If no type is specified, the the file name is returned, with no virtual prefix.

        For more information see:
            https://gdal.org/user/virtual_file_systems.html
        """
        vfile = None
        if vsf:
            filename = filename.lstrip('/')
            filename = filename.lstrip('\\')
            if vsf == 'zip':
                vfile = '/vsizip/' + filename
            elif vsf == 'gz':
                vfile = '/vsigzip/' + filename
            elif vsf == 'gzip':
                vfile = '/vsigzip/' + filename
            elif vsf == 'tar':
                vfile = '/vsitar/' + filename
            elif vsf == 'curl':
                vfile = '/vsicurl/' + filename
            elif vsf == 'mem':
                vfile = '/vsimem/' + filename
            else:
                # treat it as a normal file
                # throw except ?
                vfile = filename
        else:
            # normal file
            vfile = filename
        return vfile

    @staticmethod
    def ogr_open(file, directory_archive = None, vsf = None, mode=READ_ONLY):
        """
        ogr_open - Opens a file using an ogr driver. The Ogr component will 'search'
                   for the correct driver by trying each driver until one is found
                   that can read the supied file. This is slow.
                   If you know the file format, use ogr_driver_open
        """
        # if an archive or directory is defined, add it to the file name
        filename = None
        if file:
            filename = file
        else:
            raise NameError('Require file name')
        if directory_archive:
            filename = os.path.join(directory_archive, file)

        vfile = GDALHelper.get_vfile_name(filename, vsf)
        # get current state of exception handling
        use_exceptions = ogr.GetUseExceptions()
        # turn it on just in case
        ogr.UseExceptions()
        try:
            ds = ogr.Open(vfile, mode)
            return ds
        except Exception as e:
            raise ValueError('invalid file name: {}'.format(vfile))
        finally:
            if not use_exceptions:
                # if exceptions were not used, then turn them back off
                ogr.DontUseExceptions()

    @staticmethod
    def ogr_driver_open(driver,file, directory_archive = None, vsf = None, mode=READ_ONLY):
        """
        ogr_driver_open - Opens a file using an ogr specific driver.
        First check that the driver is a valid driver
        Check that we have a file name
        """
        if not GDALHelper.check_driver_availability(driver):
            raise OSError('driver {} not available.'.format(driver))

        filename = None
        if file:
            filename = file
        else:
            raise NameError('Require file name')
        # if an archive or directory is defined, add it to the file name
        if directory_archive:
            filename = os.path.join(directory_archive, file)

        vfile = GDALHelper.get_vfile_name(filename, vsf)
        # get current state of exception handling
        use_exceptions = ogr.GetUseExceptions()
        # turn it on just in case
        ogr.UseExceptions()
        try:
            ds = ogr.GetDriverByName(driver).Open(vfile, mode)
            return ds
        except Exception as e:
            raise ValueError('invalid file name: {}'.format(vfile))
        finally:
            if not use_exceptions:
                # if exceptions were not used, then turn them back off
                ogr.DontUseExceptions()

    @staticmethod
    def get_datasource_capabilities(ds_or_fn, mode=READ_ONLY):
        """
        Get a DataSource capabilities. Accepts a data source or a file name
        Returns a dictionary of  data source attributes:
            name (file name),
            driver
            create  (True, False)
            delete  (True, False)
        """
        ds = None
        # get current state of exception handling
        use_exceptions = ogr.GetUseExceptions()
        # turn it on just in case
        ogr.UseExceptions()
        try:
            if not isinstance(ds_or_fn,ogr.DataSource):
                ds = ogr.Open(ds_or_fn, mode)
            else:
                ds = ds_or_fn
        except Exception as e:
            raise ValueError('Argument is not a data source, or a existing file name {}, {}'.format(ds_or_fn, e))
        finally:
            if not use_exceptions:
                # if exceptions were not used, then turn them back off
                ogr.DontUseExceptions()
        info_dict = {}
        info_dict['name'] = ds.GetName()
        info_dict['driver'] = ds.GetDriver().GetName()
        info_dict['create'] = ds.TestCapability(ogr.ODsCCreateLayer)
        info_dict['delete'] = ds.TestCapability(ogr.ODsCDeleteLayer)
        if not isinstance(ds_or_fn,ogr.DataSource):
            del ds
        return info_dict

    @staticmethod
    def get_ogr_layers_info(ds_or_fn):
        """
        Get a list of layers in a data source.
        Data source can be a file name or a data source

        """
        ds = None
        try:
            if not isinstance(ds_or_fn,ogr.DataSource):
                ds = ogr.Open(ds_or_fn, 0)
            else:
                ds = ds_or_fn
        except Exception as e:
            raise ValueError('Argument is not a ds, or a existing file name {}'.format(ds_or_fn))

        lyr_dict = {}
        for i in range(ds.GetLayerCount()):
            lyr = ds.GetLayer(i)
            lyr_name = lyr.GetName()
            lyr_type = __WKBGeometryTypes__[lyr.GetGeomType()]
            lyr_capabilities = GDALHelper.__get_layer_capabilities__(lyr)
            lyr_capabilities['layer_name'] = lyr_name
            lyr_capabilities['layer_type'] = lyr_type
            lyr_dict[i] = lyr_capabilities

        return lyr_dict

    @staticmethod
    def get_ogr_layer_attributes(lyr_or_fn, n=None, fields=None, geom=True, reset=True):
        """Get attribute values in a layer.

        lyr_or_fn - OGR layer object or filename to data source (will use 1st layer)
        n         - optional number of features to print; default is all
        fields    - optional list of case-sensitive field names to print; default
                    is all
        geom      - optional boolean flag denoting whether geometry type is printed;
                    default is True
        reset     - optional boolean flag denoting whether the layer should be reset
                  - to the first record before printing; default is True
        """
        lyr, ds = GDALHelper.__get_layers__(lyr_or_fn)
        if reset:
            lyr.ResetReading()

        n = n or lyr.GetFeatureCount()
        geom = geom and lyr.GetGeomType() != ogr.wkbNone
        fields = fields or [field.name for field in lyr.schema]
        data = [['FID'] + fields]
        if geom:
            data[0].insert(1, 'Geometry')
        feat = lyr.GetNextFeature()
        while feat and len(data) <= n:
            data.append(GDALHelper.__get_atts__(feat, fields, geom))
            feat = lyr.GetNextFeature()
        lens = map(lambda i: max(map(lambda j: len(str(j)), i)), zip(*data))
        # try to add a format string to make everything evenly spaced
        #format_str = ''.join(map(lambda x: '{{:<{}}}'.format(x + 4), lens))
        first_time = True
        df = None
        columns = None
        row_count = 0
        for row in data:
            try:
                if first_time:
                    first_time = False
                    columns = row
                    df = pd.DataFrame(columns=columns)
                else:
                    df.loc[row_count] = row
                    row_count += 1
            except UnicodeEncodeError:
                raise UnicodeEncodeError()

        if reset:
            lyr.ResetReading()
        return df

    @staticmethod
    def has_spatialite():
        """Determine if the current GDAL is built with SpatiaLite support."""
        use_exceptions = ogr.GetUseExceptions()
        ogr.UseExceptions()
        try:
            ds = ogr.GetDriverByName('Memory').CreateDataSource('memory')
            sql = '''SELECT sqlite_version(), spatialite_version()'''
            lyr = ds.ExecuteSQL(sql, dialect='SQLite')
            #del lyr
            #del ds
            return True
        except Exception as e:
            return False
        finally:
            del lyr
            del ds
            if not use_exceptions:
                ogr.DontUseExceptions()

    @staticmethod
    def get_capabilities(item):
        """Returns a dictionary of  capabilities for a driver, data source, or layer."""
        capabilities = None
        if isinstance(item, ogr.Driver):
            capabilities = GDALHelper.__get_capabilities__(item, 'Driver', 'ODrC')
        elif isinstance(item, ogr.DataSource):
            capabilities = GDALHelper.__get_capabilities__(item, 'DataSource', 'ODsC')
        elif isinstance(item, ogr.Layer):
            capabilities = GDALHelper.__get_capabilities__(item, 'Layer', 'OLC')
        else:
            raise ValueError('Unsupported item')
        return capabilities

    @staticmethod
    def copy_datasource(source_fn, target_fn):
        """Copy an ogr data source."""
        ds = ogr.Open(source_fn, READ_ONLY)
        if ds is None:
            raise OSError('Could not open {0} for copying.'.format(source_fn))
        if os.path.exists(target_fn):
            ds.GetDriver().DeleteDataSource(target_fn)
        ds.GetDriver().CopyDataSource(ds, target_fn)

    @staticmethod
    def get_shp_geom(fn):
        """
        Convenience function to get the next geometry from a layer or files.
        If a file name is specified, returns first shape from first layer in file
        """
        lyr, ds = __get_layer__(fn)
        feat = lyr.GetNextFeature()
        return feat.geometry().Clone()

    @staticmethod
    def get_layers(ds_or_fn, mode= READ_ONLY):
        """Gets a dictionary of layers in a data source for a file or data source
        ds_or_fn - a data source or path to data source
        """
        ds = None
        try:
            if not isinstance(ds_or_fn,ogr.DataSource):
                ds = ogr.Open(ds_or_fn, mode)
            else:
                ds = ds_or_fn
        except Exception as e:
            raise ValueError('Argument is not a ds, or a existing file name {}'.format(ds_or_fn))
        if ds is None:
            raise OSError('Could not open {}'.format(fn))
        layer_dict = {}
        for i in range(ds.GetLayerCount()):
            lyr = ds.GetLayer(i)
            layer_dict[i] = (lyr.GetName(), __WKBGeometryTypes__[lyr.GetGeomType()])
        return layer_dict

    """
    Ogr and GDAL has many constants, and these constants are stored in dictionaries by library (lib)
    (gdal, ogr -> lib.__dict__). For Ogr, the constant name follows a <prefix><suffix_name> pattern, while GDAL follows a  <prefix>_<suffix name>. C
    The functions get_constant_value, get_gdal_constant_value and get_ogr_constant_value will return the value of a constant

    For more information of Ogr constants see:
        https://gdal.org/python/osgeo.ogr-module.html
        https://gdal.org/python/

    For more information on GDAL constants values see:
        https://naturalatlas.github.io/node-gdal/classes.html
        https://gdal.org/python/osgeo.gdalconst-module.html
        https://gdal.org/python/
    """
    @staticmethod
    def get_constant_value(lib, name):
        """
        Given a library and a constant name, return the value of
        """
        try:
            return lib.__dict__[name]
        except KeyError:
            return None

    @staticmethod
    def get_gdal_constant_value(name):
        return GDALHelper.get_constant_value(gdal, name)
    @staticmethod
    def get_ogr_constant_value(name):
        return GDALHelper.get_constant_value(ogr, name)

    @staticmethod
    def get_suffix_from_format(format):
        """
        For a common driver format, returns the  output file suffix;
        We are currently supporting only GTiff, ENVI, EHdr (ESRI bil file) and PNG
        """
        suffix = ''
        if format.lower() == 'gtiff':
            suffix = '.tif'
        elif format.lower() == 'envi':
            suffix = '.flt'
        elif format.lower() == 'ehdr':
            suffix = '.flt'
        elif format.lower() == 'png':
            suffix = '.png'
        return suffix

    @staticmethod
    def get_ogr_schema(layer, reset=True):
        if reset:
            layer.ResetReading()

        field_names = {}
        ldefn = layer.GetLayerDefn()
        for n in range(ldefn.GetFieldCount()):
            fdefn = ldefn.GetFieldDefn(n)
            name = fdefn.GetName()
            ftype = fdefn.GetType()
            type_name = fdefn.GetFieldTypeName(ftype)

            field_names[name] = (fdefn, type_name, ftype)
        return field_names

    @staticmethod
    def get_ogr_field_names(layer):
        """
        Returns a list of field names for a layer. Field names are the data columns names
        for the schema
        """
        return [field.name for field in layer.schema]

    @staticmethod
    def to_dataframe(lyr_or_fn, n=None, fields=None, geom=True, reset=True):
        """place a shape, read by ogr into a data frame.

        lyr_or_fn - OGR layer object or filename to datasource (will use 1st layer)
        n         - optional number of features to print; default is all
        fields    - optional list of case-sensitive field names to print; default
                    is all
        geom      - optional boolean flag denoting whether geometry type is printed;
                    default is True
        reset     - optional boolean flag denoting whether the layer should be reset
                  - to the first record before printing; default is True

        geopandas' geometruies are shapely objects. To convert we export the ogr geometry
        as a Well Know Text (wkt) string and import them into shapely
        """
        lyr, ds = __get_layer__(lyr_or_fn)
        if reset:
            lyr.ResetReading()

        crs = lyr.GetSpatialRef()
        n = n or lyr.GetFeatureCount()
        geom = geom and lyr.GetGeomType() != ogr.wkbNone
        fields = fields or [field.name for field in lyr.schema]
        data = [['FID'] + fields]
        if geom:
            data[0].insert(1, 'wktcolumn')
        feat = lyr.GetNextFeature()
        while feat and len(data) <= n:
            data.append(__get_df_atts__(feat, fields, geom))
            feat = lyr.GetNextFeature()
        df = pd.DataFrame(data)
        header = df.iloc[0]
        df = df[1:]
        df.columns = header
        # use shapely to convert the geometry
        geometry = df['wktcolumn'].map(wkt.loads)
        # drop the wkt column
        df.drop('wktcolumn', axis=1)
        gdf = gpd.GeoDataFrame(df, crs=crs, geometry=geometry)
        return gdf

    @staticmethod
    def wkt2ogr(shape):
        # ogr has several toolsfor 'importing' recorded in other format
        # e.g. Well Known Text (wkt), GeoJson, GML, Well Known Binary (wkb)
        # geopandas uses shapely for geometries, and <shape>.wkt will
        # exportthe shape to wkt, which can then be imported to ogr
        linelyr = ogr.CreateGeometryFromWkt(shape.wkt)
        return linelyr

    @staticmethod
    def ogr2json(geom):
        geojson = geom.ExportToJson()
        return geojson

    # gdal raster
    @staticmethod
    def gdal_open(file, directory_archive, vsf = None, mode=READ_ONLY):
        # if an archive or directory is defined, add it to the file name
        filename = file
        if directory_archive:
            filename = os.path.join(directory_archive, file)

        vfile = GDALHelper.get_vfile_name(filename, vsf)
        #try block?
        ds = gdal.Open(vfile, READ_ONLY)
        return ds

    @staticmethod
    def get_raster_band_nodata_value(ds,band):
        try:
            srcBand = ds.GetRasterBand(band)
            rtn = srcBand.GetNoDataValue()

        except RuntimeError:
            print('Band ( %i ) not found ' + str(band))
            rtn = None
        except:
            print("Unexpected error:", sys.exc_info()[0])
            return None

    @staticmethod
    def get_raster_band_scale(ds,band):
        try:
            srcBand = ds.GetRasterBand(band)
            rtn = srcBand.GetScale()

        except RuntimeError:
            print('Band ( %i ) not found ' + str(band))
            rtn = None
        except:
            print("Unexpected error:", sys.exc_info()[0])
            return None

    @staticmethod
    def get_raster_band_gdal_data_type(ds, band):
        try:
            srcBand = ds.GetRasterBand(band)
            rtn = 'GDT_' + gdal.GetDataTypeName(srcBand.DataType)
        except RuntimeError:
            print('Band ( %i ) not found ' + str(band))
            rtn = None
        except:
            print("Unexpected error:", sys.exc_info()[0])
            rtn = None
        return rtn

    @staticmethod
    def world2pixel(geo_t, x,y):
        #DBL_EPSILON = 2.2204460492503131e-16
        #geo_t = in_band.GetGeoTransform
        ulX = geo_t[0]
        ulY = geo_t[3]
        xDist = geo_t[1]
        yDist = geo_t[5]
        rtnX = geo_t[2]
        rtnY = geo_t[4]
        pixel = int((x -ulX)/ xDist)
        line = int((y - ulY) / yDist) # yDist is often negative
        return (pixel,line)

    @staticmethod
    def image2array(image):
        A = gdalnumeric.fromstring(image.tostring(), 'b')
        A.shape = image.im.size[1], image.im.size[0]
        return A

    @staticmethod
    def array2image(A):
        image = Image.fromstring('L', (A.shape[1], A.shape[0]), (A.astype('b')).tostring())
        return image

    @staticmethod
    def histogram(a, bins=range(0, 256)):
        """
        Histogram function for multi-dimensional array.
        a = array
        bins = range of numbers to match
        """
        fa = a.flat
        n = gdalnumeric.searchsorted(gdalnumeric.sort(fa), bins)
        n = gdalnumeric.concatenate([n, [len(fa)]])
        hist = n[1:] - n[:-1]
        return hist

    @staticmethod
    def stretch(a):
        # Performs a histogram stretch on a gdalnumeric array image.
        hist = GDALHelper.histogram(a)
        im = GDALHelper.array2image(a)
        lut = []
        for b in range(0, len(hist), 256):
            # step size
            step = functools.reduce(operator.add, hist[b:b+256]) / 255
            # create equalization lookup table
            n = 0
            for i in range(256):
                lut.append(n / step)
                n = n + hist[i+b]
        im = im.point(lut)
        return GDALHelper.image2array(im)

    @staticmethod
    def compute_overview_levels(band):
        """Return an appropriate list of overview levels."""
        max_dim = max(band.XSize, band.YSize)
        overviews = []
        level = 1
        while max_dim > 256:
            level *= 2
            overviews.append(level)
            max_dim /= 2
        return overviews

    @staticmethod
    def make_raster(in_ds, fn, data, data_type, nodata=None):
        """Create a one-band GeoTIFF.

        in_ds     - datasource to copy projection and geotransform from
        fn        - path to the file to create
        data      - NumPy array containing data to write
        data_type - output data type
        nodata    - optional NoData value
        """
        driver = gdal.GetDriverByName('GTiff')
        out_ds = driver.Create(
            fn, in_ds.RasterXSize, in_ds.RasterYSize, 1, data_type)
        out_ds.SetProjection(in_ds.GetProjection())
        out_ds.SetGeoTransform(in_ds.GetGeoTransform())
        out_band = out_ds.GetRasterBand(1)
        if nodata is not None:
            out_band.SetNoDataValue(nodata)
        out_band.WriteArray(data)
        out_band.FlushCache()
        out_band.ComputeStatistics(False)
        return out_ds

    @staticmethod
    def make_slices(data, win_size):
        """Return a list of slices given a window size.

        data     - two-dimensional array to get slices from
        win_size - tuple of (rows, columns) for the moving window
        """
        rows = data.shape[0] - win_size[0] + 1
        cols = data.shape[1] - win_size[1] + 1
        slices = []
        for i in range(win_size[0]):
            for j in range(win_size[1]):
                slices.append(data[i:rows+i, j:cols+j])
        return slices

    @staticmethod
    def make_masked_slices(band, win_size):
        """Return a list of slices given a window size.

        band     - band to get slices from
        win_size - tuple of (rows, columns) for the moving window
        """
        rows = band.YSize + win_size[0] - 1
        cols = band.XSize + win_size[1] - 1
        data = np.ma.masked_all((rows, cols), np.float)

        edge_rows, edge_cols = [int(n / 2) for n in win_size]
        data[edge_rows:-edge_rows, edge_cols:-edge_cols] = band.ReadAsArray()
        return data

    @staticmethod
    def stack_bands(bandFiles):
        """Returns a 3D array containing all band data from all files."""
        bands = []
        for fn in bandFiles:
            in_ds = gdal.Open(fn)
            for i in range(1, in_ds.RasterCount + 1):
                bands.append(in_ds.GetRasterBand(i).ReadAsArray())
        return np.dstack(bands)

    @staticmethod
    def files2bands(bandFiles, outFile, fileType):
        # assumes that each input files has a single band
        firstTime = True

        out_ds = None
        _driver = gdal.GetDriverByName(fileType)
        band = 1
        for fn in bandFiles:
            if firstTime:
                firstTime = False
                in_ds = gdal.Open(fn)
                in_band = in_ds.GetRasterBand(1)
                out_ds = _driver.Create(outFile, in_band.XSize, in_band.YSize, len(bandFiles),
                                        in_band.DataType)
                out_ds.SetProjection(in_ds.GetProjection())
                out_ds.SetGeoTransform(in_ds.GetGeoTransform())
                in_data = in_band.ReadAsArray()
                out_band = out_ds.GetRasterBand(band)
                out_band.WriteArray(in_data)
                band = band + 1
            else:
                out_ds.GetRasterBand(band).WriteArray(
                    gdal.Open(fn).ReadAsArray()
                )
                band = band + 1
        out_ds.FlushCache()
        del out_ds



