from datetime import datetime, date, timedelta
from dateutil.parser import parse
from dateutil.relativedelta import relativedelta
import os
import time
import tempfile


class dseCommon():
    @staticmethod
    def norm_join(dir, path):
        return os.path.normpath(os.path.join(dir, path))

    @staticmethod
    def IsNullOrEmpty(value):
        return value is None or len(value) == 0

    @staticmethod
    def all_digits(string):
        result = all([v.isdigit() for v in string])
        return result

    @staticmethod
    def parse_date(date_arg, default_month = 1, default_day = 1):
        if dseCommon.all_digits(date_arg):
            if len(date_arg) == 4:
                result = datetime(int(date_arg), default_month, default_day)

            elif len(date_arg) == 7:
                result = datetime.strptime(date_arg, '%Y%j')

            elif len(date_arg) == 8:
                result = datetime.strptime(date_arg, '%Y%m%d')

            else:
                raise ValueError("The date string {} is not in one of the supported numeric formats (e.g. 2015, 2015365, 20151231)."
                                 .format(date_arg))
        else:
            # try date parser
            try:
                result = parse(date_arg)
            except ValueError:
                raise ValueError("The date string {} is not in one of the dateutils.parse supported formats .".format(date_arg))

        return result

    @staticmethod
    def get_default_log_file_name(file_name_arg, variables, name_prefix):
        file_name = file_name_arg
        if not file_name:
            time_stamp = time.strftime("%Y%m%d-%H%M%S")
            params_label = '-'.join(variables)
            file_name = '{}_{}.{}.log'.format(name_prefix, params_label, time_stamp)

        return file_name

    @staticmethod
    def get_time_series(start_date, number_of_intervals, increment, period):
        """
        returns a list of datetime values from start_date for 'number_of_intervals' where
        interval is defined by increment 'increment' and period 'period'.
        'period must be a standard date time period'
        Example:
            get 24 hourly values starting at 2015-01-01 01:00:00.0
            start_date = datetime(2015,1,1,1,0)
            time_series = get_time_serries(start_date, 24, 1, 'hours')
        """
        valid_periods =  ['years', 'months', 'weeks', 'days', 'hours', 'minutes', 'seconds', 'microseconds']
        result = []
        # make sure that we have a valid period value
        if period in valid_periods:
            nxt = start_date
            delta = relativedelta(**{period:increment})
            i = 0
            while i < number_of_intervals:
                result.append(nxt)
                nxt += delta
                i += 1
        else:
            raise ValueError('period ({0}) is not valid. Must be one of the following: [{1}]'.format(period, valid_periods))
        return result

    @staticmethod
    def get_time_range(start_date, end_date, increment, period):
        """
        returns a list of datetime values from start_date to end_date with
        increment 'increment' and period 'period'.
        'period must be a standard date time period'
        Example:
            get 24 hourly values starting at 2015-01-01 01:00:00.0
            start_date = datetime(2015,1,1,1,0)
            end_date = datetime(2015,1,3)
            time_series = get_time_serries(start_date, end_date, 1, 'hours')
        """
        valid_periods =  ['years', 'months', 'weeks', 'days', 'hours', 'minutes', 'seconds', 'microseconds']
        result = []
        # make sure that we have a valid period value
        if period in valid_periods:
            nxt = start_date
            delta = relativedelta(**{period:increment})
            while nxt <= end_date:
                result.append(nxt)
                nxt += delta
        else:
            raise ValueError('period ({0}) is not valid. Must be one of the following: [{1}]'.format(period, valid_periods))
        return result

    @staticmethod
    def day_of_year_to_date(year, doy):
        # Given the year and day of year (doy) return [year, month, day]
        # doy is zero based (e.g. 0,1,2,3,4)
        # doy > number of days in year will increment to next year
        # doy < 0 days from end of previous year
        # example:

        # dayOfYearToYMD(2000, 45) return [2000, 2, 15]
        # doyOfYearToYMD(2000,0) returns  [2000,1,1]
        # dayOfYearToYMD(2000, 365) returns [2000,12,31]
        # dayOfYearToYMD(2000, 59) returns [2000,2, 29]
        # dayOFYearToYMD(2000, -1) returns [1999, 12, 31]
        # dayOfYearToYMD(2000, 30) returns [2001,1,15]
        d = datetime(year,1,1, 0, 0, 0)
        d = d + timedelta(doy - 1)
        return d

    @staticmethod
    def time_delta_label(elapsed):
        if elapsed.seconds < 3600:
            start_index = 1
        else:
            start_index = 0

        elapsed_parts = str(elapsed).split(':')
        elapsed_msg = ':'.join(elapsed_parts[start_index:])

        return elapsed_msg

    @staticmethod
    def get_timestamp():
        return datetime.now().strftime('%Y%m%d_%H%M%S')

    @staticmethod
    def get_app_name():
        return os.path.basename(os.path.dirname(__file__))

    @staticmethod
    def get_temp_dir():
        sub_dir = dseCommon.get_timestamp()
        out_dir = os.path.join(tempfile.gettempdir(), dseCommon.get_app_name(), 'out', 'weather-{}'.format(sub_dir))

        return out_dir


