import logging
import os
import sys
import tempfile

from datetime import datetime
from dselib.dse_tools import dseCommon

class dseLogging():
    @staticmethod
    def exit_with_exceptions(exception_objects, debug=False, exit=True):
        """
        Logs error messages for a list of exceptions and exists gracefully. If in the debug mode, shows original exceptions.
        :param exception_objects: A list of exception objects.
        :param debug: A flag indicating a debug mode.
        :param exit: A flag indicating if this method should exit the application.
        """
        for e in exception_objects:
            exit_with_error(e, debug, exit=False)

        if exit and len(exception_objects) > 0:
            raise SystemExit(1)

    @staticmethod
    def exit_with_error(message_or_exception, debug=False, exception_object=None, exit=True):
        """
        Logs error message and exits gracefully. If in the debug mode, shows the original exception.
        :param message: A message text or an exception object (containing the message).
        :param debug: A flag indicating a debug mode.
        :param exception_object: An exception object.
        :param exit: A flag indicating if this method should exit the application.
        """
        message = str(message_or_exception)
        exception = exception_object or message_or_exception
        if debug:
            logging.exception(message, exc_info=exception)
        else:
            logging.error(message)

        if exit:
            raise SystemExit(1)

    @staticmethod
    def init_log(log_dir, name, debug):
        """
        Initialize the logger so that all logged messages go both to a log file and the console.
        (This code was compiled from different online examples like those found on the StackOverflow.)
        """
        if log_dir is None:
            log_dir = os.path.join(tempfile.gettempdir(),dseCommon.get_app_name() , 'log')

        os.makedirs(log_dir, exist_ok=True)

        timestamp = dseCommon.get_timestamp()
        log_path = os.path.join(log_dir, '{}-{}.log'.format(os.path.basename(name)[:-3], timestamp))

        level = logging.DEBUG if debug else logging.INFO

        # set up logging to file
        logging.basicConfig(level=level,
                            format='%(asctime)s %(levelname)-8s %(message)s',
                            datefmt='%m-%d %H:%M',
                            filename=log_path,
                            filemode='w')

        # define a Handler which writes INFO messages or higher to the sys.stderr
        console = logging.StreamHandler(sys.stdout)
        console.setLevel(level)
        # set a format which is simpler for console use
        formatter = logging.Formatter('%(levelname)-8s %(message)s')
        # tell the handler to use this format
        console.setFormatter(formatter)
        # add the handler to the root logger
        logging.getLogger('').addHandler(console)

        stderr = logging.StreamHandler(sys.stderr)
        stderr.level = logging.ERROR
        logging.getLogger('').addHandler(stderr)

        return log_path

