import json
import os
import struct

from collections import defaultdict
from datetime import datetime

from dselib.dse_tools import Tile, TileHelper
from dselib.dse_tools import TileNodeMap

from dselib.dse_tools import Variables
from dselib.dse_tools import Weather

class DTKWeather(Weather):
    """
    Represents DTK weather information: nodes, node-tile map and tile weather time series for DTK weather variables.
    Supports two scenarios:
    1. Save: Generating new DTK weather binary and metadata files
    2. Load: Reading of existing DTK weather binary and metadata files
    """
    def __init__(self, nodes, resolution, start_date, end_date, weather_series_map=None, variables=None, tile_node_map = None):
        super().__init__( nodes, resolution, start_date, end_date, weather_series_map=weather_series_map, variables=variables, tile_node_map=tile_node_map)

    def save(self, dtk_file_args):
        """
        Generates DTK weather binary and metadata files based of nodes list and weather time series.
        dtk_file_args: An output directory string or a DTKWeather.FilesArgs object providing output files paths
        Returns: The resulting DTKWeather.FilesArgs object used for generating DTK weather files.
        """
        # Prepares file arg object (determines the output dir and sets the resolution).
        file_args = self._parse_dtk_file_args(dtk_file_args)
        errors = []
        for variable, tile_series_map in self.weather_series_map.items():
            try:

                # Get binary (.bin) and metadata (.bin.json) file paths.
                bin_file_path, meta_file_path = file_args.get_file_paths(variable)
                value_count = len(list(tile_series_map.values())[0])

                # Generates binary (.bin) file and returns tile-offset dictionary.
                tile_offset_map = self._save_binary(tile_series_map, bin_file_path)

                # Generates metadata (.bin.json) file.
                self._save_metadata(tile_offset_map, meta_file_path, value_count, file_args.id_reference, file_args.data_provenance)
            except Exception as e:
                # Keeps track of exceptions and allows processing of other variables.
                errors.append(e)

        return file_args, errors

    def _save_binary(self, tile_series_map, bin_file_path):
        """Iterates over tile ids, stores their series into a binary file, and keeps track of tile's offset."""
        offset = 0
        tile_offset_map = {}
        with open(bin_file_path, 'wb') as bin_file:
            for tile_id in tile_series_map:

                # Record the offset used for a given tile.
                tile_offset_map[tile_id] = offset
                # Reads weather series
                series = tile_series_map[tile_id]
                # Packs it into a binary format
                series_len = len(series)
                bin = struct.pack('f' * series_len, *series)
                # Write the binary data to a file
                bin_file.write(bin)
                # Prepare the offset for the next tile.
                offset += series_len * 4

        return tile_offset_map

    def _save_metadata(self, tile_offset, meta_file_path, value_count, id_reference, data_provenance):
        """Constructs the offset string and generates and saves the metadata json object."""
        offset_string = ''
        for tile_id, nodes in self.tile_node_map.items():
            # for each node adds id-offset hex values to the offset string.
            offset = tile_offset[tile_id]
            for node_id in nodes:
                offset_string += "{:08x}{:08x}".format(node_id, offset)

        # Determines the nodes envelope.
        min_lat = min([n.latitude for n in self.nodes])
        max_lat = max([n.latitude for n in self.nodes])
        min_lon = min([n.longitude for n in self.nodes])
        max_lon = max([n.longitude for n in self.nodes])

        # Determines tile and node counts.
        node_count = len(self.nodes)
        tile_count = len(self.tile_node_map)

        # Populates the metadata json object.
        meta = {
            "Metadata": {
                "Tool": "weather-files",
                "Author": "Institute for Disease Modeling",
                "WeatherSchemaVersion": "2.0",
                "IdReference": id_reference,
                "WeatherCellCount": tile_count,
                "NodeCount": node_count,
                "OffsetEntryCount": node_count,
                "NumberDTKNodes": node_count,
                "DatavaluePerCell": value_count ,
                "DatavalueCount": value_count ,
                "UpdateResolution": "CLIMATE_UPDATE_DAY",
                "OriginalDataYears": '{}-{}'.format(self.start_date.year, self.end_date.year),
                "StartDayOfYear": '{} {}'.format(self.start_date.strftime('%B'), self.start_date.day),
                "DataProvenance": data_provenance,
                "Resolution": "none" if self.resolution is None else self.resolution.arc_seconds_minutes_label,
                "UpperLatitude": max_lat,
                "LeftLongitude": min_lon,
                "BottomLatitude": min_lat,
                "RightLongitude": max_lon
            },
            "NodeOffsets": offset_string
        }

        # Save the metadata json object to a file.
        with open(meta_file_path, 'w') as f:
            f.write(json.dumps(meta, sort_keys=False, indent=4).strip('"'))

    def load(self, dtk_file_args):
        """Loads DTK weather binary and metadata files into data structures. Relies on the list of expected nodes."""

        # Parse the input DTK files arguments.
        file_args = self._parse_dtk_file_args(dtk_file_args)

        for variable in self.weather_series_map:
            # Obtain binary and metadata file paths.
            bin_file_path, meta_file_path = file_args.get_file_paths(variable)

            # Read metadata into a json object.
            with open(meta_file_path) as meta_file:
                meta = json.loads(meta_file.read())

            # Read node and tile counts and node offsets.
            value_count = meta['Metadata']['DatavalueCount']
            offset_string = meta['NodeOffsets']
            offsets_nodes = DTKWeather._unpack_offset_string(offset_string)

            # Sort offsets to optimize weather series reading.
            offsets = sorted(offsets_nodes)

            # Node id to node dictionary used for determining the expected list of node ids and tiles.
            nodes = {node.id: node for node in self.nodes}
            expected_node_ids = list(nodes)

            tile_offsets = {}
            # Instantiates an empty node-tile map, to be populated during reading.
            tile_node_map = TileNodeMap(self.resolution)
            with open(bin_file_path, 'rb') as bin_file:
                for offset in offsets:
                    # For each offset only take node ids which are from the expected node list (see __init__).
                    offset_node_ids = list(set(offsets_nodes[offset]).intersection(expected_node_ids))

                    if len(offset_node_ids) == 0:
                        continue

                    # At least one node per offset is sufficient to generate a tile (since nodes were grouped by tiles).
                    tile = Tile.from_point(nodes[offset_node_ids[0]], file_args.resolution)
                    if tile.id in tile_node_map:
                        raise ValueError("Offset {} maps to the tile {} which is already mapped to the offset {}.".format(offset, tile.id, tile_offsets[tile.id]))

                    # Record the tile-nodes map.
                    tile_node_map.add_tile_node_ids(tile.id, offset_node_ids)
                    tile_offsets[tile.id] = offset

                    # Read weather series for the given DTK variable and tile.
                    series = DTKWeather._read_series_from_file_object(bin_file, offset, value_count)
                    self.weather_series_map[variable][tile.id] = series

            if self.tile_node_map != tile_node_map:
                raise ValueError("Variable {} tile-node map {} doesn't match the expected value map {}.", variable)

        return file_args

    def _parse_dtk_file_args(self, dtk_file_args):
        """Prepares the file args object to be used for DTK weather files generation."""

        if type(dtk_file_args) == str:
            file_args = DTKWeather.FilesArgs(dtk_file_args, self.resolution)

        elif type(dtk_file_args) == DTKWeather.FilesArgs:
            file_args = dtk_file_args
            file_args.resolution = self.resolution

        else:
            raise ValueError("Unsupported type of dtk weather file arguments: {}.", type(dtk_file_args))

        return file_args

    @staticmethod
    def _unpack_offset_string(offset_string):
        """
        Converts the offset string to offset-nodes dictionary. Inspired by Dennis Harding's script:
        https://github.com/InstituteforDiseaseModeling/LargeDataTools/blob/master/Tools/IDM.LD.Tools/IDM.LD.Tools.Utilities/Python-Scripts/ClimateCompact2Full..py#L61
        """
        n = 16
        m = 8
        offset_pairs = [offset_string[i:(i + n)] for i in range(0, len(offset_string), n)]
        offset_nodes = defaultdict(list)

        for pair  in offset_pairs:
            node_id = int('0x{}'.format(pair[:m]), 16)
            offset = int('0x{}'.format(pair[m:]), 16)
            offset_nodes[offset].append(node_id)

        return offset_nodes

    @staticmethod
    def _read_series_from_file_object(bin_file, offset, value_count, size=4):
        """Reads series of float values from a binary file object."""
        bin_file.seek(offset)
        bin_bytes = bin_file.read(value_count * size)
        series = struct.unpack('{}f'.format(value_count), bin_bytes)

        return series


    class FilesArgs:
        """Encapsulates the information needed to construct DTK weather binary and metadata file paths."""
        # Defaults based on current use. The actual values come from data source metadata.
        default_args = {
            'file_pattern': '{place}{label}_{arc_seconds_minutes_label}_{variable}_{time_step}.bin',
            'place': 'dtk',
            'time_step': 'daily',
            'id_reference': 'default',
            'data_provenance': 'unspecified'
        }

        # TODO: propose new patten
        # file_pattern='{place}-{size_in_km}km-{variable}-{time_step}.bin',

        # Default DTK weather variable names used in file names.
        variable_file_names = {
            Variables.airtemp: 'air_temperature',
            Variables.grndtemp: 'land_temperature',
            Variables.humid: 'relative_humidity',
            Variables.rainfall: 'rainfall'
        }

        def __init__(self,
                     dtk_dir,
                     resolution,
                     place=None,
                     time_step=None,
                     file_pattern=None,
                     id_reference=None,
                     data_provenance=None,
                     label=None,
                     **kwargs
                     ):

            self.dtk_dir = dtk_dir
            self.resolution = resolution
            self.place = place or self.default_args['place']
            self.time_step = time_step or self.default_args['time_step']
            self.file_pattern = file_pattern or self.default_args['file_pattern']
            self.id_reference = id_reference or self.default_args['id_reference']
            self.data_provenance = data_provenance or self.default_args['data_provenance']
            self.label = self._parse_label(label)
            self.kwargs = kwargs

        @staticmethod
        def _parse_label(label):
            label = label or ''
            label = label.strip()

            if label.strip() != '':
                label = (label or '').strip()
                label_prefix = '' if label.startswith('_') else '_'
                label = label_prefix + label

            return label

        def get_file_paths(self, variable, variable_labels_map=variable_file_names):
            # Determine the DTK weather variable name to be used in file name.
            if variable_labels_map is not None and variable in variable_labels_map:
                variable_label = variable_labels_map[variable]
            else:
                # If variable_file_names is not provided use the actual variable name.
                variable_label = variable

            file_name = self.file_pattern.format(place=self.place,
                                                 arc_seconds_minutes_label= "none" if self.resolution is None else self.resolution.arc_seconds_minutes_label,
                                                 arc_seconds= "none" if self.resolution is None else self.resolution.arc_seconds,
                                                 size_in_degrees= 0 if self.resolution is None else self.resolution.size_in_degrees,
                                                 size_in_km= 0 if self.resolution is None else self.resolution.size_in_km,
                                                 variable=variable_label,
                                                 time_step=self.time_step,
                                                 label=self.label,
                                                 **self.kwargs)

            bin_file_name = file_name
            meta_file_name = '{}.json'.format(file_name)
            bin_file_path = os.path.join(self.dtk_dir, bin_file_name)
            meta_file_path = os.path.join(self.dtk_dir, meta_file_name)

            return bin_file_path, meta_file_path
