import sys, os
import numpy as np
import pandas as pd
import xarray as xr

from enum import Enum

from abc import ABC, abstractmethod

"""
    Relative Humidity (RH) is the ratio to the Saturation Vapor Pressure (Es)
    to the Actual Vapor Pressure. There are several functions available
    and there are on line calculators to use for checking. In all cases the
    functions provided will provide an approximation ofthe Relative Humidity
    and are optimized for a range of temperatures.

    humidityType identifies the source of the function used to calculate Humidity.

"""
class humidityType (Enum):
    arm = 'arm'
    viasala = 'viasala'

    def __str__(self):
        return self.value

    @staticmethod
    def from_string(s):
        try:
            return humidityType[s]
        except KeyError:
            raise ValueError('Invalid calcType: ' + s)


"""
    humidityCalculator is an abstract class that provides the common structure
    for humidity calculators.
"""

class HumidityCalculator(ABC):
    """
    HumidityCalculator is the base abstract class for preforming humidity calculations.

    """

    @staticmethod
    def humidity_factory(humidity_type: humidityType):
        if humidity_type == humidityType.arm:
            return CalculateHumidityARM()
        if humidity_type == humidityType.viasala:
            return CalculateHumidityViasala()
        return None

    @abstractmethod
    def calculate_humidity(self, T,TD):
        pass

class CalculateHumidityARM(HumidityCalculator):

    def _vapor_pressure(self,T):
        E  = np.exp(((17.625*T)/(243.04+T)))
        return E

    def calculate_humidity(self, T, TD):
        # see http://andrew.rsmas.miami.edu/bmcnoldy/Humidity.html
        #     https://journals.ametsoc.org/doi/pdf/10.1175/BAMS-86-2-225
        # August-Roche-Magnus approximation.
        E  = self._vapor_pressure(TD)
        Es = self._vapor_pressure(T)
        RH =  (E/Es)
        return np.clip(RH, 0.0,1.0)

class CalculateHumidityViasala(HumidityCalculator):
    def __init__(self):
        self._A = np.float32(6.116441)
        self._m = np.float32(7.591386)
        self._Tn = np.float32(240.7263)

    def _vapor_pressure(self,T):
        E = self._A*np.power(np.float32(10.0), ((self._m * T)/(T + self._Tn)))
        return E

    def calculate_humidity(self, T, TD):
        """
            Calculating humidity from "Humidity Conversion Formulas" Viasala
            https://www.vaisala.com/sites/default/files/documents/Humidity_Conversion_Formulas_B210973EN-F.pdf
            Es = A*10^(mT/T+Tn)
            Es = A*10^(mTd/(Td+Tn)
            for -20 to +50 degrees C, A = 6.116441. m = 7.591386, Tn = 240.7263
            will calculate Es and E with a max error of 0.083
        """

        Es = self._vapor_pressure(T)
        E  = self._vapor_pressure(TD)
        RH = (E/Es)
        return np.clip(RH, 0.0, 1.0)

