import csv
import json

from abc import ABC, abstractmethod

from dselib.dse_tools import Node
from dselib.dse_tools import TileHelper


class NodeParser(ABC):
    """Base class for reading nodes information from a file."""
    def __init__(self, nodes_path):
        """
        :param nodes_path: The path to the file containing node coordinates.
        """
        super().__init__()
        self.nodes_path = nodes_path
        self.nodes = []

    @abstractmethod
    def parse_nodes(self):
        """
        Reads input file and returns the list of nodes.
        Responsible for validating coordinates and converting longitudes to 0 to 360 range if needed.
        """
        return []

    def _raise_file_exception(self, exception_object, exception_message=None):
        exception_msg = exception_message or str(exception_object)
        exception_type = type(exception_object).__name__
        parser_name = type(self).__name__

        msg_pattern = '{}: {}: {}'
        if self.nodes_path not in exception_msg:
            msg_pattern += ': in the file "{}"'

        msg = msg_pattern.format(parser_name, exception_type, exception_msg, self.nodes_path)

        raise Exception(msg)

class CSVNodeParser(NodeParser):
    """Nodes parser for .csv file."""
    def __init__(self, nodes_path, latitude_column, longitude_column, node_id_column=None, columns_case=False):
        """
        :param nodes_path: The path to the file containing node coordinates.
        :param latitude_column: The name of the latitude column.
        :param longitude_column: The name of the longitude column.
        :param node_id_column: The name of the node ID column.
        :param columns_case: The flag indicating if columns name comparison is case sensitive or insensitive.
        """
        super().__init__(nodes_path)

        self.latitude_column = latitude_column
        self.longitude_column = longitude_column
        self.node_id_column = node_id_column
        self.columns_case = columns_case

    def parse_nodes(self):
        try:
            nodes = self._parse_nodes()
        except Exception as e:
            self._raise_file_exception(e)

        return nodes

    def _parse_nodes(self):
        """Reads input points from a .csv file and generates a list of nodes (each point instantiates one node)."""
        with open(self.nodes_path, newline='') as csv_file:
            rows = list(csv.reader(csv_file))

            # Prepare header column names.
            columns = [c for c in rows[0] if c is not None]
            columns = self._prep_list_case(columns)

            # Prepare column names.
            self.latitude_column = self._prep_coordinate_column(columns, self.latitude_column, 'lat')
            self.longitude_column = self._prep_coordinate_column(columns, self.longitude_column, 'lon')
            self.node_id_column = self._prep_name_case(self.node_id_column)

            # Get column indexes.
            lat_idx = columns.index(self.latitude_column)
            lon_idx = columns.index(self.longitude_column)
            node_id_idx = self._node_id_column_index(columns)

            # Read file line by line.
            for i in range(1, len(rows)):
                row = rows[i]

                # Parse latitude and longitude values.
                latitude = self._parse_column(float, row, lat_idx, i)
                longitude = self._parse_column(float, row, lon_idx, i)

                # Parse node ID values if node ID column is specified.
                has_node_id = node_id_idx is not None
                node_id = self._parse_column(int, row, node_id_idx, i) if has_node_id else i

                # Validate coordinates
                TileHelper.validate_latitude(latitude)
                TileHelper.validate_longitude(longitude)

                # Instantiate a node and add it to the list.
                node = Node(latitude, longitude, node_id)
                self.nodes.append(node)

        return self.nodes

    def _parse_column(self, parse_type, row, column_index, row_id):
        """
        Parse specified column to a specified type and raise as informative error in case it fails.
        :param parse_type: The target type (int or float).
        :param row: The row (from the .csv file) which is being processed .
        :param column_index: The index of the target column.
        :param row_id: The ID of the row.
        :return: Parsed value of the specified type.
        """
        try:
            value = parse_type(row[column_index])
        except Exception as e:
            msg = "Unable to parse the '{}' column in the row #{} in the file {}"
            raise ValueError(msg.format(self.latitude_column, row_id, self.nodes_path))

        return value

    def _node_id_column_index(self, columns):
        """
        Return node ID column index if node ID column is specified in args and exists in file header.
        :param columns: The list of file column names.
        :return: The index of a node ID column.
        """
        if self.node_id_column is not None:
            # If node ID column is specified
            if self.node_id_column in columns:
                # If exists, find the index
                node_id_idx = columns.index(self.node_id_column)
            else:
                # If it doesn't exist, raise an error.
                raise NameError("Node ID column {} was not found in file's header.".format(self.node_id_column))
        else:
            node_id_idx = None

        return node_id_idx

    def _prep_coordinate_column(self, columns, name, key):
        """Prepares latitude or longitude column name for comparison.
        :param columns: The list of column names from the file header.
        :param name: The column name specified in args.
        :param key: 'lat' or 'lon' (the key in the coordinates alias dictionary).
        :return: The coordinate column name used to find the column file index (used for reading a row value for that column).
        """
        # Returns lower case names for default case insensitive comparison. Otherwise returns unchanged list.
        all_colums = self._prep_list_case(columns)
        # If column name was spacified in args.
        if name is not None:
            # By default returnes lower case name, or in case of case sensitive comparison leaves the name unchanged.
            name = self._prep_name_case(name)
            # Looks for the name in the columns list and raises errors if it doesn't exist or if there are dups.
            name = CSVNodeParser._match_column_name(all_colums, [name])
        else:
            # Find coordinate column.
            aliases = CSVNodeParser._get_column_aliases(key)
            if self.columns_case:
                # In case of case sensitive comparison add a copy of the list with first capital letter.
                aliases.extend([a.title() for a in aliases])

            # Looks for the aliases in the columns list and raises errors if none is found or if there are dups.
            name = CSVNodeParser._match_column_name(all_colums, aliases)

        return name

    def _prep_name_case(self, name):
        """
        Either returns lower case name (default) or original name if case sensitive comparison is used.
        :param name: A columns name.
        :return: A column name ready for file columns header lookup.
        """
        result = name if (name is None or self.columns_case) else name.lower()

        return result

    def _prep_list_case(self, names):
        """
        Either returns lower case list of names (default) or original list if case sensitive comparison is used.
        :param names: A list of names.
        :return: A list of names ready for string comparison.
        """
        result = names if self.columns_case else [v.lower() for v in names]

        return result

    @staticmethod
    def _match_column_name(columns, aliases):
        """
        Find a column name which matches one of the aliases or raise an error if none is found or there are dups.
        :param columns: A list of file column names.
        :param aliases: A list of column aliases.
        :return: The file column names which matches one of the aliases.
        """
        # Find common values in both lists.
        names = []
        for a in aliases:
            # For each alias (from the alias list) filter the file columns list and collect found names.
            found = [c for c in columns if c == a]
            names.extend(found)

        if len(names) == 1:
            # If only one name was found return it.
            name = names[0]
        elif len(names) == 0:
            # If no names was found raise an error.
            raise ValueError('Unable to find {} column'.format(aliases[0]))
        else:
            # If more than one name was found raise an error.
            raise ValueError('Ambiguous {} column names: {}'.format(aliases[0], ','.join(names)))

        return name

    @staticmethod
    def _get_column_aliases(column_key):
        """
        Returns the list of aliases for the given column key.
        :param column_key: One of keys: 'lat' or 'lon'.
        :return: The list of aliases.
        """
        aliases = {
            'lat': ['latitude', 'lat'],
            'lon': ['longitude', 'lon', 'long']
        }

        return aliases[column_key]


class DemographicsNodeParser(NodeParser):
    """Nodes parser for .csv file."""
    def __init__(self, nodes_path):
        super().__init__(nodes_path)
        self.id_reference = None

    def parse_nodes(self):
        try:
            nodes = self._parse_nodes()
        except KeyError as e:
            self._raise_file_exception(e, e.args[0])
        except (json.decoder.JSONDecodeError, Exception ) as e:
            self._raise_file_exception(e)

        return nodes

    def _parse_nodes(self):
        """Reads input nodes from a demographics file."""
        with open(self.nodes_path, 'r') as demog_file:
            content = json.load(demog_file)

        # read metadata (id_ref)
        self.id_reference = content['Metadata']['IdReference']
        content_nodes = content["Nodes"]

        # Load the nodes, for each node read lat. lon and node id and
        for node in content_nodes:
            node_id = int(node['NodeID'])
            latitude = float(node['NodeAttributes']['Latitude'])
            longitude = float(node['NodeAttributes']['Longitude'])

            # Validate coordinates
            TileHelper.validate_latitude(latitude)
            TileHelper.validate_longitude(longitude)

            # Instantiate a node and add it to the list.
            node = Node(latitude, longitude, node_id)
            self.nodes.append(node)

        return self.nodes
