import os
import sys
from collections import defaultdict

from dselib.dse_tools import Point,  Tile


class Node(Point):
    """Node represents a DTK node object. It is a point with an id. The id can be any integer value."""
    def __init__(self, latitude, longitude, node_id, **kwargs):
        super().__init__(latitude, longitude, node_id)
        # Allows nodes to keep other DTK related attributes.
        self.__dict__.update(kwargs)


class TileNodeMap(defaultdict):
    """
    Maps nodes to tiles by grouping them based on geo-coded ids calculated from nodes coordinates and grid's resolution.
    It is a dictionary where keys are tile ids and values are lists of node ids.
    """
    def __init__(self, resolution, nodes=[]):
        # This ensures that each tile has a list to which node ids can be added.
        super().__init__(list)
        # keep track of tile and node objects used to construct the map.
        self._nodes = []
        self._tiles = []
        self.resolution = resolution
        self.add_nodes(nodes)

    @property
    def nodes(self):
        return self._nodes

    @property
    def tiles(self):
        return self._tiles

    def add_nodes(self, nodes):
        """Adds a set of nodes to the map."""
        for node in nodes:
            if node is None:
                continue

            tile = Tile(node.latitude, node.longitude, self.resolution)
            if node.id is None:
                # For simplicity, node ids must be determined before mapping starts.
                raise ValueError("Node id is not defined.")

            # Ensure tile list is unique.
            if tile.id not in self:
                self._tiles.append(tile)

            # Map node to tile.
            self[tile.id].append(node.id)
            # Keep track of actual nodes.
            self._nodes.append(node)

        # Ensure node list is unique.
        self._nodes = list(set(self._nodes))

        # Ensure node id list for each tile is unique and sorted.
        for tile_id in self:
            self._order_node_list(tile_id)

    def add_csv_tiles(self,nodes,tiles, default_fill=False):
        #
        #self._tiles = tiles
        for node in nodes:
            if node.id is None:
                # For simplicity, node ids must be determined before mapping starts.
                raise ValueError("Node id is not defined.")
                continue

            if tile.id not in self:
                if default_fill:
                    tile[-9999].appen(node.id)
                    self._nodes.append(node)
                else:
                    raise ValueError("No climate data for node {0}.".format(node.id))
            else:
                self[tile.id].append(node.id)
                self._nodes.append(node)

    def add_tile_node_ids(self, tile_id, node_ids):
        """Adds mapping directly. This support the load scenario when only ids are known."""
        self[tile_id].extend(node_ids)
        self._order_node_list(tile_id)

    def _order_node_list(self, tile_id):
        """Ensures the list of node ids is unique and sorted."""
        self[tile_id] = sorted(list(set(self[tile_id])))
