class Point:
    def __init__(self, latitude, longitude, id_number=None):
        self.latitude = latitude
        self.longitude = longitude
        self.id = id_number

    def __str__(self):
        return '({}, {}, {})'.format(self.latitude, self.longitude, self.id)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        return hash((self.latitude, self.longitude, self.id))

    def __eq__(self, other):
        return (self.latitude, self.longitude, self.id) == (other.latitude, other.longitude, other.id)

    def __ne__(self, other):
        return not(self == other)


class Tile(Point):
    """
    Tile is a cell of a global grid which center is specified with latitude and longitude and the size is determined by
    the grid's resolution. This class is used for spatial grouping of DTK nodes. That grouping optimizes the process of
    reading and storing weather time series. Nodes are grouped by a geo-coded id calculated based on node's coordinates,
    (latitude, longitude) and grid's resolution. All nodes within the same tile have the same tile id.
    """
    def __init__(self, latitude, longitude, resolution, tile_id=None):
        self.resolution = resolution
        if tile_id is None:
            # Take input node coordinates and 'snap' them to the grid tile (tile center coordinates).
            tile_id = TileHelper.calculate_tile_id(latitude, longitude, resolution)
            center_latitude, center_longitude = TileHelper.tile_coordinates(tile_id, resolution)
        else:
            center_latitude = latitude
            center_longitude = longitude
        super().__init__(center_latitude, center_longitude, tile_id)

    @staticmethod
    def from_point(point, resolution):
        return Tile(point.latitude, point.longitude, resolution)

class TileHelper:
    MIN_LATITUDE = -90
    MAX_LATITUDE = 90
    MIN_LONGITUDE = 0
    MAX_LONGITUDE = 360
    SEC_IN_DEGREE = 3600.0
    KM_IN_DEGREE = 120

    @staticmethod
    def is_longitude_range_0_360(longitudes):
        min_lon = min(longitudes)
        max_lon = max(longitudes)

        longitude_0_360 = min_lon >= 0 and max_lon <= 360

        if not longitude_0_360 and not (min_lon >= -180 and max_lon <= 180):
            raise ValueError("Invalid longitude range: min: {}, max: {}".format(min_lon, max_lon))

        return longitude_0_360

    @staticmethod
    def validate_latitude(latitude):
        """Validate latitude is within the expected range (-90 to 90)."""
        TileHelper._validate_coordinate("Latitude", latitude, TileHelper.MIN_LATITUDE, TileHelper.MAX_LATITUDE)

    @staticmethod
    def validate_longitude(longitude, is_360=None):
        """Validate longitude is within the expected ranges (-180 to 180 or 0 to 360)."""
        min_lon, max_lon = TileHelper.get_longitude_range(is_360)
        TileHelper._validate_coordinate("Longitude", longitude, min_lon, max_lon)

    @staticmethod
    def _validate_coordinate(name, value, min_value, max_value):
        msg = "{} {} is outside of the expected range {} to {}"

        if not min_value <= value <= max_value:
            raise ValueError(msg.format(name.capitalize(), value, min_value, max_value))

    @staticmethod
    def get_longitude_range(is_360):
        """Get min and max values for one of two longitude ranges."""

        if is_360 is None:
            # If is_360 is not set return union of two possible ranges.
            min_lon = TileHelper.MIN_LONGITUDE - 180
            max_lon = TileHelper.MAX_LONGITUDE
        elif is_360 is False:
            min_lon = TileHelper.MIN_LONGITUDE - 180
            max_lon = TileHelper.MAX_LONGITUDE - 180
        else:
            min_lon = TileHelper.MIN_LONGITUDE
            max_lon = TileHelper.MAX_LONGITUDE

        return min_lon, max_lon

    @staticmethod
    def convert_longitude_range(longitude, to_360):
        """Convert between two longitude ranges."""
        if to_360 is None:
            raise ValueError("Longitude range flag not set.")

        TileHelper.validate_longitude(longitude)

        # Convert between longitude ranges
        lon = longitude
        if to_360:
            if -180 <= lon < 0:
                lon = 360 - abs(lon)
            elif lon == 360:
                lon = 0
        else:
            if 180 <= lon <= 360:
                lon = lon - 360

        return lon

    @staticmethod
    def calculate_tile_id(latitude, longitude, resolution):
        """Calculates geo-coded tile id. In case longitude is negative, id will be negative."""
        y = round((latitude - TileHelper.MIN_LATITUDE) / resolution.size_in_degrees)
        x = round((longitude - TileHelper.MIN_LONGITUDE) / resolution.size_in_degrees)
        n = (x << 32) + y
        tile_id = n + 1

        return tile_id

    @staticmethod
    def tile_coordinates(tile_id, resolution):
        """'Unpack tile's center coordinates by reversing the tile id formula."""
        tile_id_1 = tile_id - 1
        y = tile_id_1 & 0xFFFFFFFF
        x = tile_id_1 >> 32
        latitude = (y * resolution.size_in_degrees) + TileHelper.MIN_LATITUDE
        longitude = (x * resolution.size_in_degrees) + TileHelper.MIN_LONGITUDE

        return latitude, longitude


class Units:
    """Resolution units supported by the tool. Range dictionary contains exclusive valid values ranges."""
    arc_second = 'arc_second'
    degree = 'degree'
    km = 'km'
    RANGE = {
        arc_second: {'min': 1, 'max': 3600},
        degree: {'min': 1.0/3600, 'max': 1.0},
        km: {'min': 1.0/30, 'max': 120}
    }

    @staticmethod
    def expected_type(unit):
        """Return expected data type for the given unit."""
        expected = type(Units.RANGE[unit]['max'])

        return expected


class Resolution:
    """
    Resolution is an abstraction of the grid's resolution. It encapsulates the grid resolution information and exposes
    it in frequently used units. The grid resolution is stored as arc seconds because that value is always an integer,
    which prevents loss of precision which would occur if resolution was stored as a float unit (like size in degrees).
    """
    def __init__(self, size, unit):
        # check unit
        if unit not in Units.RANGE:
            raise ValueError("The 'unit' argument have one of the values: {}".format(', '.join(Units.RANGE)))

        # check size type
        if type(size) != Units.expected_type(unit):
            raise ValueError("Resolution in {} must be of type {}".format(unit, Units.expected_type(unit)))

        # check size range
        expected_min = Units.RANGE[unit]['min']
        expected_max = Units.RANGE[unit]['max']
        if not(expected_min <= size <= expected_max):
            raise ValueError("Resolution in {}s must be between {} and {}.".format(unit, expected_min, expected_max))

        # assign or calculate resolution in arc seconds
        if unit == Units.arc_second:
            self._resolution_in_sec = size

        elif unit == Units.degree:
            self._resolution_in_sec = int(size * TileHelper.SEC_IN_DEGREE)

        elif unit == Units.km:
            self._resolution_in_sec = int(TileHelper.SEC_IN_DEGREE * size / TileHelper.KM_IN_DEGREE)

        else:
            raise ValueError("Resolution unit is not supported.")

    @property
    def arc_seconds_minutes_label(self):
        """Returns the resolution label in arc seconds or minutes. Used for naming DTK weather files."""
        if self._resolution_in_sec < 60:
            label = '{}arcsec'.format(self._resolution_in_sec)
        else:
            is_divisible = self._resolution_in_sec % 60 == 0
            arc_min = self._resolution_in_sec / 60.0
            if is_divisible:
                arc_min = int(arc_min)

            label = '{}arcmin'.format(arc_min)

        return label

    @property
    def arc_seconds(self):
        return self._resolution_in_sec

    @property
    def size_in_degrees(self):
        return self._resolution_in_sec / TileHelper.SEC_IN_DEGREE

    @property
    def size_in_km(self):
        return int(TileHelper.KM_IN_DEGREE * self._resolution_in_sec / TileHelper.SEC_IN_DEGREE)

    # Built-in string representation and comparison functions
    # https://docs.python.org/3.6/reference/datamodel.html#object.__str__

    def __str__(self):
        return '({} arc seconds, {} degrees, {} km)'.format(self.arc_seconds, self.size_in_degrees, self.size_in_km)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        return hash(self.arc_seconds)

    def __eq__(self, other):
        return self.arc_seconds == other.arc_seconds

    def __ne__(self, other):
        return not(self == other)

