
import unittest
import sys
import os
from pathlib import Path
import datetime
import numpy as np

from dselib.dse_tools import RasterAggregateTotals, RasterAggregateAverage, RasterKelvinToC, RasterMetersTomm, RasterFToC, RasterFirstIn, RasterLastIn
from dselib.dse_tools import calcType, RasterCalculator

"""
Note Raster Calculation are calculations on numpy arrays
"""

RUN_ALL = os.environ.get('RUN_ALL', False)

Array_meters_idx = np.asarray([[[1.0, 0.5], [0.0, -0.25]], [[1.0, 0.5], [0.0, -0.25]]])
Array_meters = np.asarray([[1.0, 0.5], [0.0, -0.25]])

Array_kelvin = np.array([[0.0, 273.15], [298.15, 373.15]])
Array_kelvin_idx = np.array([[[0.0, 273.15], [298.15, 373.15]], [[0.0, 273.15], [298.15, 373.15]]])

Array_fahrenheit = np.array([[50.0, 32.0], [212.0, -40.]])
Array_fahrenheit_idx = np.array([[[50.0, 32.0], [212.0, -40.0]], [[50.0, 32.0], [212.0, -40.0]]])

class TestCalc_Helper(unittest.TestCase):
    @staticmethod
    def set_random_seed(seed):
        np.random.seed(seed)

    @staticmethod
    def generate3x3():
        # generate a random 3x3 array with values between 0 and 100
        return np.random.random((3,3)) * 100

    @staticmethod
    def generate2x3x3():
        # generate a random 2x3x3 array with values between 0 and 100
        return np.random.random((2,3,3)) * 100

    def testRasterMetersToMMCalculateNoClip(self):
        Target = np.asarray([[1000.0, 500.0], [0.0, -250.0]])
        rcalc = RasterMetersTomm()
        Result = rcalc.calculate(Array_meters)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterMetersToMMCalculateClip(self):
        Target = np.asarray([[900.0, 500.0], [0.0, 0.0]])
        rcalc = RasterMetersTomm()

        Result = rcalc.calculate(Array_meters, 0.0, 900.)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterMetersToMMCalculate_IndexNoClip(self):
        Target = np.array([[[1000.0, 500.0], [0.0, -250.0]], [[1000.0, 500.0], [0.0, -250.0]]])
        rcalc = RasterMetersTomm()
        Result = rcalc.calculate(Array_meters_idx)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterMetersToMMCalculate_IndexNClip(self):
        Target = np.array([[[900.0, 500.0], [0.0, 0.0]], [[900.0, 500.0], [0.0, 0.0]]])
        rcalc = RasterMetersTomm()
        Result = rcalc.calculate(Array_meters_idx, 0.0, 900.0)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterKelvinToCCalculateNoClip(self):
        Target = np.asarray([[-273.15, 0.0], [25., 100.0]])
        rcalc = RasterKelvinToC()
        Result = rcalc.calculate(Array_kelvin)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterKelvinToCCalculateClip(self):
        Target = np.asarray([[-80.0, 0.0], [25, 100.0]])
        rcalc = RasterKelvinToC()
        Result = rcalc.calculate(Array_kelvin, -80.0)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterKelvinToCCalculate_IndexNoClip(self):
        Target = np.asarray([[[-273.15, 0.0], [25., 100.0]], [[-273.15, 0.0], [25., 100.0]]])
        rcalc = RasterKelvinToC()
        Result = rcalc.calculate(Array_kelvin_idx)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterKelvinToCCalculate_IndexClip(self):
        Target = np.asarray([[[-80.0, 0.0], [25, 100.0]], [[-80.0, 0.0], [25, 100.0]]])
        rcalc = RasterKelvinToC()
        Result = rcalc.calculate(Array_kelvin_idx, -80.0)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFToCCalculateNoClip(self):
        Target = np.asarray([[10., 0.0], [100., -40.0]])
        rcalc = RasterFToC()
        Result = rcalc.calculate(Array_fahrenheit)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFToCCalculateClip(self):
        Target = np.asarray([[10., 0.0], [80., -40.0]])
        rcalc = RasterFToC()
        Result = rcalc.calculate(Array_fahrenheit, clip_max = 80.0)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFToCCalculate_IndexNoClip(self):
        Target = np.asarray([[[10., 0.0], [100., -40.0]], [[10., 0.0], [100., -40.0]]])
        rcalc = RasterFToC()
        Result = rcalc.calculate(Array_fahrenheit_idx)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFToCCalculate_IndexClip(self):
        Target = np.asarray([[[10., 0.0], [80., -40.0]], [[10., 0.0], [80., -40.0]]])
        rcalc = RasterFToC()
        Result = rcalc.calculate(Array_fahrenheit_idx, clip_max= 80.0)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterAggregateTotalsCalculate(self):
        # generate some arrays
        TestCalc_Helper.set_random_seed(1000)
        Array1 = TestCalc_Helper.generate3x3()
        Array2 = TestCalc_Helper.generate3x3()
        Array3 = TestCalc_Helper.generate3x3()
        Target = Array1 + Array2 + Array3
        rcalc = RasterAggregateTotals()
        Result = rcalc.calculate(Array1, Array2, Array3)
        self.assertEqual(Result.shape, Array1.shape)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterAggregateTotalsCalculate_Index(self):
        TestCalc_Helper.set_random_seed(1000)
        Array = TestCalc_Helper.generate2x3x3()
        Target = Array[0,:,:] + Array[1,:,:]
        rcalc = RasterAggregateTotals()
        Result = rcalc.calculate_index(Array,[0,1])
        self.assertEqual(Result.shape[0], 3)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterAggregateAverageCalculate(self):
        # generate some arrays
        TestCalc_Helper.set_random_seed(1000)
        Array1 = TestCalc_Helper.generate3x3()
        Array2 = TestCalc_Helper.generate3x3()
        Array3 = TestCalc_Helper.generate3x3()
        Target = (Array1 + Array2 + Array3)/3.0
        rcalc = RasterAggregateAverage()
        Result = rcalc.calculate(Array1, Array2, Array3)
        self.assertEqual(Result.shape, Array1.shape)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterAggregateAverageCalculate_Index(self):
        TestCalc_Helper.set_random_seed(1000)
        Array = TestCalc_Helper.generate2x3x3()
        Target = (Array[0,:,:] + Array[1,:,:])/2.0
        rcalc = RasterAggregateAverage()
        Result = rcalc.calculate_index(Array, [0,1])
        self.assertEqual(Result.shape, (3, 3))
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFirstInCalculate_Index(self):
        TestCalc_Helper.set_random_seed(1000)
        Array = TestCalc_Helper.generate2x3x3()
        Target = Array[0,:,:]
        rcalc = RasterFirstIn()
        Result = rcalc.calculate_index(Array,[0,1])
        self.assertEqual(Result.shape[0], 3)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterFirstInCalculate(self):
        # generate some arrays
        TestCalc_Helper.set_random_seed(1000)
        Array1 = TestCalc_Helper.generate3x3()
        Array2 = TestCalc_Helper.generate3x3()
        Array3 = TestCalc_Helper.generate3x3()
        Target = Array1
        rcalc = RasterFirstIn()
        Result = rcalc.calculate(Array1, Array2, Array3)
        self.assertEqual(Result.shape, Target.shape)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterLastInCalculate_Index(self):
        TestCalc_Helper.set_random_seed(1000)
        Array = TestCalc_Helper.generate2x3x3()
        Target = Array[-1,:,:]
        rcalc = RasterLastIn()
        Result = rcalc.calculate_index(Array,[0,1])
        self.assertEqual(Result.shape[0], 3)
        self.assertTrue(np.array_equal(Result, Target))

    def testRasterLastInCalculate(self):
        # generate some arrays
        TestCalc_Helper.set_random_seed(1000)
        Array1 = TestCalc_Helper.generate3x3()
        Array2 = TestCalc_Helper.generate3x3()
        Array3 = TestCalc_Helper.generate3x3()
        Target = Array3
        rcalc = RasterLastIn()
        Result = rcalc.calculate(Array1, Array2, Array3)
        self.assertEqual(Result.shape, Target.shape)
        self.assertTrue(np.array_equal(Result, Target))

    def testCalculatorType(self):
        calcs = ['average','total','kelvin2c','f2c','meter2mm','firstin','lastin']
        calc_type = calcType.average
        self.assertEqual(calcType.from_string('average'), calcType.average)
        self.assertEqual(calcType.from_string('total'), calcType.total)
        self.assertEqual(calcType.from_string('kelvin2c'),calcType.kelvin2c)
        self.assertEqual(calcType.from_string('f2c'), calcType.f2c)
        self.assertEqual(calcType.from_string('meter2mm'), calcType.meter2mm)
        self.assertEqual(calcType.from_string('firstin'), calcType.firstin)
        self.assertEqual(calcType.from_string('lastin'), calcType.lastin)

    def testCalc_Factory(self):
        self.assertEqual(type(RasterCalculator.calc_factory('average')), type(RasterAggregateAverage()))
        self.assertEqual(type(RasterCalculator.calc_factory('total')), type(RasterAggregateTotals()))
        self.assertEqual(type(RasterCalculator.calc_factory('kelvin2c')), type(RasterKelvinToC()))
        self.assertEqual(type(RasterCalculator.calc_factory('f2c')), type(RasterFToC()))
        self.assertEqual(type(RasterCalculator.calc_factory('meter2mm')), type(RasterMetersTomm()))
        self.assertEqual(type(RasterCalculator.calc_factory('firstin')), type(RasterFirstIn()))
        self.assertEqual(type(RasterCalculator.calc_factory('lastin')), type(RasterLastIn()))

if __name__ == '__main__':
    unittest.main()
