import ctypes
import fnmatch
import json
import operator
import os
import subprocess
import sys
from collections import OrderedDict
from distutils.version import LooseVersion
from enum import Enum
from urllib.error import HTTPError
from urllib.request import Request, urlopen

import pkg_resources

from setuptools import setup, find_packages

current_directory = os.path.dirname(os.path.abspath(__file__))
install_directory = os.path.join(current_directory, 'wheelhouse')

# create the install directory if it does not exist
os.makedirs(install_directory, exist_ok=True)

# Get the installed packages on the system
installed_packages = None

# Prepare the operators
operators = {">": operator.gt, ">=": operator.ge, "<": operator.lt, "<=": operator.le, "==": operator.eq}

class PackageStatus(Enum):
    MISSING = 0
    VALID = 1
    WRONGVERSION = 2

def main():
    global installed_packages
    # Upgrade pip before install other packages
    upgrade_pip()

    # Get the installed packages on the system
    installed_packages = {package.project_name: package.version for package in pkg_resources.working_set}

    # Load the list of requirements
    requirements = json.load(open('requirements.json', 'r'), object_pairs_hook=OrderedDict)

    # Install required packages
    install_packages(requirements)

    # Success !
    print("\n=======================================================")
    print("| dselib and dependencies installed successfully.  |")
    print("=======================================================")

def upgrade_pip():
    completed = subprocess.call(['python','-m', 'pip', 'install', '--upgrade', 'pip'])

def install_packages(reqs):
    """
    Install required packages
    """
    # Go through the requirements
    accept_all = False
    deny_all = False
    for name, val in reqs.items():
        print("\n--- {} ---".format(name))

        val = val or {}
        version = val.get('version', None)
        test = val.get('test', '>=')
        wheel = val.get('wheel', None)
        result = test_package(name, version, test)

        # Valid package -> just display info
        if result == PackageStatus.VALID:
            print("Package {} installed and with correct version.".format(name))

        # Missing -> install
        elif result == PackageStatus.MISSING:
            print("Package {} is missing. Installing...".format(name))
            install_package(name, version, wheel)

        # Wrong version -> Prompt user
        elif result == PackageStatus.WRONGVERSION:
            current_v = installed_packages[name]
            print("Package {} is installed with version {}, but we recommend {}.".format(name, current_v, version))
            if deny_all:
                user_input = 'N'
            elif accept_all:
                user_input = 'Y'
            else:
                user_input = None

            #while user_input not in ('Y', 'N', 'A', 'L'):
            #    user_input = input(
            #        'Would you like to install the recommended version over your system version? #[Y]es/[N]o/Yes to [A]ll/No to a[L]l : ').upper()
            user_input = 'L'
            if user_input == 'Y' or user_input == 'A':
                install_package(name, version, wheel, upgrade=True)
            else:
                print("Keeping system package {} (v. {})".format(name, current_v))

            if user_input == 'A':
                accept_all = True
            if user_input == 'L':
                deny_all = True


    print("---")

def test_package(package, version, test):
    """
    Test if a package is present and with the correct version installed.
    :param package: Package name
    :param version: Needed version
    :param test: can be >= == or <=
    :param extra: used to store extra info
    :return: PackageStatus (MISSING, VALID, WRONGVERSION)
    """
    # The package is not present -> Missing
    if package not in installed_packages:
        return PackageStatus.MISSING

    # The package is in the installed packages
    # The package has no particular version requirement -> all good
    if not version:
        return PackageStatus.VALID

    # The version is required, test if correct
    operator = operators[test]
    if not operator(LooseVersion(installed_packages[package]), LooseVersion(version)):
        return PackageStatus.WRONGVERSION

    # If we made it here, we have the correct package/version
    return PackageStatus.VALID

def install_package(package, version=None, wheel=None, upgrade=False):
    # A wheel is present => easy just install it
    url = None
    if wheel:
        install_str , url = download_wheel(wheel)
        print(install_str)
    else:
        # No wheel, we need to construct the string for pip
        install_str = package
        if version:
            install_str += "=={}".format(version)

    # Handle the upgrade by forcing the reinstall
    install_args = ['pip', 'install', install_str]
    if url:
        install_args.append ("--extra-index-url")
        install_args.append(url)
    print(install_args)
    if upgrade:
        subprocess.call(['pip', 'uninstall', package])
    subprocess.call(install_args)

def download_wheel(wheel):
    # Prepare the urls
    # may want to use multiple package managers in the future
    url =  "https://packages.idmod.org/api/pypi/idm-pypi-production/simple"
    return wheel, url


if __name__ == "__main__":
    # check python installation first
    if ctypes.sizeof(ctypes.c_voidp) != 8 or sys.version_info < (3, 6):
        print("""\nFATAL ERROR: dselib only supports Python 3.6 x64 and above.\n
         Please download and install a x86-64 version of python at:\n
        - Windows: https://www.python.org/downloads/windows/
        Installation is now exiting...""")
        exit()

    main()
