import json
import math
import os

import pandas as pd

from idm_test.dtk_test.general_support import CampaignKeys, ConfigKeys, DemographicsKeys
from idm_test.dtk_test.sft import has_match, get_val

class MyDemographicsKeys:
    InitialPopulation = DemographicsKeys.NodesKeys.InitialPopulation
    InfectivityReservoirSize = DemographicsKeys.NodesKeys.InfectivityReservoirSize
    

class Stdout:
    prob = "prob="
    stat_pop = "StatPop: "
    infected = "Infected: "
    contagion = "Contagion"


matches = ["Update(): Time: ",
           "total contagion = "
           ]
NATURAL_LOG_OF_2 = 0.693147  # round(math.log(2), 6)


class Constants:
    DAYS_IN_YEAR = 365


def calculate_infectiousness(infected_pop, index, simulation_timestep, phase, base_infectivity, amplitude, debug=False):
    """
    Calculate infectiousness at each time step
    Args:
        infected_pop:
        index:
        simulation_timestep:
        phase:
        base_infectivity:
        amplitude:
        debug:

    Returns:

    """
    timestep = index * simulation_timestep
    day_in_year = float(timestep % Constants.DAYS_IN_YEAR)
    infectiousness = base_infectivity * (
            1.0 + amplitude * math.sin(2.0 * math.pi * (day_in_year - phase) / Constants.DAYS_IN_YEAR))
    infectiousness *= infected_pop
    if debug:
        print(infectiousness)
    return infectiousness


def load_campaign_file(campaign_filename="campaign.json", debug=False):
    """
    Reads campaign file to object
    Args:
        campaign_filename: campaign.json file
        debug:

    Returns:
        campaign_obj structure, dictionary with KEY_START_DAY, etc., keys (e.g.)

    """
    with open(campaign_filename) as infile:
        cf = json.load(infile)
    campaign_obj = {}
    start_day = cf["Events"][0][CampaignKeys.Start_Day]
    campaign_obj[CampaignKeys.Start_Day] = int(start_day)
    coverage = cf["Events"][0]["Event_Coordinator_Config"][CampaignKeys.Demographic_Coverage]
    campaign_obj[CampaignKeys.Demographic_Coverage] = float(coverage)

    if debug:
        with open("DEBUG_campaign_object.json", 'w') as outfile:
            json.dump(campaign_obj, outfile, indent=4)

    return campaign_obj


def load_demo_file(demo_filename="demographics.json", demo_path="Assets", debug=False):
    """
    Read demographics.json to object

    Args:
        demo_filename: demographics.json
        demo_path: Assets in COMPS
        debug:

    Returns:demo_obj structure, dictionary

    """
    with open(os.path.join(demo_path, demo_filename), 'r') as infile:
        demo_json = json.load(infile)
    demo_obj = {}
    for demo_param in [getattr(MyDemographicsKeys, attr) for attr in dir(MyDemographicsKeys) if not attr.startswith("__")]:
        if demo_param in demo_json["Nodes"][0]["NodeAttributes"]:
            demo_obj[demo_param] = demo_json["Nodes"][0]["NodeAttributes"][demo_param]

    if debug:
        with open("DEBUG_demo_object.json", 'w') as outfile:
            json.dump(demo_obj, outfile, indent=4)

    return demo_obj


def parse_stdout_file(stdout_filename="stdout.txt", simulation_timestep=1, debug=False):
    """
    Creates a dataframe to store filtered information for each time step
    Args:
        stdout_filename: stdout.txt
        simulation_timestep:
        debug:

    Returns: dataframe of  stdout
    """

    filtered_lines = []
    with open(stdout_filename) as logfile:
        for line in logfile:
            if has_match(line, matches):
                filtered_lines.append(line)
    if debug:
        with open("DEBUG_filtered_lines.txt", "w") as outfile:
            outfile.writelines(filtered_lines)

    # initialize variables
    time_step = index = 0
    stdout_df = pd.DataFrame(columns=[ConfigKeys.Simulation_Timestep, Stdout.stat_pop, Stdout.infected,
                                      Stdout.contagion, Stdout.prob])
    stdout_df.index.name = 'index'
    contagion = prob = 0
    for line in filtered_lines:
        if matches[0] in line:
            time_step += simulation_timestep
            stat_pop = int(get_val(Stdout.stat_pop, line))
            infected = int(get_val(Stdout.infected, line))
            stdout_df.loc[index] = [time_step, stat_pop, infected, contagion, prob]
            contagion = 0
            prob = 0
            index += 1
        elif matches[1] in line:
            contagion = float(get_val(matches[1], line))
            prob = float(get_val(Stdout.prob, line))
    if debug:
        res_path = r'./DEBUG_filtered_from_logging.csv'
        stdout_df.to_csv(res_path)
    return stdout_df
