#!/usr/bin/python

import time
import re
import os
from hashlib import md5

from typing import Union
import json

"""
This module centralizes some small bits of functionality for SFT tests
to make sure we are using the same strings for messages and filenames.
"""

sft_output_filename = "scientific_feature_report.txt"
sft_no_test_data = "BAD: Relevant test data not found.\n"
sft_test_filename = "test.txt"
SFT_EOF = "[Controller] Exiting"
DAYS_IN_MONTH = 30
DAYS_IN_YEAR = 365
MONTHS_IN_YEAR = 12


def start_report_file(output_filename, config_name, already_started=False):
    """
    Start a file handle to write config_name and time step into an output file.
    Args:
        output_filename:
        config_name:
        already_started:

    Returns:

    """
    if already_started:
        with open(output_filename, 'a') as outfile:
            outfile.write("Beginning validation for {0} at time {1}.\n".format(
                config_name,
                time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
            ))
    else:
        with open(output_filename, 'w') as outfile:
            outfile.write("Beginning validation for {0} at time {1}.\n".format(
                config_name,
                time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
            ))
    

def format_success_msg(success):
    """
    String to report a SFT result.
    Args:
        success:

    Returns:

    """
    report_msg = "SUMMARY: Success={0}\n".format(success)
    return report_msg


def add_time_stamp(time_stamp, line):
    """
    Add time stamp to a line
    Args:
        time_stamp:
        line:

    Returns:

    """
    temp_array = line.split()
    temp_array.append("time= " + str(time_stamp))
    new_line = " ".join(temp_array) + "\n"
    return new_line


def get_val(key, line):
    """
    Get algebraic value from a string with regex
    Args:
        key:
        line:

    Returns:

    """
    regex = key + r"(\d*\.*\d*)"
    match = re.search(regex, line)
    if match is not None:
        return match.group(1)
    else:
        raise LookupError


def wait_for_done(filename=sft_test_filename):
    """
    Wait until simulation is done to start dtk_post_process
    Args:
        filename: test.txt or StdOut.txt

    Returns:

    """

    done_writing = False
    while not done_writing:
        with open(filename) as test_file:
            for line_val in test_file:
                if SFT_EOF in line_val:
                    done_writing = True
                    break

    with open("touchfile", "w") as touchfile:
        touchfile.write("{} file completely written. Move on to read.\n".format(filename))
        touchfile.write("Output size = {}\n".format(os.path.getsize(filename)))
        touchfile.write(str(md5_hash_of_file(filename)))

    return


def md5_hash(handle):
    """
    Get mds hash from file handle
    Args:
        handle:

    Returns:

    """
    handle.seek(0)
    md5calc = md5()
    while True:
        data = handle.read(10240)  # reasonable value from examples
        if len(data) == 0:
            break
        md5calc.update(data.encode("UTF-8"))
    hash = md5calc.hexdigest()
    return hash


def md5_hash_of_file(filename):
    """
    Get mds_hash from file
    Args:
        filename:

    Returns:

    """
    # print( "Getting md5 for " + filename )
    file_handle = open(filename)
    hash = md5_hash(file_handle)
    file_handle.close()
    # hash = md5calc.hexdigest()
    return hash


def has_match(target, matches):
    """
    Check if any string from a list(matches) is in the target string.
    Args:
        target:
        matches:

    Returns:

    """
    for match in matches:
        if match in target:
            return True
    return False


def get_char(key, line):
    """
    Get numerical values from a string with regex
    Args:
        key:
        line:

    Returns:

    """
    regex = key + r"(\w*\d*\w*\d*)"
    match = re.search(regex, line)
    if match:
        return match.group(1)
    else:
        raise LookupError


def get_config_parameter(config_filename="config.json", parameters: Union[list, str] = "Config_Name"):
    """
    Get values for given config parameters from config json file.
    Args:
        config_filename:
        parameters:

    Returns:

    """
    with open(config_filename, 'r') as config_file:
        cf = json.load(config_file)["parameters"]
        try:
            if isinstance(parameters, str):
                return [cf[parameters]]
            elif isinstance(parameters, list):
                res = []
                for parameter in parameters:
                    res.append(cf[parameter])
                return res
            else:
                raise ValueError(f"parameters is {type(parameters)}, expected list or str.\n")
        except KeyError as ke:
            raise KeyError(f"{ke} is not found in {config_filename}.\n")


def get_config_name(config_filename="config.json"):
    """
    Get value of Config_Name parameter from config json file.
    Args:
        config_filename:

    Returns:

    """
    return get_config_parameter(config_filename, "Config_Name")[0]


def get_simulation_duration(config_filename="config.json"):
    """
    Get value of Simulation_Duration parameter from config json file.
    Args:
        config_filename:

    Returns:

    """
    return get_config_parameter(config_filename, "Simulation_Duration")[0]


def get_simulation_timestep(config_filename="config.json"):
    """
    Get value of Simulation_Timestep parameter from config json file.
    Args:
        config_filename:

    Returns:

    """
    return get_config_parameter(config_filename, "Simulation_Timestep")[0]
