"""
Here we implement the JSON Metadata operations.

Copyright 2021, Bill & Melinda Gates Foundation. All rights reserved.
"""
import os
import json
from pathlib import Path
from typing import TYPE_CHECKING, Dict, List, Type, Union
from dataclasses import dataclass, field
from idmtools.core import ItemType
from idmtools.core.interfaces import imetadata_operations
from idmtools.entities import Suite
from idmtools.entities.experiment import Experiment
from idmtools.entities.simulation import Simulation
from idmtools.utils.json import IDMJSONEncoder
from idmtools_platform_file.platform_operations.utils import FileSuite, FileExperiment

if TYPE_CHECKING:
    from idmtools_platform_file.file_platform import FilePlatform


@dataclass
class JSONMetadataOperations(imetadata_operations.IMetadataOperations):
    """
    JSON operations used in File Platform.
    """
    platform: 'FilePlatform'  # noqa: F821
    platform_type: Type = field(default=None)
    metadata_filename: str = field(default='metadata.json')

    @staticmethod
    def _read_from_file(filepath: Union[Path, str]) -> Dict:
        """
        Utility: read metadata from a file.
        Args:
            filepath: metadata file path
        Returns:
            JSON
        """
        filepath = Path(filepath)
        with filepath.open(mode='r') as f:
            metadata = json.load(f)
        return metadata

    @staticmethod
    def _write_to_file(filepath: Union[Path, str], data: Dict, indent: int = None) -> None:
        """
        Utility: save metadata to a file.
        Args:
            filepath: metadata file path
            data: metadata as dictionary
            indent: indent level for pretty printing the JSON file. None for compact JSON.
        Returns:
            None
        """
        filepath = Path(filepath)
        filepath.parent.mkdir(parents=True, exist_ok=True)
        with filepath.open(mode='w') as f:
            json.dump(data, f, indent=indent, cls=IDMJSONEncoder)

    def get_metadata_filepath(self, item: Union[Suite, Experiment, Simulation]) -> Path:
        """
        Retrieve item's metadata file path.
        Args:
            item: idmtools entity (Suite, Experiment and Simulation)
        Returns:
            item's metadata file path
        """
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("get_metadata_filepath method supports Suite/Experiment/Simulation only.")
        item_dir = self.platform.get_directory(item)
        filepath = Path(item_dir, self.metadata_filename)
        return filepath

    def get_metadata_filepath_by_id(self, item_id: str, item_type: ItemType) -> Path:
        """
        Retrieve item's metadata file path.
        Args:
            item_id: item id
            item_type: the type of metadata to search for matches (simulation, experiment, suite, etc.)
        Returns:
            item's metadata file path
        """
        if item_type not in (ItemType.SUITE, ItemType.EXPERIMENT, ItemType.SIMULATION):
            raise RuntimeError("get_metadata_filepath method supports Suite/Experiment/Simulation only.")
        item_dir = self.platform.get_directory_by_id(item_id, item_type)
        filepath = Path(item_dir, self.metadata_filename)
        return filepath

    def get(self, item: Union[Suite, Experiment, Simulation]) -> Dict:
        """
        Obtain item's metadata.
        Args:
            item: idmtools entity (Suite, Experiment and Simulation)
        Returns:
             key/value dict of metadata from the given item
        """
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("Get method supports Suite/Experiment/Simulation only.")
        data = item.to_dict()
        if isinstance(item, Suite):
            data.pop('experiments', None)
        meta = json.loads(json.dumps(data, cls=IDMJSONEncoder))
        meta['id'] = meta['_uid']
        meta['uid'] = meta['_uid']
        meta['status'] = 'CREATED'
        meta['dir'] = os.path.abspath(self.platform.get_directory(item))

        if isinstance(item, Suite):
            meta['experiments'] = [experiment.id for experiment in item.experiments]
        elif isinstance(item, Experiment):
            meta['suite_id'] = meta["parent_id"]
            meta['simulations'] = [simulation.id for simulation in item.simulations]
        elif isinstance(item, Simulation):
            meta['experiment_id'] = meta["parent_id"]
        return meta

    def dump(self, item: Union[Suite, Experiment, Simulation]) -> Dict:
        """
        Save item's metadata to a file and also save tags.json file.
        Args:
            item: idmtools entity (Suite, Experiment and Simulation)
        Returns:
            key/value dict of metadata from the given item
        """
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("Dump method supports Suite/Experiment/Simulation only.")
        dest = self.get_metadata_filepath(item)
        meta = self.get(item)
        self._write_to_file(dest, meta)

        # Also write tags.json file
        keys_to_extract = ["id", "item_type", "tags"]
        extracted = {key: meta[key] for key in keys_to_extract}

        tags_path = dest.parent / "tags.json"
        self._write_to_file(tags_path, extracted, indent=2)
        return meta

    def load(self, item: Union[Suite, Experiment, Simulation]) -> Dict:
        """
        Obtain item's metadata file.
        Args:
            item: idmtools entity (Suite, Experiment and Simulation)
        Returns:
             key/value dict of metadata from the given item
        """
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("Load method supports Suite/Experiment/Simulation only.")
        meta_file = self.get_metadata_filepath(item)
        meta = self._read_from_file(meta_file)
        return meta

    def load_from_file(self, metadata_filepath: Union[Path, str]) -> Dict:
        """
        Obtain the metadata for the given filepath.
        Args:
            metadata_filepath: str
        Returns:
             key/value dict of metadata from the given filepath
        """
        if not Path(metadata_filepath).exists():
            raise RuntimeError(f"File not found: '{metadata_filepath}'.")
        meta = self._read_from_file(metadata_filepath)
        return meta

    def update(self, item: Union[Suite, Experiment, Simulation], metadata: Dict = None, replace=True) -> None:
        """
        Update or replace item's metadata file.
        Args:
            item: idmtools entity (Suite, Experiment and Simulation.)
            metadata: dict to be updated or replaced
            replace: True/False
        Returns:
             None
        """
        if metadata is None:
            metadata = {}
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("Set method supports Suite/Experiment/Simulation only.")
        meta = metadata
        if not replace:
            meta = self.load(item)
            meta.update(metadata)
        meta_file = self.get_metadata_filepath(item)
        self._write_to_file(meta_file, meta)

    def clear(self, item: Union[Suite, Experiment, Simulation]) -> None:
        """
        Clear the item's metadata file.
        Args:
            item: clear the item's metadata file
        Returns:
            None
        """
        if not isinstance(item, (Suite, Experiment, Simulation)):
            raise RuntimeError("Clear method supports Suite/Experiment/Simulation only.")
        self.update(item=item, metadata={}, replace=True)

    def get_children(self, item: Union[Suite, Experiment, FileSuite, FileExperiment]) -> List[Dict]:
        """
        Fetch item's children.
        Args:
            item: idmtools entity (Suite, FileSuite, Experiment, FileExperiment)
        Returns:
            Lis of metadata
        """
        if not isinstance(item, (Suite, FileSuite, Experiment, FileExperiment)):
            raise RuntimeError("Get children method supports [File]Suite and [File]Experiment only.")
        item_list = []
        if isinstance(item, (Suite, FileSuite)):
            meta = self.load(item)
            for exp_id in meta['experiments']:
                meta_file = self.get_metadata_filepath_by_id(exp_id, ItemType.EXPERIMENT)
                exp_meta = self._read_from_file(meta_file)
                item_list.append(exp_meta)
        else:
            item_dir = self.platform.get_directory(item)
            pattern = f'*/{self.metadata_filename}'
            for meta_file in item_dir.glob(pattern=pattern):
                meta = self.load_from_file(meta_file)
                item_list.append(meta)
        return item_list

    def get_all(self, item_type: ItemType, item_id: str = '') -> List[Dict]:
        """
        Obtain all the metadata for a given item type.
        Args:
            item_type: the type of metadata to search for matches (simulation, experiment, suite, etc.)
            item_id: item id
        Returns:
            list of metadata with given item type
        """
        root = Path(self.platform.job_directory)
        item_list = []

        if item_type is ItemType.SIMULATION:
            # Match sim under experiment, under optional suite
            patterns = [
                f"s_*/e_*/*{item_id}/{self.metadata_filename}",  # suite/experiment/simulation
                f"e_*/*{item_id}/{self.metadata_filename}",  # experiment/simulation (no suite)
            ]
        elif item_type is ItemType.EXPERIMENT:
            patterns = [
                f"s_*/e_*{item_id}/{self.metadata_filename}",  # suite/experiment
                f"e_*{item_id}/{self.metadata_filename}",  # standalone experiment
            ]
        elif item_type is ItemType.SUITE:
            patterns = [
                f"s_*{item_id}/{self.metadata_filename}",  # suite only
            ]
        else:
            raise RuntimeError(f"Unknown item type: {item_type}")

        # Search each pattern
        for pattern in patterns:
            for meta_file in root.glob(pattern):
                try:
                    meta = self.load_from_file(meta_file)
                    item_list.append(meta)
                except Exception as e:
                    print(f"Warning: Failed to load metadata from {meta_file}: {e}")

        return item_list

    @staticmethod
    def _match_filter(item: Dict, metadata: Dict, ignore_none=True):
        """
        Utility: verify if item match metadata.
        Note: compare key/value if value is not None else just check key exists
        Args:
            item: dict represents metadata of Suite/Experiment/Simulation
            metadata: dict as a filter
            ignore_none: True/False (ignore None value or not)
        Returns:
            list of Dict items
        """
        for k, v in metadata.items():
            if ignore_none:
                if v is None:
                    is_match = k in item
                else:
                    is_match = k in item and item[k] == v
            else:
                if v is None:
                    is_match = k in item and item[k] is None
                else:
                    is_match = k in item and item[k] == v
            if not is_match:
                return False
        return True

    def filter(self, item_type: ItemType, property_filter: Dict = None, tag_filter: Dict = None,
               meta_items: List[Dict] = None, ignore_none=True) -> List[Dict]:
        """
        Obtain all items that match the given properties key/value pairs passed.
        The two filters are applied on item with 'AND' logical checking.
        Args:
            item_type: the type of items to search for matches (simulation, experiment, suite, etc.)
            property_filter: a dict of metadata key/value pairs for exact match searching
            tag_filter: a dict of metadata key/value pairs for exact match searching
            meta_items: list of metadata
            ignore_none: True/False (ignore None value or not)
        Returns:
            a list of metadata matching the properties key/value with given item type
        """
        if meta_items is None:
            item_id = property_filter["id"] if property_filter and "id" in property_filter else ''
            meta_items = self.get_all(item_type, item_id=item_id)
        item_list = []
        for meta in meta_items:
            is_match = True
            if property_filter:
                is_match = self._match_filter(meta, property_filter, ignore_none=ignore_none)
            if tag_filter:
                is_match = is_match and self._match_filter(meta['tags'], tag_filter, ignore_none=ignore_none)
            if is_match:
                item_list.append(meta)
        return item_list
