"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const core_2 = require("@aws-solutions-constructs/core");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_2.generateIntegStackName(__filename), {
    env: { account: core_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", myVpc, false, undefined, {
    internetFacing: false,
    vpc: myVpc,
}, true, {
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    autoDeleteObjects: true,
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcucHVibGljQXBpRXhpc3RpbmdSZXNvdXJjZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5wdWJsaWNBcGlFeGlzdGluZ1Jlc291cmNlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsVUFBVTtBQUNWLHdDQUErRDtBQUMvRCxnQ0FBdUQ7QUFDdkQseURBQXdFO0FBQ3hFLDhDQUE4QztBQUM5QywyREFBMkQ7QUFDM0QsOENBQW1FO0FBRW5FLGlGQUFpRjtBQUNqRix5RkFBeUY7QUFDekYsMkVBQTJFO0FBRTNFLFFBQVE7QUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSw2QkFBc0IsQ0FBQyxVQUFVLENBQUMsRUFBRTtJQUMvRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsVUFBRyxDQUFDLFVBQVUsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0NBQ3RELENBQUMsQ0FBQztBQUNILEtBQUssQ0FBQyxlQUFlLENBQUMsV0FBVztJQUMvQix1RUFBdUUsQ0FBQztBQUUxRSxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBRXpDLE1BQU0sTUFBTSxHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO0lBQ25ELEdBQUcsRUFBRSxLQUFLO0lBQ1YsZ0JBQWdCLEVBQUUsS0FBSztDQUN4QixDQUFDLENBQUM7QUFFSCxNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFO0lBQ3pELG1CQUFtQixFQUFFO1FBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7UUFDeEIsR0FBRyxFQUFFLEtBQUs7UUFDVixjQUFjLEVBQUUsQ0FBQyxNQUFNLENBQUM7S0FDekI7Q0FDRixDQUFDLENBQUM7QUFFSCxNQUFNLFlBQVksR0FBRyxRQUFRLENBQUMsU0FBUyxDQUNyQyxLQUFLLEVBQ0wsY0FBYyxFQUNkLEtBQUssRUFDTCxLQUFLLEVBQ0wsU0FBUyxFQUNUO0lBQ0UsY0FBYyxFQUFFLEtBQUs7SUFDckIsR0FBRyxFQUFFLEtBQUs7Q0FDWCxFQUNELElBQUksRUFDSjtJQUNFLGFBQWEsRUFBRSxvQkFBYSxDQUFDLE9BQU87SUFDcEMsaUJBQWlCLEVBQUUsSUFBSTtDQUN4QixDQUNGLENBQUM7QUFFRixNQUFNLEtBQUssR0FBcUI7SUFDOUIsaUJBQWlCLEVBQUUsY0FBYztJQUNqQyx1QkFBdUIsRUFBRSxZQUFZO0lBQ3JDLFdBQVcsRUFBRSxLQUFLO0lBQ2xCLGFBQWEsRUFBRTtRQUNiLFFBQVEsRUFBRSxNQUFNO0tBQ2pCO0lBQ0QsU0FBUyxFQUFFLElBQUk7Q0FDaEIsQ0FBQztBQUNGLE1BQU0sV0FBVyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBRTlELFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFO0lBQ2pEO1FBQ0UsRUFBRSxFQUFFLEtBQUs7UUFDVCxNQUFNLEVBQ0osd0VBQXdFO0tBQzNFO0NBQ0YsQ0FBQyxDQUFDO0FBRUgsTUFBTSxnQkFBZ0IsR0FBRyxXQUFXLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO0tBQzVFLElBQUksQ0FBQyxZQUFnQyxDQUFDO0FBQ3pDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRTtJQUM3QyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLDJDQUEyQyxFQUFFO0lBQ2xFLEVBQUUsRUFBRSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsOEJBQThCLEVBQUU7SUFDcEQsRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSw4QkFBOEIsRUFBRTtDQUNyRCxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsbUJBQW1CLENBQUMsTUFBTSxFQUFFO0lBQ25DLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsMkNBQTJDLEVBQUU7Q0FDbkUsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLGlDQUFpQyxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ2xELFFBQVE7QUFDUixHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMiBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IEF3cywgQXBwLCBTdGFjaywgUmVtb3ZhbFBvbGljeSB9IGZyb20gXCJAYXdzLWNkay9jb3JlXCI7XG5pbXBvcnQgeyBBbGJUb0xhbWJkYSwgQWxiVG9MYW1iZGFQcm9wcyB9IGZyb20gXCIuLi9saWJcIjtcbmltcG9ydCB7IGdlbmVyYXRlSW50ZWdTdGFja05hbWUgfSBmcm9tIFwiQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSBcIkBhd3MtY2RrL2F3cy1sYW1iZGFcIjtcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gXCJAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmVcIjtcbmltcG9ydCB7IFNlY3VyaXR5R3JvdXAsIENmblNlY3VyaXR5R3JvdXAgfSBmcm9tIFwiQGF3cy1jZGsvYXdzLWVjMlwiO1xuXG4vLyBOb3RlOiBBbGwgaW50ZWdyYXRpb24gdGVzdHMgZm9yIGFsYiBhcmUgZm9yIEhUVFAgQVBJcywgYXMgY2VydGlmaWNhdGVzIHJlcXVpcmVcbi8vIHZhbGlkYXRpb24gdGhyb3VnaCBETlMgYW5kIGVtYWlsLiBUaGlzIHZhbGlkYXRpb24gaXMgaW1wb3NzaWJsZSBkdXJpbmcgb3VyIGludGVncmF0aW9uXG4vLyB0ZXN0cyBhbmQgdGhlIHN0YWNrIHdpbGwgZmFpbCB0byBsYXVuY2ggd2l0aCBhbiB1bnZhbGlkYXRlZCBjZXJ0aWZpY2F0ZS5cblxuLy8gU2V0dXBcbmNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbmNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgZ2VuZXJhdGVJbnRlZ1N0YWNrTmFtZShfX2ZpbGVuYW1lKSwge1xuICBlbnY6IHsgYWNjb3VudDogQXdzLkFDQ09VTlRfSUQsIHJlZ2lvbjogXCJ1cy1lYXN0LTFcIiB9LFxufSk7XG5zdGFjay50ZW1wbGF0ZU9wdGlvbnMuZGVzY3JpcHRpb24gPVxuICBcIkludGVncmF0aW9uIFRlc3QgZm9yIHB1YmxpYyBIVFRQIEFQSSB3aXRoIGEgZXhpc3RpbmcgZnVuY3Rpb24gYW5kIEFMQlwiO1xuXG5jb25zdCBteVZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2spO1xuXG5jb25zdCB0ZXN0U2cgPSBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgXCJsYW1iZGEtc2dcIiwge1xuICB2cGM6IG15VnBjLFxuICBhbGxvd0FsbE91dGJvdW5kOiBmYWxzZSxcbn0pO1xuXG5jb25zdCBsYW1iZGFGdW5jdGlvbiA9IGRlZmF1bHRzLmJ1aWxkTGFtYmRhRnVuY3Rpb24oc3RhY2ssIHtcbiAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xNF9YLFxuICAgIGhhbmRsZXI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgIHZwYzogbXlWcGMsXG4gICAgc2VjdXJpdHlHcm91cHM6IFt0ZXN0U2ddLFxuICB9LFxufSk7XG5cbmNvbnN0IGxvYWRCYWxhbmNlciA9IGRlZmF1bHRzLk9idGFpbkFsYihcbiAgc3RhY2ssXG4gIFwiZXhpc3RpbmctYWxiXCIsXG4gIG15VnBjLFxuICBmYWxzZSxcbiAgdW5kZWZpbmVkLFxuICB7XG4gICAgaW50ZXJuZXRGYWNpbmc6IGZhbHNlLFxuICAgIHZwYzogbXlWcGMsXG4gIH0sXG4gIHRydWUsXG4gIHtcbiAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgYXV0b0RlbGV0ZU9iamVjdHM6IHRydWUsXG4gIH1cbik7XG5cbmNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICBleGlzdGluZ0xhbWJkYU9iajogbGFtYmRhRnVuY3Rpb24sXG4gIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBsb2FkQmFsYW5jZXIsXG4gIGV4aXN0aW5nVnBjOiBteVZwYyxcbiAgbGlzdGVuZXJQcm9wczoge1xuICAgIHByb3RvY29sOiBcIkhUVFBcIixcbiAgfSxcbiAgcHVibGljQXBpOiB0cnVlLFxufTtcbmNvbnN0IGFsYlRvTGFtYmRhID0gbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCBcInRlc3Qtb25lXCIsIHByb3BzKTtcblxuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyhhbGJUb0xhbWJkYS5saXN0ZW5lciwgW1xuICB7XG4gICAgaWQ6IFwiVzU2XCIsXG4gICAgcmVhc29uOlxuICAgICAgXCJBbGwgaW50ZWdyYXRpb24gdGVzdHMgbXVzdCBiZSBIVFRQIGJlY2F1c2Ugb2YgY2VydGlmaWNhdGUgbGltaXRhdGlvbnMuXCIsXG4gIH0sXG5dKTtcblxuY29uc3QgbmV3U2VjdXJpdHlHcm91cCA9IGFsYlRvTGFtYmRhLmxvYWRCYWxhbmNlci5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXVxuICAubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuU2VjdXJpdHlHcm91cDtcbmRlZmF1bHRzLmFkZENmblN1cHByZXNzUnVsZXMobmV3U2VjdXJpdHlHcm91cCwgW1xuICB7IGlkOiBcIlcyOVwiLCByZWFzb246IFwiQ0RLIGNyZWF0ZWQgcnVsZSB0aGF0IGJsb2NrcyBhbGwgdHJhZmZpYy5cIiB9LFxuICB7IGlkOiBcIlcyXCIsIHJlYXNvbjogXCJSdWxlIGRvZXMgbm90IGFwcGx5IGZvciBFTEIuXCIgfSxcbiAgeyBpZDogXCJXOVwiLCByZWFzb246IFwiUnVsZSBkb2VzIG5vdCBhcHBseSBmb3IgRUxCLlwiIH0sXG5dKTtcblxuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyh0ZXN0U2csIFtcbiAgeyBpZDogXCJXMjlcIiwgcmVhc29uOiBcIkNESyBjcmVhdGVkIHJ1bGUgdGhhdCBibG9ja3MgYWxsIHRyYWZmaWMuXCIgfSxcbl0pO1xuXG5kZWZhdWx0cy5zdXBwcmVzc0F1dG9EZWxldGVIYW5kbGVyV2FybmluZ3Moc3RhY2spO1xuLy8gU3ludGhcbmFwcC5zeW50aCgpO1xuIl19