"""
The parser implementation.

.. include:: ./pdoc/ast.md
"""
# Note for those reading the source code: some of the docstrings in this module refer to the above
# included markdown file.

import dataclasses
import enum
import json
import logging
import types
import typing
import typing as t

from . import errors

__all__ = (
    "AST",
    "ASTNode",
    "ASTNodeType",
    "ASTStateError",
    "Token",
    "Tokenizer",
    "TokenType",
    "CharStream",
    "StringStream",
    "REPLStream",
    "parse_scroll",
    "parse_statement"
)

logger = logging.getLogger(__name__)

ParserT = t.Callable[['ParseContext'], 'ASTNode']

EOF = ""
COMMAND_SEP = [";", "\n"]
BLOCK_OPEN = "{"
BLOCK_CLOSE = "}"
OPEN_PAREN = "("
CLOSE_PAREN = ")"
CONTROL_SIGIL = "!"
EXPANSION_SIGIL = "$"
MULTI_SIGIL = "^"
COMMENT_SIGIL = "#"
QUOTE = "\""
ESCAPE_SIGIL = "\\"


class TokenType(enum.Enum):
    """
    All possible token types generated by `Tokenizer`.
    """

    OPEN_PAREN = 1
    CLOSE_PAREN = 2
    OPEN_BLOCK = 3
    CLOSE_BLOCK = 4
    EXPANSION_SIGIL = 5
    MULTI_SIGIL = 6
    CONTROL_SIGIL = 7
    COMMAND_SEP = 8
    STRING_LITERAL = 9
    EOF = 10
    WHITESPACE = 11
    COMMENT = 12


class TokenizeConsumeRestState(enum.Enum):
    """
    Internal `Tokenizer` state for the CONSUME_REST feature.
    """

    OFF = 0
    COUNTING = 1
    CONSUME = 2


class ASTNodeType(enum.Enum):
    """
    All possible AST node types.
    """

    NONE = 0
    """AST nodes with this type should be ignored."""

    EOF = 1
    """A node representing EOF."""

    ROOT = 2
    """The root AST node."""

    STRING = 3
    """A string literal."""

    COMMAND_CALL = 4
    """The parent node for a command call."""

    COMMAND_ARGUMENTS = 5
    """The parent node for a list of command arguments."""

    BLOCK = 6
    """A block of statements."""

    CONTROL_CALL = 7
    """The parent node for a control call."""

    CONTROL_ARGUMENTS = 8
    """The parent node for a list of control arguments."""

    EXPANSION = 9
    """The parent node for an expansion, either variable or call."""

    EXPANSION_SINGLE = 10
    """Indicates an expansion should use normal expansion."""

    EXPANSION_MULTI = 11
    """Indicates an expansion should use vector expansion."""

    EXPANSION_VAR = 12
    """Parent node for a variable expansion."""

    EXPANSION_CALL = 13
    """Parent node for an expansion call."""

    EXPANSION_ARGUMENTS = 14
    """Parent node for a list of expansion arguments. Applies only to calls."""


# TODO: Add slots=True for python 3.10
@dataclasses.dataclass
class Token:
    """A token. See [Tokenizing](#tokenizing)."""

    type: TokenType
    """The type of this token."""

    value: str
    """The value of this token."""

    line: int
    """The line this token *started* generating on. Some tokens may span multiple lines."""

    position: int
    """The column along the line that this token *started* generating on. """

    tokenizer: "Tokenizer"
    """The tokenizer that generated this token."""

    consume_rest: bool = False
    """Sets whether this token was generated by CONSUME_REST."""

    def __str__(self) -> str:
        return f"{self.type.name}:{repr(self.value)}"


class AST:
    """An Abstract Syntax Tree.

    Represents the semantic structure of a script, without the specific syntax.
    """

    def __init__(self, root: 'ASTNode', script: str):
        self.root: ASTNode = root
        """The root `ASTNode` of this AST."""

        self.script: str = script
        """The script that generated this AST."""

    def prettify(self) -> str:
        """Returns a JSON-formatted string showing the full structure of this AST.

        .. WARNING::
            For debugging and demonstration purposes only.
        """
        return self.root.prettify()

    def __repr__(self) -> str:
        return f"ScrollAST({repr(self.root)}"


class ASTStateError(errors.ScrollError):
    """Generic tokenizer/parser error that includes an entire AST node.

    Raised by ASTNode functions on invalid state.

    Generally internal to the scrolls module. If one of these errors makes it out,
    something is probably wrong.
    """
    def __init__(self, node: 'ASTNode', message: str):
        self.node = node
        self.message = message

    def __str__(self) -> str:
        return self.message


class ASTNode:
    """
    A node within an `AST`.
    """
    __slots__ = (
        "children",
        "type",
        "_tok"
    )

    def __init__(self, type: ASTNodeType, token: t.Optional[Token]):
        self.children: t.MutableSequence['ASTNode'] = []
        """The child `ASTNode` objects of this node."""

        self.type = type
        """The `ASTNodeType` of this node."""

        self._tok: t.Optional[Token] = token

    def to_dict(self) -> t.Mapping[str, t.Any]:
        """
        Converts this object into a dict demonstrating its structure.

        Returns:
            A dictionary of the following form:

            ```json
            {
                "_type": "TYPENAME",
                "_tok": "TOKTYPE:'TOKVALUE'",
                "children": [...]
            }
            ```

            .. WARNING::
                This dictionary cannot be converted 1-1 back to a `ASTNode`. It is mainly meant for display
                purposes. See `ASTNode.prettify`.
        """

        mapping = {
            "_type": self.type.name,
            "_tok": str(self._tok),
            "children": [child.to_dict() for child in self.children]
        }

        return mapping

    def prettify(self) -> str:
        """
        Returns a JSON-formatted string showing the full structure of this `ASTNode`.

        .. WARNING::
            For debugging and demonstration purposes only.
        """

        s = json.dumps(self.to_dict(), sort_keys=True, indent=4)
        return s

    @property
    def tok(self) -> Token:
        """
        The token that generated this node. This should always be populated by `parse_scroll` under normal
        circumstances.

        Raises:
            ASTStateError: On get, if the token was never assigned.
        """

        if self._tok is None:
            raise ASTStateError(self, "cannot get token, is None")

        return self._tok

    @tok.setter
    def tok(self, token: Token) -> None:
        self._tok = token

    def has_token(self) -> bool:
        """
        Checks whether this node has a token assigned to it.
        """
        return self._tok is not None

    def wrap(self, node_type: ASTNodeType, as_child: bool = False) -> 'ASTNode':
        """
        Create a new node, and assign this node's token to the new node.

        .. WARNING::
            This is used internally by the parser during parsing and should generally not be called on finished ASTs.

        Args:
            node_type: The type of the new node.
            as_child: If `True`, add this node as a child of the new wrapper node.

        Returns:
            The newly created wrapper node.
        """
        new_node = ASTNode(
            node_type,
            self.tok
        )

        if as_child:
            new_node.children.append(self)

        return new_node

    def str_content(self) -> str:
        """
        Gets the string value of a `ASTNodeType.STRING` node.

        Raises:
            ASTStateError: If this node is not `ASTNodeType.STRING`.
        """
        if self.type != ASTNodeType.STRING:
            raise ASTStateError(self, "str_content requires STRING type node")

        assert self._tok is not None
        return self._tok.value

    def find_all(self, func: t.Callable[['ASTNode'], bool]) -> t.Sequence['ASTNode']:
        """
        Find all nodes in this tree for which `func` returns true.

        Args:
            func: A predicate which takes an `ASTNode` as input.

        Returns:
            A sequence of matching nodes.
        """

        found = []

        if func(self):
            found.append(self)

        for child in self.children:
            found.extend(child.find_all(func))

        return found

    def __str__(self) -> str:
        return repr(self)

    def __repr__(self) -> str:
        if self.type is ASTNodeType.STRING:
            return f"ScrollASTNode({self.type.name}, '{str(self._tok)}')"
        else:
            return f"ScrollASTNode({self.type.name}, {repr(self.children)})"


def str_ensure(s: str, ensure: str) -> str:
    if ensure not in s:
        return s + ensure
    else:
        return s


def str_remove(s: str, remove: str) -> str:
    return s.replace(remove, "")


def str_switch(s: str, switch: str, en: bool) -> str:
    """
    Utility function for enabling/disabling detection of certain characters.
    """

    if en:
        return str_ensure(s, switch)
    else:
        return str_remove(s, switch)


class StreamError(errors.ScrollError):
    pass


class StreamEofError(StreamError):
    pass


class CharStream(typing.Protocol):
    """
    A character stream. Used as a generic source of characters to tokenize in
    a `Tokenizer`.
    """

    def dump_state(self) -> None:
        """
        Prints internal state for debug purposes. Content of the message
        is entirely dependent on the `CharStream` implementation.
        """
        ...

    def current_line(self) -> int:
        """
        Get the line this stream is on. Increments when a newline (`\n`) character
        is encountered.
        """
        ...

    def current_pos(self) -> int:
        """
        Get the character within the current line this stream is on. Resets to 0
        when a new line is encountered.
        """
        ...

    def get_char(self) -> str:
        """
        Get the current character this stream is on.
        """
        ...

    def next_char(self) -> None:
        """
        Advance to the next character.
        """
        ...

    def at_eof(self) -> bool:
        """
        Returns `True` if `CharStream.next_char` just advanced from the last character.
        `CharStream.get_char` should not be called in this state. Since EOF is
        considered a token, at_eof can effectively be considered a virtual character.
        It is always the last one.
        """
        ...

    def after_eof(self) -> bool:
        """
        Returns `True` if `CharStream.next_char` just advanced from the EOF state.
        This is the true end of the stream, after all characters and EOF have been
        streamed.
        """
        ...

    def history(self) -> str:
        """
        Get the history of this stream. Guaranteed to return at least all
        text returned through `CharStream.get_char` and `CharStream.next_char`.

        May return text not yet streamed, depending on the implementation.
        """
        ...


class StringStream(CharStream):
    """
    A `CharStream` that streams an existing string. This is the default implementation
    used by `Tokenizer` instances if no stream is specified.
    """
    def __init__(self, string: str):
        self._current_line = 0
        self._current_pos = 0
        self._string = ""
        self._stringlen = 0
        self._more_chars = True

        self.feed(string)
        self._char = 0

    def dump_state(self) -> None:
        print(
            f"current line: {self._current_line}\n"
            f"current pos:  {self._current_pos}\n"
            f"more chars:   {self._more_chars}\n"
        )

    def current_line(self) -> int:
        return self._current_line

    def current_pos(self) -> int:
        return self._current_pos

    def at_eof(self) -> bool:
        off_end = self._char >= self._stringlen

        if off_end and self.after_eof():
            return False

        return off_end

    def get_char(self) -> str:
        if self.after_eof():
            raise StreamEofError("Cannot read from stream stream after EOF.")
        elif self.at_eof():
            raise StreamEofError("Cannot read from string stream at EOF.")

        return self._string[self._char]

    def next_char(self) -> None:
        # If we're at EOF, and try to get the next character, we've
        # exhausted everything.
        if self.at_eof():
            logger.debug("StringStream: set _more_chars False")
            self._more_chars = False
            return

        if self.after_eof():
            raise StreamEofError("Cannot read from string stream after EOF.")

        char = self.get_char()
        if char == "\n":
            self._current_line += 1
            self._current_pos = 0
        else:
            self._current_pos += 1

        self._char += 1

    def after_eof(self) -> bool:
        return not self._more_chars

    def history(self) -> str:
        return self._string

    def feed(self, string: str) -> None:
        """
        Add additional strings to the stream. May be done at any time.
        """
        trimmed_str = string.replace("\r", "")
        self._string += trimmed_str
        self._stringlen += len(trimmed_str)
        self._more_chars = True

        logging.debug(f"StringStream: Fed with \n{string}")


class REPLStream(StringStream):
    """
    A `CharStream` that streams input from stdin. If an EOF is ever encountered,
    instead of entering an EOF state, more input is requested from the user.

    This is done on a per-line basis.
    """
    def __init__(self) -> None:
        super().__init__("")
        self.prefix = ""
        self.set_statement()

    def set_statement(self) -> None:
        """
        Sets the input prefix of the REPL to ">>>". Must be
        called manually on successful execution of a statement.
        """
        self.prefix = ">>> "

    def set_continuation(self) -> None:
        """
        Sets the input prefix of the REPL to "...". Typically, this is
        done automatically.
        """
        self.prefix = "... "

    def consume_line(self) -> None:
        """
        Consume the next line of user input.
        """
        logger.debug("REPLStream: Requesting additional input.")

        # Get new line, ignoring empty lines.
        next_str = ""
        while not next_str.strip():
            next_str = input(self.prefix) + "\n"

        self.feed(next_str)
        self.set_continuation()

    def get_char(self) -> str:
        if self.at_eof():
            logger.debug("REPLStream: get_char: consuming new line")
            self.consume_line()

        return super().get_char()

    def next_char(self) -> None:
        if self.at_eof():
            logger.debug("REPLStream: next_char: consuming new line")
            self.consume_line()

        super().next_char()

    def next_line(self) -> None:
        """
        Skip forward until the next line.
        """
        logger.debug("REPLStream: Skipping current line.")

        # if at EOF, then we're already at a new line, so stream in a new one
        if self.at_eof():
            logger.debug("REPLStream: next_line: consuming new line")
            self.consume_line()
            return

        # Otherwise, skip forward in the current line until we're at a new one.
        # inefficient implementation for now, since this is only needed for
        # errors in the REPL.
        current_line = self.current_line()
        while self.current_line() == current_line:
            self.next_char()


class Tokenizer:
    """
    The tokenizer. This class is responsible for identifying meaningful pieces of scripts
    (such as string literals, block open and close, etc.), and tagging them.

    .. WARNING::
        If the tokenizer is supplied with a string, then this Tokenizer is **single use**.
        If you wish to stream input, implement a `CharStream`. See `StringStream.feed` and
        see if that works for you. See `REPLStream` for an example of streaming input
        from a user.

    Args:
        stream: The script to tokenize. May be a string or a `CharStream` instance.
        consume_rest_triggers: Triggers for CONSUME_REST.

    Related:
        `Token`, [Tokenizing](#tokenizing)
    """
    def __init__(
        self,
        stream: t.Union[str, CharStream],
        consume_rest_triggers: t.Mapping[str, int] = types.MappingProxyType({})
    ):
        if isinstance(stream, str):
            self.stream: CharStream = StringStream(stream.strip())
        else:
            self.stream = stream

        self.consume_rest_triggers = consume_rest_triggers
        self.consume_rest_state = TokenizeConsumeRestState.OFF
        self.consume_rest_count = 0
        self.previous_token_was_sep = True
        self.whitespace = "\t "

        # Map of single characters to token types
        self.charmap = {
            "\n": TokenType.COMMAND_SEP,
            ";": TokenType.COMMAND_SEP,
            OPEN_PAREN: TokenType.OPEN_PAREN,
            CLOSE_PAREN: TokenType.CLOSE_PAREN,
            BLOCK_OPEN: TokenType.OPEN_BLOCK,
            BLOCK_CLOSE: TokenType.CLOSE_BLOCK,
            EXPANSION_SIGIL: TokenType.EXPANSION_SIGIL,
            CONTROL_SIGIL: TokenType.CONTROL_SIGIL,
            MULTI_SIGIL: TokenType.MULTI_SIGIL
        }

        self.escape_sequences: t.MutableMapping[
            str,
            t.Union[str, t.Callable[[Tokenizer], str]]
        ] = {
            "n": "\n",
            "t": "\t",
            "r": "\r",
            ESCAPE_SIGIL: ESCAPE_SIGIL,
            QUOTE: QUOTE,
            "u": Tokenizer._unicode_escape
        }

        # Set up stop chars for unquoted string literals.
        self._string_literal_always_stop = self.whitespace
        self._string_literal_stop_quoted = QUOTE
        self._string_literal_stop_comment = COMMENT_SIGIL

        # Note: Add an exception for newlines. Even when we don't consider newlines to be command separators,
        # we would normally want newlines to separate string literals. So remove \n from this switch.
        self._string_literal_stop_single_char = str_remove(
            "".join(self.charmap.keys()),
            "\n"
        )

        # Override flag for behavior when single_char_token_enable is False.
        self.newlines_separate_strings = True

        self.string_literal_stop: str = self._string_literal_always_stop
        self.single_char_token_enable = True
        self.set_single_char_token_enable(True)

        # Set up stop chars for CONSUME_REST.
        self._consume_rest_stop_switch: str = "".join(COMMAND_SEP + [BLOCK_CLOSE, BLOCK_OPEN])
        self.consume_rest_stop: str = ""
        self.set_consume_rest_all(False)

        # Set up stop chars for quoted literals.
        self.quoted_literal_stop: str = QUOTE  # For now, quoted literals ONLY stop on another quote.
        self.quoted_literal_enable = True
        self.set_quoted_literals_enable(True)

        # Set up stop chars for comments. (Note: No need for specific comment stop char here, it's hardcoded to
        # be \n at the moment.)
        self.comments_enable = True
        self.set_comments_enable(True)

    def _unicode_escape(self) -> str:
        code_point = ""  # Initialization not needed, just satisfies some linters.
        try:
            code_point = self.next_n_chars(4)
        except errors.TokenizeEofError:
            self.error(
                errors.TokenizeEofError,
                "Ran off end of script trying to parse unicode escape."
            )

        if QUOTE in code_point:
            self.error(
                errors.TokenizeError,
                f"Encountered {QUOTE} while consuming unicode escape.",
                pos=self.stream.current_pos() - 4
            )

        char = ""
        try:
            char = chr(int(code_point, 16))
        except ValueError:
            self.error(
                errors.TokenizeError,
                f"Bad hex number {code_point}.",
                pos=self.stream.current_pos() - 4
            )

        return char

    def set_consume_rest_all(self, consume_all: bool) -> None:
        """
        Set whether CONSUME_REST consumes until EOF. Defaults to `False`.

        If `False`, CONSUME_REST will stop on block open/close, and command separators.

        If `True`, CONSUME_REST will not stop until EOF.
        """
        self.consume_rest_stop = str_switch(
            self.consume_rest_stop,
            self._consume_rest_stop_switch,
            not consume_all
        )

    def set_single_char_token_enable(self, en: bool) -> None:
        """
        Set whether single character tokens should be parsed. This includes ALL token types except for
        `TokenType.STRING_LITERAL` and `TokenType.COMMENT`. Defaults to `True`.

        If `False`, then all special characters that would otherwise be their own token will be rolled
        into string literals.
        """
        if self.newlines_separate_strings and not self.single_char_token_enable and en:
            # If we're re-enabling single char tokens and the newline separator behavior is still on,
            # we need to undo that first.
            self.set_newlines_separate_strings(False)

        self.single_char_token_enable = en

        self.string_literal_stop = str_switch(
            self.string_literal_stop,
            self._string_literal_stop_single_char,
            en
        )

        if not en:
            self.set_newlines_separate_strings(True)
        else:
            # If single char tokens are enabled, newlines must stop string literals for this to work properly.
            self.string_literal_stop = str_ensure(self.string_literal_stop, "\n")

    def set_quoted_literals_enable(self, en: bool) -> None:
        """
        Set whether quoted string literals are enabled. If disabled, quotes will be rolled into normal string token
        parsing.

        For instance, if quoted literals are disabled, `"Hello World"` would be interpreted as `"Hello`, `World"`.
        """
        self.quoted_literal_enable = en
        self.string_literal_stop = str_switch(
            self.string_literal_stop,
            self._string_literal_stop_quoted,
            en
        )

    def set_comments_enable(self, en: bool) -> None:
        """
        Set whether comments are enabled. If disabled, the comment character will be ignored, and anything that
        would be a comment will be treated as ordinary code.
        """
        self.comments_enable = en
        self.string_literal_stop = str_switch(
            self.string_literal_stop,
            self._string_literal_stop_comment,
            en
        )

    def set_newlines_separate_strings(self, en: bool) -> None:
        """
        Set whether newlines separate string literals. This can only be modified if
        `Tokenizer.set_single_char_token_enable` has been set to `False`, and will raise a `ScrollsError` otherwise.

        By default, when `Tokenizer.set_single_char_token_enable` is set to `False`, newlines will instead be
        considered whitespace, and will separate strings without producing `TokenType.COMMAND_SEP` tokens.

        To override this behavior, this function may be set to `False`. In this case, newlines will be rolled into
        string literals, and ONLY spaces and tabs will separate string literals.
        """
        if self.single_char_token_enable:
            raise errors.ScrollError("Cannot use set_newlines_separate_strings when single char tokens are enabled.")

        self.newlines_separate_strings = en
        self.string_literal_stop = str_switch(
            self.string_literal_stop,
            "\n",
            en
        )
        self.whitespace = str_switch(
            self.whitespace,
            "\n",
            en
        )

    def error(
        self,
        err_type: t.Type[errors.PositionalError],
        message: str,
        line: t.Optional[int] = None,
        pos: t.Optional[int] = None
    ) -> t.NoReturn:
        if line is not None:
            _line = line
        else:
            _line = self.stream.current_line()

        if pos is not None:
            _pos = pos
        else:
            _pos = self.stream.current_pos()

        raise err_type(
            _line,
            _pos,
            self.stream.history(),
            message
        )

    def forbid_eof(self, msg: str = "", *args: t.Any, **kwargs: t.Any) -> None:
        if not msg:
            msg = "Unexpected EOF while parsing script."

        if self.stream.at_eof() or self.stream.after_eof():
            self.error(errors.TokenizeEofError, msg.format(*args, **kwargs))

    def next_n_chars(self, n: int) -> str:
        """
        Unconditionally consume N characters and return them.
        """
        chars: t.MutableSequence[str] = []
        for _ in range(n):
            self.forbid_eof(
                "Ran into EOF while consuming characters. Got {}, wanted {}.",
                len(chars), n
            )

            chars.append(self.stream.get_char())
            self.stream.next_char()

        return "".join(chars)

    # Get a single char token.
    def accept_single_char(self) -> t.Optional[Token]:
        if not self.single_char_token_enable:
            return None

        char = self.stream.get_char()

        if char in self.charmap:
            tok = Token(
                self.charmap[char],
                char,
                self.stream.current_line(),
                self.stream.current_pos(),
                self
            )
            self.stream.next_char()
            return tok

        return None

    def accept_eof(self) -> t.Optional[Token]:
        if self.stream.at_eof():
            # Once an EOF is generated, there are no more tokens.
            # Any attempts after this to generate a token will
            # result in an exception.
            self.stream.next_char()  # Put stream into after eof state

            return Token(
                TokenType.EOF,
                EOF,
                self.stream.current_line(),
                self.stream.current_pos(),
                self
            )
        else:
            return None

    def accept_whitespace(self) -> t.Optional[Token]:
        char = self.stream.get_char()
        if char in self.whitespace:
            self.stream.next_char()
            return Token(
                TokenType.WHITESPACE,
                char,
                self.stream.current_line(),
                self.stream.current_pos(),
                self
            )

        return None

    def try_consume_escape(self) -> t.Optional[str]:
        if self.stream.get_char() != ESCAPE_SIGIL:
            return None

        self.stream.next_char()
        self.forbid_eof()

        escape_char = self.stream.get_char()
        if escape_char not in self.escape_sequences:
            self.error(errors.TokenizeError, f"Invalid escape '{escape_char}'")

        self.stream.next_char()
        self.forbid_eof()

        replacement = self.escape_sequences[escape_char]
        if isinstance(replacement, str):
            return replacement
        elif callable(replacement):
            return replacement(self)
        else:
            raise TypeError(f"Bad type for escape sequence {escape_char}, "
                            "must be 'str' or '(Tokenizer) -> str'")

    def accept_string_literal(
        self,
        stop_chars: t.Sequence[str] = (),
        error_on_eof: bool = False,
        allow_escapes: bool = False
    ) -> t.Optional[Token]:
        self.forbid_eof("String literal should not start on EOF")

        char = self.stream.get_char()
        pos = self.stream.current_pos()
        line = self.stream.current_line()
        chars = []

        while char not in stop_chars:
            if allow_escapes:
                escape = self.try_consume_escape()
                if escape is not None:
                    chars.append(escape)
                    char = self.stream.get_char()
                    continue

            chars.append(char)
            self.stream.next_char()
            if self.stream.at_eof():
                if error_on_eof:
                    self.error(
                        errors.TokenizeEofError,
                        "Unexpected EOF while parsing string literal."
                    )
                else:
                    break

            char = self.stream.get_char()

        return Token(
            TokenType.STRING_LITERAL,
            "".join(chars),
            line,
            pos,
            self
        )

    def accept_comment(self) -> t.Optional[Token]:
        if not self.comments_enable:
            return None

        char = self.stream.get_char()
        pos = self.stream.current_pos()
        line = self.stream.current_line()
        chars = []

        if char != COMMENT_SIGIL:
            return None

        self.stream.next_char()
        while char != "\n":
            chars.append(char)
            self.stream.next_char()

            if self.stream.at_eof():
                break

            char = self.stream.get_char()

        return Token(
            TokenType.COMMENT,
            "".join(chars),
            line,
            pos,
            self
        )

    # Accepts a normal string literal. No CONSUME_REST, not quoted.
    def accept_string_literal_normal(self) -> t.Optional[Token]:
        return self.accept_string_literal(
            stop_chars=self.string_literal_stop,
            error_on_eof=False  # Just stop on EOF, no errors.
        )

    # Accept a CONSUME_REST literal.
    def accept_string_literal_consume_rest(self) -> t.Optional[Token]:
        return self.accept_string_literal(
            stop_chars=self.consume_rest_stop,
            error_on_eof=False  # Stop on EOF. No errors.
        )

    # Accept a quoted string literal.
    def accept_string_literal_quoted(self) -> t.Optional[Token]:
        if not self.quoted_literal_enable:
            return None

        if self.stream.get_char() != QUOTE:
            return None
        else:
            self.stream.next_char()

        literal = self.accept_string_literal(
            stop_chars=self.quoted_literal_stop,
            error_on_eof=True,  # Quoted literals must be closed.
            allow_escapes=True  # Escapes only allowed in quoted literals.
        )

        if literal is None:
            self.error(
                errors.TokenizeError,
                "internal: Got None from accept_string_literal, shouldn't have."
            )

        if self.stream.get_char() != QUOTE:
            self.error(
                errors.TokenizeError,
                "internal: Missing end quote, should have resulted in EOF error."
            )
        else:
            self.stream.next_char()

        return literal

    @staticmethod
    def accept_any_of(*f: t.Callable[[], t.Optional[Token]]) -> t.Optional[Token]:
        for fun in f:
            tok = fun()
            if tok is not None:
                return tok

        return None

    def handle_consume_rest_off(self, tok: Token) -> None:
        if tok.type in (TokenType.COMMAND_SEP, TokenType.CLOSE_BLOCK, TokenType.CLOSE_PAREN):
            self.previous_token_was_sep = True
            return

        # Test to see if we should enter CONSUME_REST state.
        # Only trigger CONSUME_REST if the previous token was a command separator.
        should_enter_consume_rest = (
                self.previous_token_was_sep and
                tok.type == TokenType.STRING_LITERAL and
                tok.value in self.consume_rest_triggers
        )
        self.previous_token_was_sep = False
        if should_enter_consume_rest:
            count = self.consume_rest_triggers[tok.value]

            if count == 0:
                self.consume_rest_state = TokenizeConsumeRestState.CONSUME
            else:
                self.consume_rest_state = TokenizeConsumeRestState.COUNTING
                self.consume_rest_count = count

    def handle_consume_rest_counting(self, tok: Token) -> None:
        self.previous_token_was_sep = False

        # Only count down on string literals.
        if tok.type == TokenType.STRING_LITERAL:
            self.consume_rest_count -= 1

            # Once countdown is over, CONSUME_REST on next token.
            if self.consume_rest_count == 0:
                self.consume_rest_state = TokenizeConsumeRestState.CONSUME

        # If we get any other token type, then cancel CONSUME_REST
        else:
            self.consume_rest_state = TokenizeConsumeRestState.OFF
            self.consume_rest_count = 0

    def handle_consume_rest_consume(self, tok: Token) -> None:
        # This function runs AFTER a CONSUME_REST consumption. So, just set consume_rest back to OFF.
        self.consume_rest_state = TokenizeConsumeRestState.OFF
        self.consume_rest_count = 0

    # TODO
    # Consume rest state handler. All this code is pretty ugly, and does not account
    # for more advanced usage.
    def handle_consume_rest(self, tok: Token) -> None:
        f_map: t.Mapping[TokenizeConsumeRestState, t.Callable[[Token], None]] = {
            TokenizeConsumeRestState.OFF: self.handle_consume_rest_off,
            TokenizeConsumeRestState.COUNTING: self.handle_consume_rest_counting,
            TokenizeConsumeRestState.CONSUME: self.handle_consume_rest_consume
        }

        f_map[self.consume_rest_state](tok)

    def next_token(self) -> Token:
        """
        Extract the next token. If the tokenizing is finished, this will return a `Token` of type `TokenType.EOF`

        Raises:
            scrolls.errors.TokenizeEofError: If EOF was hit unexpectedly.
            scrolls.errors.TokenizeError: If a generic issue happened while tokenizing.
        """
        if self.consume_rest_state == TokenizeConsumeRestState.CONSUME:
            while True:
                tok = self.accept_any_of(
                    self.accept_whitespace
                )

                if tok is None:
                    break

                if tok.type == TokenType.WHITESPACE:
                    continue

            tok = self.accept_string_literal_consume_rest()
            if tok is None:
                self.error(
                    errors.TokenizeError,
                    "Got bad string literal during consume_rest"
                )
            logger.debug(f"tokenize: Got token {tok.type.name}:{repr(tok.value)}")
            tok.consume_rest = True  # Signal we got this token using CONSUME_REST

            self.handle_consume_rest(tok)
            return tok
        else:
            while True:
                if self.stream.after_eof():
                    self.error(
                        errors.TokenizeEofError,
                        "No more tokens."
                    )

                tok = None

                try:
                    tok = self.accept_any_of(
                        self.accept_whitespace,
                        self.accept_comment,
                        self.accept_single_char,
                        self.accept_string_literal_quoted,
                        self.accept_string_literal_normal
                    )
                except StreamEofError:
                    # !!! HACK
                    # I really, really need to rethink how EOF is handled
                    # throughout this entire module. It's broken.
                    pass

                if tok is None:
                    # If tok is None, then all tokenizing functions got
                    # rejected. So, try to accept and return EOF.

                    eof_tok = self.accept_eof()

                    if eof_tok is None:
                        self.error(
                            errors.TokenizeError,
                            "Unexpectedly rejected all tokenizing functions."
                        )
                    else:
                        return eof_tok

                # Loop until we get a non-whitespace, non-comment token.
                if tok.type not in [TokenType.WHITESPACE, TokenType.COMMENT]:
                    logger.debug(f"tokenize: Got token {tok.type.name}:{repr(tok.value)}")
                    self.handle_consume_rest(tok)
                    return tok

    def get_all_tokens(self) -> t.Sequence[Token]:
        """
        Extracts all tokens at once, until the end of the script. A sequence of tokens obtained this way
        will always end with a token of type `TokenType.EOF`.

        Raises:
            scrolls.errors.TokenizeEofError: If EOF was hit unexpectedly.
            scrolls.errors.TokenizeError: If a generic issue happened while tokenizing.
        """
        tokens: t.MutableSequence[Token] = []

        while True:
            tok = self.next_token()
            tokens.append(tok)
            if tok.type == TokenType.EOF:
                return tokens


class ParseContext:
    def __init__(self, tokenizer: Tokenizer):
        self.tokenizer = tokenizer
        self.token: Token = Token(TokenType.WHITESPACE, "", 0, 0, tokenizer)

        self.next_token()

    def current_token(self) -> Token:
        return self.token

    def next_token(self) -> None:
        if self.tokenizer.stream.after_eof():
            logger.debug("End of tokens.")
            parse_error(
                self,
                errors.ParseEofError,
                "Unexpected end of script."
            )
        else:
            prev_token = self.token
            self.token = self.tokenizer.next_token()

            logger.debug(f"Advance token: {str(prev_token)}->{str(self.token)}")


def parse_error(
    ctx: ParseContext,
    error: t.Type[errors.ParseError],
    message: str,
    fatal: bool = False
) -> t.NoReturn:
    e = error(
        ctx.token.line,
        ctx.token.position,
        ctx.tokenizer.stream.history(),
        message
    )

    e.fatal = fatal

    if not fatal:
        logger.debug("error")
    else:
        logger.debug("fatal error")

    raise e


def parse_get(
    ctx: ParseContext,
    type: TokenType
) -> t.Optional[Token]:
    token = ctx.current_token()

    logger.debug(f"parse_get: want {type.name}")

    if token.type == type:
        ctx.next_token()
        logger.debug(f"parse_get: accepted {str(token)}")
        return token
    else:
        logger.debug(f"parse_get: rejected {str(token)}")
        return None


def parse_expect(
    ctx: ParseContext,
    type: TokenType,
    fatal_on_error: bool = False
) -> Token:
    tok = parse_get(ctx, type)

    if tok is None:
        parse_error(
            ctx,
            errors.ParseExpectError,
            f"expected {type.name} here, but got {ctx.token.type.name}({ctx.token.value})",
            fatal=fatal_on_error
        )

    else:
        return tok


def parse_strtok(
    ctx: ParseContext
) -> ASTNode:
    node = ASTNode(
        ASTNodeType.STRING,
        parse_expect(ctx, TokenType.STRING_LITERAL)
    )

    return node


def parse_greedy(
    parser: ParserT
) -> t.Callable[[ParseContext], t.Sequence[ASTNode]]:
    def _(ctx: ParseContext) -> t.Sequence[ASTNode]:
        nodes: t.MutableSequence[ASTNode] = []

        while True:
            try:
                nodes.append(parser(ctx))
                logger.debug("parse_greedy: append success")
            except errors.ParseError as e:
                if e.fatal:
                    raise

                logger.debug("parse_greedy: append fail, return")
                return nodes

    return _


def parse_choice(
    *parsers: ParserT
) -> ParserT:
    def _(ctx: ParseContext) -> ASTNode:
        last_exc: t.Optional[errors.ParseError] = None

        for parser in parsers:
            try:
                node = parser(ctx)
                return node
            except errors.ParseError as e:
                last_exc = e

                if e.fatal:
                    break

        if last_exc is None:
            parse_error(
                ctx,
                errors.ParseError,
                "internal: no parsers provided for parse_choice"
            )
        else:
            raise last_exc

    return _


def expect(
    t_type: TokenType,
    fatal_on_error: bool = False
) -> ParserT:
    def _(ctx: ParseContext) -> ASTNode:
        node = ASTNode(
            ASTNodeType.NONE,
            parse_expect(ctx, t_type, fatal_on_error)
        )

        return node

    return _


def parse_try(
    parser: ParserT
) -> t.Callable[[ParseContext], bool]:
    def _(ctx: ParseContext) -> bool:
        try:
            parser(ctx)
            return True
        except errors.ParseError:
            return False

    return _


def expect_eof(ctx: ParseContext) -> ASTNode:
    try:
        parse_expect(ctx, TokenType.EOF)
    except errors.ParseEofError:
        pass

    return ASTNode(
        ASTNodeType.EOF,
        ctx.token
    )


expect_command_separator = expect(TokenType.COMMAND_SEP)


def parse_expansion_var(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_expansion_var")
    var_name_node = parse_eventual_string(ctx).wrap(
        ASTNodeType.EXPANSION_VAR, as_child=True
    )

    return var_name_node


def parse_expansion_call_args(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_expansion_call_args")

    args = parse_greedy(parse_eventual_string)(ctx)
    first_tok: t.Optional[Token] = None

    if args:
        first_tok = args[0].tok

    args_node = ASTNode(
        ASTNodeType.EXPANSION_ARGUMENTS,
        first_tok
    )
    args_node.children.extend(args)

    return args_node


def parse_expansion_call(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_expansion_call")
    call_node = expect(TokenType.OPEN_PAREN)(ctx).wrap(
        ASTNodeType.EXPANSION_CALL
    )

    call_node.children.append(parse_eventual_string(ctx))  # Expansion name
    call_node.children.append(parse_expansion_call_args(ctx))

    expect(TokenType.CLOSE_PAREN, fatal_on_error=True)(ctx)

    return call_node


def parse_expansion(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_expansion")

    expansion_node = expect(TokenType.EXPANSION_SIGIL)(ctx).wrap(
        ASTNodeType.EXPANSION
    )

    multi_tok = parse_get(ctx, TokenType.MULTI_SIGIL)
    if multi_tok is None:
        expansion_node.children.append(
            ASTNode(ASTNodeType.EXPANSION_SINGLE, None)
        )
    else:
        expansion_node.children.append(
            ASTNode(ASTNodeType.EXPANSION_MULTI, multi_tok)
        )

    expansion_node.children.append(
        parse_choice(parse_expansion_call, parse_expansion_var)(ctx)
    )

    return expansion_node


parse_eventual_string = parse_choice(
    parse_expansion,
    parse_strtok
)


def parse_command_args(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_command_args")
    args_node = ASTNode(ASTNodeType.COMMAND_ARGUMENTS, None)
    args_node.children.extend(parse_greedy(parse_eventual_string)(ctx))

    if args_node.children:
        args_node.tok = args_node.children[0].tok

    return args_node


def parse_command(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_command")

    command_node = parse_eventual_string(ctx).wrap(
        ASTNodeType.COMMAND_CALL,
        as_child=True
    )
    command_node.children.append(parse_command_args(ctx))

    return command_node


def parse_control_args(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_control_args")
    args_node = expect(TokenType.OPEN_PAREN)(ctx).wrap(
        ASTNodeType.CONTROL_ARGUMENTS
    )
    args_node.children.extend(parse_greedy(parse_eventual_string)(ctx))
    expect(
        TokenType.CLOSE_PAREN,
        fatal_on_error=True
    )(ctx)

    return args_node


def parse_control(ctx: ParseContext) -> ASTNode:
    logger.debug("parse_control")

    control_node = expect(TokenType.CONTROL_SIGIL)(ctx).wrap(
        ASTNodeType.CONTROL_CALL
    )
    control_node.children.append(parse_strtok(ctx))

    try:
        # Try to parse statement first, and count no () as no arguments
        statement_node = _parse_statement(ctx)
        args_node = ASTNode(ASTNodeType.CONTROL_ARGUMENTS, statement_node.tok)

        control_node.children.append(args_node)
        control_node.children.append(statement_node)
    except errors.ParseError:
        control_node.children.append(parse_control_args(ctx))
        control_node.children.append(_parse_statement(ctx))

    return control_node


def parse_block_body(ctx: ParseContext, top_level: bool = False) -> t.Sequence[ASTNode]:
    logger.debug("parse_block_body")

    nodes: t.MutableSequence[ASTNode] = []

    while True:
        if ctx.token.type == TokenType.CLOSE_BLOCK:
            if top_level:
                parse_error(
                    ctx,
                    errors.ParseError,
                    "Unexpected block close.",
                    fatal=True
                )
            else:
                return nodes

        if ctx.token.type == TokenType.EOF:
            if top_level:
                return nodes
            else:
                parse_error(
                    ctx,
                    errors.ParseEofError,
                    "Unexpected end of script while parsing block.",
                    fatal=True
                )

        # If we hit a command separator, just consume it and continue.
        if parse_try(expect_command_separator)(ctx):
            continue

        # Actually try to parse the next statement. If that fails, it means we found some non-statement
        # structure inside a block, which is not legal. Error out with something more descriptive.
        try:
            node = _parse_statement(ctx)
        except errors.ParseError:
            parse_error(
                ctx,
                errors.ParseError,
                "Expected statement or block here.",
                fatal=True
            )
            raise  # Not necessary, but satisfies linters.

        nodes.append(node)


def parse_block(ctx: ParseContext) -> ASTNode:
    node = expect(TokenType.OPEN_BLOCK)(ctx).wrap(
        ASTNodeType.BLOCK
    )
    node.children.extend(parse_block_body(ctx))
    expect(
        TokenType.CLOSE_BLOCK,
        fatal_on_error=True
    )(ctx)

    return node


_parse_statement = parse_choice(
    parse_block,
    parse_control,
    parse_command
)


def parse_root(tokenizer: Tokenizer) -> ASTNode:
    ctx = ParseContext(tokenizer)
    root_node = ASTNode(ASTNodeType.ROOT, None)
    root_node.children.extend(parse_block_body(ctx, top_level=True))

    return root_node


def parse_scroll(tokenizer: Tokenizer) -> AST:
    """
    Parse a script (wrapped in a `Tokenizer`) and convert it to an `AST`. See [Using The Parser](#using-the-parser).
    """
    return AST(parse_root(tokenizer), tokenizer.stream.history())


def parse_statement(tokenizer: Tokenizer) -> ASTNode:
    """
    Parse a single statement from a `Tokenizer`.
    """
    ctx = ParseContext(tokenizer)
    return _parse_statement(ctx)
