"""
Errors not dependent on any specific Scrolls types.

Typically, you won't need to instantiate any of these yourself. The base exception
for _all_ Scrolls errors is `ScrollError`. Any error that occurs while validating
script syntax or interpreting scripts will inherit from `PositionalError`.
"""

import functools
import math
import typing as t

__all__ = (
    "format_positional_error",
    "ScrollError",
    "PositionalError",
    "ParseError",
    "ParseEofError",
    "ParseExpectError",
    "TokenizeError",
    "TokenizeEofError"
)


@functools.lru_cache(128)
def format_positional_error(
    line: int,
    pos: int,
    string: str,
    message: str,
    prior_lines: int = 3
) -> str:
    """Format a positional error generated by Scrolls.

    Args:
        line: The line the error was generated on.
        pos: The character the error was generated on.
        string: The script that generated the error.
        message: The message associated with the error.

        prior_lines: The number of lines that should be printed before the line
            the error occurred on. The line containing the error will always
            be printed.

    Returns:
        The formatted error message.

        For example:
        ```text
        ...
        1 print "World"
        2 print "Foo"
        3 print "Bar"
        4 print "bad string
                           ^
        line 4 - Unexpected EOF while parsing string literal.
        ```

        If there are more than `prior_lines` lines before the error, `...` will be
        prepended to the output.
    """
    zfill = max(1, int(math.log10(len(string))))
    lines = [f"{n:0{zfill}} {l}" for n, l in enumerate(string.splitlines())]

    printed_lines = lines[max(0, line - prior_lines): line + 1]

    output_lines = [
        *(["..."] if line - prior_lines >= 1 else []),
        *printed_lines,
        " "*(pos + 1 + zfill) + "^",
        f"line {line}: {message}"
    ]

    return "\n".join(output_lines)


class ScrollError(Exception):
    """Base class for all Scrolls-related errors."""
    pass


class PositionalError(ScrollError):
    """Generic error that happened somewhere in a script.

    Any error in tokenizing, parsing, or interpreting should inherit from this.
    Typically you'll never need to instantiate one of these yourself, just catch it
    and call `str` on it. This will return a formatted error message pointing to
    where the error happened. See `format_positional_error` for more details.

    Example usage:

    ```
    try:
        some_scrolls_function(...)
    except PositionalError as e:
        print("error:")
        print(str(e))
    ```

    Note that this will apply to any error that inherits from `PositionalError` as well.
    If you want to do your own formatting, you can use the instance variables below to
    generate your own messages.
    """
    def __init__(
        self,
        line: int,
        pos: int,
        string: str,
        message: str
    ):
        self.line = line
        """The line the error occurred on."""

        self.pos = pos
        """The character along `line` the error occurred at."""

        self.string = string
        """The string that triggered the error. In all normal cases, this is a script."""

        self.message = message
        """The message associated with this error."""

    def __str__(self) -> str:
        """
        Return a formatted error string pointing out in the script where this error
        happened.
        """
        return format_positional_error(
            self.line,
            self.pos,
            self.string,
            self.message
        )


class TokenizeError(PositionalError):
    """Generic error raised while lexing/tokenizing a script."""
    pass


class TokenizeEofError(TokenizeError):
    """Raised when the lexer/tokenizer hits an unexpected EOF (end of script)."""
    pass


class ParseError(PositionalError):
    """Generic error raised during the parsing stage."""
    def __init__(
        self,
        line: int,
        pos: int,
        string: str,
        message: str
    ):
        super().__init__(
            line,
            pos,
            string,
            message
        )

        # IMPLEMENTATION DETAIL
        # Sets whether this parse error is fatal or not. Defaults to `False`.
        # If `True`, a `ParseError` will cause all parsing to stop immediately and
        # raise the error. If `fatal`  is `False`, a parse function may try alternative
        # parsing. Internally, `fatal = False` is used by `parse_choice` to determine
        # which parsing function to choose. See `scrolls.ast` for more details.
        self.fatal = False


class ParseEofError(ParseError):
    """Raised when an EOF is encountered too early while parsing a script."""
    pass


class ParseExpectError(ParseError):
    """Raised when an unexpected token is encountered during parsing."""
    pass
