MOCK_PRES_REQ = {
    "nonce": "182453895158932070575246",
    "name": "Step 3 Send your Health Information",
    "version": "1.0",
    "requested_attributes": {
        "biomarker_attrs_0": {
            "names": [
                "name",
                "concentration",
                "unit",
                "range",
                "collected_on",
                "biomarker_id",
                "researcher_share",
            ],
            "restrictions": [
                {"schema_name": "MYCO Biomarker", "attr::name::value": "Iron"}
            ],
        },
        "consent_attrs": {
            "restrictions": [
                {
                    "schema_name": "MYCO Consent Enablement",
                    "schema_version": "0.1.0",
                    "attr::jti_unique_identifier::value": "205b1ea0-7848-48d4-b52b-339122d84f62",
                }
            ],
            "name": "jti_unique_identifier",
        },
    },
    "requested_predicates": {},
}
CONTEXT = [
    "https://www.w3.org/2018/credentials/v1",
    "https://w3id.org/security/data-integrity/v2",
    {"@vocab": "https://www.w3.org/ns/credentials/issuer-dependent#"},
]
MOCK_W3CPRES = {
    "@context": CONTEXT,
    "type": ["VerifiablePresentation"],
    "verifiableCredential": [
        {
            "@context": [
                "https://www.w3.org/2018/credentials/v1",
                "https://w3id.org/security/data-integrity/v2",
                {"@vocab": "https://www.w3.org/ns/credentials/issuer-dependent#"},
            ],
            "type": ["VerifiableCredential"],
            "issuer": "PuC7nqM4PnFXHmwQXVbuZ7",
            "credentialSubject": {
                "degree": "Maths",
                "name": "Alice Smith",
                "birthdate_dateint": True,
            },
            "proof": {
                "cryptosuite": "anoncreds-2023",
                "type": "DataIntegrityProof",
                "proofPurpose": "assertionMethod",
                "verificationMethod": "PuC7nqM4PnFXHmwQXVbuZ7:3:CL:1255411:faber.agent.degree_schema",
                "proofValue": "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",
            },
            "issuanceDate": "2024-07-11T14:33:48.869474016Z",
        }
    ],
    "proof": {
        "cryptosuite": "anoncreds-2023",
        "type": "DataIntegrityProof",
        "proofPurpose": "authentication",
        "verificationMethod": "PuC7nqM4PnFXHmwQXVbuZ7:3:CL:1255411:faber.agent.degree_schema",
        "proofValue": "ukgOBqmFnZ3JlZ2F0ZWSCpmNfaGFzaNwAIMyBzNUrzIpQWMzSzOjMhiPMhjgYzL5ZzLUZL8zzzJI0zJB1M8zHzJnMnVBteQnM6qZjX2xpc3SW3AEBATQFzNFMWMylTMy_zMvM6EbMpDDMkXnMxz7M2QVFzPXM9lzMosz_M0PMrQhFEAVIzME2f1PMmsyJV1TM8XfM-kVPFsyWzJRLzMPM9HTM0nLMm1UBzK0AzIjMqxI-MsyLcQVzHWx-I8ynzN_MsszDLWt0zNnMgcyZzKRzZ8y_OQdsZXjMqy_Mh1LMkMzvzLnM7ic3XxHMvczLVw1qzI7M7wExNcz5zK9_zJ46zNEKzIbM6szcQAfMuktUzOHMlXUizIHMvVfMuMzLBsz3elAhzI_MqcyTLEfM9MyWzOnMksylKxo9byjM6sySbSDMuSoAP3_MvCLMrsydzM7MhcyCRyDMwA_M3X_M18y1zPbM-8zYTMztcS3MysyUYmRYUWbM5czzEmHM_MytEcyBCzpRzOHM2szySwnMnczMzNbMm38PzKjM3syuzIQnc8y9zKZEzJzMhcyjGiLM5MyxUsy_cMyrzPkxNczYPMyuzNMBNcyxYyvM8FIOZzZn3AEAzNYTzIDMncyQzOHMjcymGGDM_n7MlMzZPxPMow7M6cyRVyAybsyQGMzXGXpaPAXM9MyQLMz4zOPMt8zxzKnMnTlAHsylzOhozIjMtsz5V8yqZRDM9MyJWmxzzIQmzK1xTGhXdmdWzJbMmsyJzJ_MkMyKS3rM5My5EHFnVszVzKdIzNpIfMyizMPM8MyWUMzEBMySdGsjAXDMgsyKCszWUwPM0EXM5czyHkhdFMyeQRDMgsyIUsyUzMXM_czszNfM48ynzLXM93RNXMyozIYPzINaGD1XzKAJZ2g1zO_MwMzMWTvMiRnM-HXMg0oJGsy9zKLMuWwweMzvfj15zObM7F7MpMzwCszdzKJnb8yeMkjMr8zlzNhDaX5wOsyqZArMxszuzJcrzMjMtsybISLM4szVzMNxT39yKmNZHgtNzNXM3Mz9SszURG8PzNFOCcz0CEnMiV_Mwsz6S8yPQHAvacyPzLktQszfLCHMnszVzPfM-8ynW8zVzNZ43AEAWDkyzJ_Mqy0qzPvM7My8zJbM-HzMuw3MyMzYMzrM-ThzJ2MwzO9dN8zjzIXMgRF_EDPM-Mz_zI8qzIobzJfMq8zgaH02zIHMn8yRNkZKVczqPMz5f8yMzKbMusz9G8yNzI_MwszfzJLM-8yXzNDMxhRczLw7BMzDzK_M58zhQsy1zK0azI3Mtm_MwcygzLkZzNXMgczpMVfMjszCzObMxgUVE8yYzM3M6syEzOnMm2l_dF3M2HHM4FYDzKVjzKPMqczCzNIyzI8xzNHM9zl0zIjMoMyERMzgzK9qzM3MlMyUzK7M_3DM3MzKJMyQKhLMzXzM2HvMsszoQ8zszKfMvS8fzMdfzJbMsSNqzO1FHszKzLgVUcyYNszczL8hzL1MzIdoMEHMlszbzMTMviDMoMztzLlszM4bbXPMocy9zOAvQ2N7zOUoWszwQkXMqXnMwsy8zMNEzPUtzJzMqGLMyszcJsyWzJ7M234bYMzwzNLMxTXMuz7Mr8y8zIbM-w3MsMztzIrM48yhzMLMncz8zORyGVzcAQEBzP7MuyBOzMjM38yVzPdwPszqWCE0zLnMlhLM3MzlzPzM2syyzOQ7zM5Yc8zbzJXM68yozLYHzJHM5jXMtsybzMPMywBszK5-zIh0zP8zzPvM3UN_zPvM0MyqzMNTzLYoMczozPPMt8zAYFIrzIBkFMzYO8yODS_M88yvDAzM0My4zIvMpsyAZx3MkirMrjXMhyEqzPYkYDzMrwU2WczvzPDMyn3MkWzMqlnMk8yEUMy6zJTMmsz8c8yhzPzM9mRGdcy8QMzeKG9ozKXMycyNzPHMkcz-TszuKsyxHTUlOcyDzLdjzIdjzKcBCMzRWwRczMdIfz1JPszCOMzLzKgmzN7M5VwAzL_M9cyxzPjMuxR_zMnM2MyYzLbMjFseVMzERQPM3TFjzIhZEhPM9MyPC3vMw8z6zOnM6cz_XzgQzNsizLJRKxxazOTMvjHMkFhczO0cKEJwOszZHsy9zMdSzIoSzIwvSGzM-yLM8yA8MyjMlczZZ1HMocywMhYGzK9GzOvcAQECGRBXzKDM5czoP8yKbMyozMVyFlDM_MzmbD_MvXNPzPLMmBXMlMywaknMynnM9czKSlwjzMwgQczezIEmcMyAzM_MhyPMqsyIzM3M4MyFJcy9zPMNAczHPSHMx8ywzOfM3zlwJszODsyuzJBxchrMvDjMnMzKH35hB1gBzLfMjwMLX8yMccyDAivM6S_Mm8z1SMzdzM7M4czdzN_MqszIzLXMpljMlWXM7AkdzOpsXl_M9sztzOhHzMbMk8yYf3hWWsysUlbMjczczJDM4CcQJcyAzLwBzPbMl8zXzMPM6szUNMydasyqMBJKzJIjZ8y7KFgPzKxRc8zNCsypDcy4QG3M0U0IzMvM38y7zNPMpAPM_l5NzMkcKjYnzOcvIszJzO8oEErMzCRbPxfM58yIzJrMrhFyOczJzOXMusyszO3M5gw2agXM_GXMrH7M5DtBG8zLzNzM2zDMj8yrzNZhb8zvFQIlNMzIJXY5RhfMxUfMgC_M0cyUf8yLzJnM7NwBAQIVzOzMlsylbMz5RwvMuRA_P8y5aTDMpFhTTyNCccyezJbMtlrMrMy1awVJRczAzKfM0wbM4lzMvcz4zIXMujUEzPHMwnzMmg8OzJ_MznAszOstzL_M98yOzO7Mu8yazObMwU96zJ8rzLQOzOMgFWJPzLHMwTgAdw89am_MsBrM9XMOMCXMmsz6zN3MwszCzNovYszvzK9PzMxkzMNmd2cqRcyKzLNAYi5JzPDMjMzYzPALXhk0HcyQzIg_VyjM58zpY0HM38y7HhQIHX3MqwzMpTPMtCDM2szYVyrMoxwwXSM_zJ3MxxxgzMA1zOwIzNzM7sz2zNxxzMF8zMt7dszMzKbMlsyFVUrM7QXMv8yrzPoVEw3M6czQIMz5zIjM1l3M18yJzOlRQMzwE8yTIczyzLJEe3rM3SbMrsyYBATMpMyOHjEDzIIWLVLMqlzM6czRzPVzzOrMoitVK2_M6sy9zJwDSRDMgRLMisz0zPvMwConzMcYYcz9zIA",
        "challenge": "10275921379803826423",
    },
}

W3C_CRED_DEF_ID = "DyZ6N8yDTQsdJazNGRG1Wa:3:CL:1255405:faber.agent.degree_schema"
MOCK_W3C_CRED = {
    "@context": CONTEXT,
    "type": ["VerifiableCredential"],
    "issuer": "DyZ6N8yDTQsdJazNGRG1Wa",
    "credentialSubject": {
        "name": "Alice Smith",
        "birthdate_dateint": 20000711,
        "timestamp": 1720707393,
        "date": "2018-05-28",
        "degree": "Maths",
    },
    "proof": [
        {
            "cryptosuite": "anoncreds-2023",
            "type": "DataIntegrityProof",
            "proofPurpose": "assertionMethod",
            "verificationMethod": W3C_CRED_DEF_ID,
            "proofValue": "ukgGEqXNjaGVtYV9pZNkvRHlaNk44eURUUXNkSmF6TkdSRzFXYToyOmRlZ3JlZSBzY2hlbWE6OTQuNDYuNzGrY3JlZF9kZWZfaWTZPUR5WjZOOHlEVFFzZEphek5HUkcxV2E6MzpDTDoxMjU1NDA1OmZhYmVyLmFnZW50LmRlZ3JlZV9zY2hlbWGpc2lnbmF0dXJlgqxwX2NyZWRlbnRpYWyEo21fMtwAIC3M5nrMyTjMr0bM6syCRRgDB8z4TMzdzOrMucy5zKPMisy6zJxTzPB4bVHMszJEzIahYdwBAHTMiCVWzN_M98zRzPPMgkVezLDMlMz8fnN8UcyFzJXMigNmzMLMwiPMsczEJcz1zN7M0cyxzK7MwljMg2IqdVrMy8yHzM3M_mNpBMz9XF1SzPVIzLEmzPbM1BjMyHzM43zMkMzuEMznzKBkJ8zadMz8zMLM_CBqE2fMtRjMuczDzIR4SRhQSRbMhUrM1gHM18yUzP7MsszJFszqJRzMnczxCMzazO4LBcyIDMzEzLRmScyYzLXMgMyIzKHMnmBTzIvMysyHzIZ0AsyEzMvMmMz_SczrLczAzPhsecySzP3MysynzMzM0l0STcyPcszWzKFJOQUAzLbM5cybzLV7zOHM-Xw6zMdZzNEOzKlwZcyezOolV8yqzOIqQ3LMkMzmQczWzLELFcz0D8zRWVXMuMyHEszdF8z4REknzJvM7MyVzP52zPjMnkDMwyAmFg_MzgfM32TM7sy6zKZQzPhazJnMuCrMwGdpzIPM0szYzPZ1zPgMHTnM9MzozLJTzMIwzP7M0czwbkLMqh7MyqFl3ABLEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAALynM23zM3iFzAcy0IDwJUczoJ6F23AFVCcytzP_MusyjzKXMnRXMzcyOzI5wDS5VOMy0dMyVH8zXzJ_M-cyIASNgJns2zPfMzzLMlW1MU8zgBHlMzP0AzKvMssyYzLHM_U5vzJnMnmrM4DfM91k4PBIIVMzFzKl4F0bMmR8XzJfMxU7M2sz0Bcz_GMy2zMFEJsyybgwbzPZsaTPMlQcSzKY6eczbbTRZTMzyzIzMmxxTzKgKzMnM0czuHWjM9My0c8zKzMU9cEfMySXM98zbPw7MsHlyzOTM6lLMucyQzJjM7syHQQPMtUrMgczOaD7MlXp4SszXNcyFzLDMgGRxCczCClxHAVXM-goMLsyLY8yZK8yHzMY3zKVXKwp2zLjM4MzwzLpvUcyfzMYYAQAsG8zlzP0MMHwHJAfMgAPMtQDMhCsLzNjM1cy8B3fM6szVDczuYMy4zILMo0MizJgczOp8zLUFzLExzNPMkBMKzKbMhczLL8z6Tcy-dMzXRGXMnF0tR8yQzLVOTwXMp8yeC8yrdnnM9Myeb8zVPg8rzO9WzOLM0cy2zMtkBszjJszozNHMmgZhzOsHzNRSzLbMiszTJszbzMDMrszkLMyhzLJ4aRYnVw7M3cygzNxHzIhDSk_MpVPM-MzmzLkAVMyzzJMjID_MvEjMisz5zOnMy8zSFMzMbUXMu8zjzMBCUESscl9jcmVkZW50aWFswLtzaWduYXR1cmVfY29ycmVjdG5lc3NfcHJvb2aConNl3AEAzJBUX8zkPVXMwMyhzPwxdWrM58z6JMz3zLDMlVHMg8yvzJYOWcyKzJZozPk1zPcJP1ZZzK3MxDFJzOwyF0rMvC9xzP0XzNXMyH7MgMy4CMywzPgkzJPMy8zQzMhwMkvM4cz2zPYXzJMlzJXMnMyZzM00zMjM2lITzKFHLTjM2MzgzP_MnGzMvQTM6cy_zP_M4QnMznvM7y1bNMzJzNV-zK7MhczmzOEwzN_MrczpzLUcX8zfzKViDzRhzOjMlcycNQxxzK3MuMzFQMzpJ0NFZ1VZb8zhzLbMnx9eITLMnkvMn8yMKWnM8Mysaj7M0sylVsyXzKglM3TMhczzKMyqditdMMznfzx3JcyPQWNNzJzMpVPM3gHM5AYTzPUFHMzIzOLMrczEzKbMzklyzLdFATFczOrM7My9Sszdb0vM5HDMkFfMthApKMzfzNTMmXU7T8zEVMyfzM3MgszJzP7MlMy3zNwwzMwrzKPMuW7M2h1laTZ8DUEJzNknzMjMhWrMiaFj3AAgbszDzP0zBBoOzP7Mw8yNzPxmR8zPzJjMmnXMsxbM5cyxbMziDsyfOGPMkn83Hcy5",
        }
    ],
    "issuanceDate": "2024-07-11T14:16:37.818699601Z",
}


MOCK_W3C_PRES_REQ = {
    "version": "1.0",
    "name": "Age Verification",
    "nonce": "10275921379803826423",
    "requested_attributes": {
        "age-verification_attribute": {
            "names": ["name", "degree"],
            "restrictions": [
                {
                    "cred_def_id": "XwfxmPHisu4ZeZJBxkU9yG:3:CL:1255397:faber.agent.degree_schema"
                }
            ],
            "non_revoked": None,
        }
    },
    "requested_predicates": {
        "age-verification_predicate_0": {
            "name": "birthdate_dateint",
            "p_type": "<=",
            "p_value": 20060711,
            "restrictions": [
                {
                    "cred_def_id": "XwfxmPHisu4ZeZJBxkU9yG:3:CL:1255397:faber.agent.degree_schema"
                }
            ],
            "non_revoked": None,
        }
    },
}

CREDS_W3C_METADATA = [
    {
        "schema_id": "DyZ6N8yDTQsdJazNGRG1Wa:2:degree schema:94.46.71",
        "cred_def_id": W3C_CRED_DEF_ID,
        "proof_attrs": ["age-verification_attribute"],
        "proof_preds": ["age-verification_predicate_0"],
    }
]
CRED_DEFS = {
    W3C_CRED_DEF_ID: {
        "issuerId": "DyZ6N8yDTQsdJazNGRG1Wa",
        "schemaId": "1255405",
        "type": "CL",
        "tag": "faber.agent.degree_schema",
        "value": {
            "primary": {
                "n": "78608620890873858734716139623637557207236748041036599152087860745126232800410792474686186054868968424497768674471659783477234573895718704536655822053872184575008614841292292755773977336348717399858827225516378890147249022988821063695512091515834364155476868945304641152797312953150054985407637208945958685410474825963780210138068617216701521772433881960713239497125537276072745050872848837052182246304003140975539152490891455762930011410881319057199204939478636329525536320694776615565703536734007580747895464767619347642637144494483687798355974644728869518384135823542735258079646551610214657144799764002337455867901",
                "s": "16188463849412546891640922552292465562838101367601277992128848162820202387935769608332958870545695803678135322910482469331967995649945684153453613428194936775252383829449263033940304402931776444119090156613642012390697464244089486711501043789784798197678982862606452646894440861748599500239708128926125180765218509712903580782249517852590710823884952607759544656047904494741963723819532571620288071623944725608849021321079727999632523648744821503554842477891408906567788717005344644282892621994511877629914049255769558464858232935357888813001725324739131860348653048620151868133228245279216651937803301007796859809148",
                "r": {
                    "birthdate_dateint": "42321523370621472586426295102920092200716331198439684880751631005169912502513392348097827405713382392870351057417515373100336421627456337831914687021550420466679438139143550828676924942214829658708598705747050568068185496205586201490901792162868231737144423909851894492126047402918174761809991274506216232444807425545257717043121113480192917245436523222737120696808266827384291992173584922757202272224069178866180763756530481426792811525864900148511627086927934936010624074649095185853314015773559488856913353502772303104846002365950937203557327886208101237935658269328817806137015577417487042129268739188266656455838",
                    "date": "47282259420559086943282354673047808620932419444685357054526207482261794311207196772606516809551819368209083141763687874977322077707872712376289631235921781401772584236188435782335773816464936538343314437193149188695496363754402835761105905586890326049759144994498739076391071587660559576777847661304403271245065165765908143795884639501186137644910876556367493709012019351539913881016951128164156123676486321940301768797942946239395712213761026864750547641644598432981007440760115711494359109796696075613999758983159720327332976592446600931387265379696287489201164302606210759812951251459749023665486884533213382958924",
                    "degree": "60153759390848714691146484479392274133562816428435528963700343565611839246367965177204366729270994987186528861839431634550705896062648963500744334263396360649172661053623663271877417714825487333228521750963805284497902848380144500097326818107221339461337355057464350057644489048251733402813904879274691818024566869295619847347992487894269500406434050269901008003427810448810680954782112900343085845895480492152666064981934626021734769089741480544408251390015024949353947950985833441693746870940371240088472660349008250865323036018382101819925605477908670083234510669977333706521636643517669757263712285651480671951438",
                    "master_secret": "69257669973177891195832782226385437484565319036176021981079514766058012037020075082472532547922159493694471894000485304834996412469208708596022878079539321977476360893056967116987888112646019355423293211625152730912914998240197562948008429593765189331528456680630558434041646526770282490181462733482684331422552007518675879888699350145492730604666054815294351263210025713657554257619799792234183125395509745994947644401679118272165189772096427816057022796046378025001876575565286124494995797524148310731022099328971450166406932241022324582104980165553352886278647451559452244959090635336848709227787779160570063933995",
                    "name": "20458995966529508877165523940497074335676254937851332719993093001097100744552306136953000062015369385339571060757160555146949716351802164339509695035318750654149402579329767870427364614476802086365989932781978496588454470338424829968021806632220468888978073679053857540337854521557335332056805950661368230443201713983136437108228688581590915142044148670631100462101211547552233924677659787731955302143306592219729468409193258738692878243862745911138912819040092930225998295659341590790374387917484110201330953217976441156017380252924144810828368472774513296773468638305997951977112244485597812185563323173743994253211",
                    "timestamp": "55767528953315429878051834231669793196600608692476348478822466838866892203667049770363514845399988470984876484824791194529693386828453840143191575371505007314911858032119317394684557071814305693207914561247007744526376234461663749763344626892275790375329303602003127448116512649256486837242640739786646268214738380126354252590557742599164523249833790304064018841314251464684190236688674401539272344244633494008666223871816336334479751355462072021761471760400457594130730361236036333007268192137073491655784241381271398943622768412747645159079022640144426459562916039147003053408876716780772931082906004155301164003958",
                },
                "rctxt": "69242002209077646444514310939470989200750388630582386990437341114792494266313027866570067579621344642614422329584658119884298738951843345841692033119293233421722515070214666128612922992675853749542760511183059733091196542169569319157890885402821119674016451086771944772499261807433463241712959303954392628818958622889962938071354860935850496242072882611559360046105600408652229857914279710273402320462281517179305020310166125688712729993830879005117145310831038034017916827127155942964544984737924267675600573634963611253743739793194319943261044536689179850449758205865990098516834996191473012897372733669638888940862",
                "z": "51242731616392012121352727847246541820577049021543380134688343217924953181612446379624475830535519309732251115093334802429445309772964258385091473539691104656210384235471360862671875319609225020080430096553962226044417812537000598208481560358691321186674211172554176620640649721800215424554979904593840857301101168898237019175668547291667328531278280287272373147669305081767897596818974802633242089621024618909466800155483416470100552474374035099404782672849917829899639003946061969254481700312516981709486034820809575367370664657305843859286826375363922294149444916134844356018258300050455851007457182607232004610457",
            }
        },
    }
}
SCHEMAS = {
    "DyZ6N8yDTQsdJazNGRG1Wa:2:degree schema:94.46.71": {
        "issuerId": "DyZ6N8yDTQsdJazNGRG1Wa",
        "attrNames": [
            "date",
            "timestamp",
            "birthdate_dateint",
            "degree",
            "name",
        ],
        "name": "degree schema",
        "version": "94.46.71",
    }
}
MOCK_PRES = {
    "proof": {
        "proofs": [
            {
                "primary_proof": {
                    "eq_proof": {
                        "revealed_attrs": {
                            "biomarker_id": "33034450023603237719386825060766757598085121996569112944281451290292212516012",
                            "collected_on": "92231735610070911075924224447204218356256133056723930517696107260511721601349",
                            "concentration": "10",
                            "name": "85547618788485118809771015708850341281587970912661276233439574555663751388073",
                            "range": "106828626115908025842177441696860557581575579893927923198365300598359723920768",
                            "researcher_share": "101264834079306301897660576123112461042861436742738894013248454492965796383403",
                            "unit": "38351211041892038382023569421847544683371072212679556578649761181279472893849",
                        },
                        "a_prime": "80156520245352052628208149565161465200964633377479145197038408116901327106468493831807000641577246417448908134495822028339761705905365398613527463662816881507291787145610182891716009505407072490691097943029471835157968113065071523597746984296197661560454442163361095634052138951650373193896962906203169809352467024247772052836999799731422581068645748537557874869718897034120634529002420631012358510111427944993245065350416694516913472010105229188198167306183788520891926236449848811955646933539477960935319919207451858981065765523367984374104834278565184338252025155136368869580505679884921590811310587077071610172673",
                        "e": "115602723672843258810892161808995599340414281260248127600708536325470178701996999306086286379312077726886107268519700961209712187789855371",
                        "v": "1250383260306407741656763352595256748825474237767244783206569756476708112785930898966696687140808011529311298553822794830872826226191807175199015541611342880032928005827271961840046208463350458298210749103878893742434685172894883857423865293195542824393317226300133796527531436931435189766065404966370796699897584860421160155369018136946091524266742514828667575397735892093187106092545876795688095293610064164136737808333322708435913545499149948994191514980395955519036106660001526586674248282052492138917987323789012051794441548696998993861159018178474063785171288325900474499496141522583368982451169653258746506425495702762445790848698570457196767532483566475068200091609719957656394696938689265240025099424248587121592521826940348286940172887963179718337593603053496022182071613592070622825622277436966372346642772481567879001423472517233061740522533372490151585309457871632521280719357505751796940152868034526426510835",
                        "m": {
                            "master_secret": "3455871040557234123393960708120725061759594951341120214330342075748561632734634451036095543889895409812764789858455375956895105746442946098665140470124325622343440794421325163223",
                            "client_share": "4233663763294709836704307308997831519311512039775169744174375585917035614714239153287862168426091336550799195245481707264207548331415960277065672755643752404180562900805493953484",
                        },
                        "m2": "12942698897200869280316481431207639453433287089474860040781378232999349549981799159560238504559317917040820580596635945161264308025301203452846862479261473387068350544024561412435",
                    },
                    "ge_proofs": [],
                }
            },
            {
                "primary_proof": {
                    "eq_proof": {
                        "revealed_attrs": {
                            "jti_unique_identifier": "46414468020333259158238797309781111434265856695713363124410805958145233348633"
                        },
                        "a_prime": "52825780315318905340996188008133401356826233601375100674436798295026172087388431332751168238882607201020021795967828258295811342078457860422414605408183505911891895360825745994390769724939582542658347473498091021796952186290990181881158576706521445646669342676592451422000320708168877298354804819261007033664223006892049856834172427934815827786052257552492013807885418893279908149441273603109213847535482251568996326545234910687135167595657148526602160452192374611721411569543183642580629352619161783646990187905911781524203367796090408992624211661598980626941053749241077719601278347846928693650092940416717449494816",
                        "e": "40342480172543061520030194979861449480343743039487113094246205723322643070249538229638327935935486373873622430409109409257546971244601965",
                        "v": "217871997575635857881367472262154388060800564043554848081521162883333745687724235201324121915821236796357195214089699645741515836727882126142579489701412861659136426497703162695983681701205672924385915403141611021784136285588350763399255203187442277784718461565122805239422370067600654500115262174706580098147603414365915243447789285877195068031630371954678432401446457453517813298670236942253026249413255471803997869331683293818651006043399070308083119054618677128448043841313844695654424369871669436628257531643623230026240200330490039607166147891705813033761093730859310423856156850596341547950105490585959768382544221555877471751940512766452511773683786023245283041103270102119125303027835868565240336923422734962345750992898991606841120358203160628015844345063465293475128118937815965000466081345494616126511595974927544434058100817176268040385848789013718618727873445834393897904247054897801708217939187593785671914",
                        "m": {
                            "iat_consent_timestamp": "7919242808448912829024078929834347184203169048480606699350973804205285806978474375691141504249426249676222104091995582731720654507393708298132400435805626192291975477967402460279",
                            "master_secret": "3455871040557234123393960708120725061759594951341120214330342075748561632734634451036095543889895409812764789858455375956895105746442946098665140470124325622343440794421325163223",
                            "data_controller": "16070549690575784944224634793654539357398383214512772967411296056738507137421264813779497172425030465490587794790393434847583852932544021088761347641812155158324233253206392974293",
                            "notice": "2790610958721083178459621377821800672322230987466716467063649577108407884592339521820875278264969393963213925568888672412150769438560815981777952572004955362915245795447078373509",
                            "sensitive": "13552814315985495030467505807226704038231487014593307078913973520081443107274508887651839292151852713782653522711975492131914644109941607616672243509214979259100892541150351227883",
                            "services": "14860984314279608355643170908802532226194914773406547259519961082467311361623076451869406343140860447342041426195737612897540117192702117380288330928866665314831926780606136352645",
                            "sub_subject_identifier": "11736177517163751882849070942823049196298287414132249166618760803125435466270948777194044507635346721244111946358927525083691171695431736819244809221351813271261283779276670885101",
                            "moc_method_of_collection": "10026360820367693771310999595495505533281326977349798360729122862705999157070660881611421445424239119786180921960380892002204780026072600494332540208429642332890963846523547470729",
                            "jurisdiction_data_processing": "15829143141425514118932461858094583045441924952665872659029333578019676797278419825311275014912077620757631693167948665554731430154156737419706553672424812320891308795411687679270",
                            "iss_internet_processing_uri": "6900796243066434651671715348976599009606292569990892886896520779618011026060325075822786686418461731663661832508437549373109822105600719490952253743950241384782222356411498407620",
                            "version_consent_specification": "7796257942256624260327966366702213561879098947042014532961291550019706546662478888172243088973621029223408695289700984802154645011280488167967047321149956253054269901250137513345",
                            "policy_url": "12241676508867847022708464707584814145889660003604359058532137895063826021524887759921830911553663255421852525705197991376264187781979066233701110706958983099645275940668404311601",
                        },
                        "m2": "6509130065158989037891281073557909501783443634141673890142284302459280804904096303151728187237486991775852971807701594247754409108836089746736345158069365449802597798950172729241",
                    },
                    "ge_proofs": [],
                }
            },
        ],
        "aggregated_proof": {
            "c_hash": "81763443376178433216866153835042672285397553441148068557996780431098922863180",
            "c_list": [
                [2, 122, 246, 66, 85, 35, 17, 213, 1],
                [1, 162, 117, 246, 95, 154, 129, 32],
            ],
        },
    },
    "requested_proof": {
        "revealed_attrs": {
            "consent_attrs": {
                "sub_proof_index": 1,
                "raw": "205b1ea0-7848-48d4-b52b-339122d84f62",
                "encoded": "46414468020333259158238797309781111434265856695713363124410805958145233348633",
            }
        },
        "revealed_attr_groups": {
            "biomarker_attrs_0": {
                "sub_proof_index": 0,
                "values": {
                    "researcher_share": {
                        "raw": "bf712cb328a92862b57f0dc806dec12a",
                        "encoded": "101264834079306301897660576123112461042861436742738894013248454492965796383403",
                    },
                    "unit": {
                        "raw": "μM",
                        "encoded": "38351211041892038382023569421847544683371072212679556578649761181279472893849",
                    },
                    "concentration": {"raw": "10", "encoded": "10"},
                    "name": {
                        "raw": "Iron",
                        "encoded": "85547618788485118809771015708850341281587970912661276233439574555663751388073",
                    },
                    "range": {
                        "raw": "9.00-30.0",
                        "encoded": "106828626115908025842177441696860557581575579893927923198365300598359723920768",
                    },
                    "collected_on": {
                        "raw": "2020-07-05",
                        "encoded": "92231735610070911075924224447204218356256133056723930517696107260511721601349",
                    },
                    "biomarker_id": {
                        "raw": "c9ace7dc-0485-4f3f-b466-16a27a80acf1",
                        "encoded": "33034450023603237719386825060766757598085121996569112944281451290292212516012",
                    },
                },
            }
        },
        "self_attested_attrs": {},
        "unrevealed_attrs": {},
        "predicates": {},
    },
    "identifiers": [
        {
            "schema_id": "CsQY9MGeD3CQP4EyuVFo5m:2:MYCO Biomarker:0.0.3",
            "cred_def_id": "CsQY9MGeD3CQP4EyuVFo5m:3:CL:14951:MYCO_Biomarker",
        },
        {
            "schema_id": "FbozHyf7j5q7TDn2s8MXZN:2:MYCO Consent Enablement:0.1.0",
            "cred_def_id": "TUku9MDGa7QALbAJX4oAww:3:CL:531757:MYCO_Consent_Enablement",
        },
    ],
}

MOCK_SCHEMA = {
    "issuerId": "https://example.org/issuers/74acabe2-0edc-415e-ad3d-c259bac04c15",
    "name": "Example schema",
    "version": "0.0.1",
    "attrNames": ["name", "age", "vmax"],
}

MOCK_CRED_DEF = {
    "issuerId": "did:indy:sovrin:SGrjRL82Y9ZZbzhUDXokvQ",
    "schemaId": "did:indy:sovrin:SGrjRL82Y9ZZbzhUDXokvQ/anoncreds/v0/SCHEMA/MemberPass/1.0",
    "type": "CL",
    "tag": "latest",
    "value": {
        "primary": {
            "n": "779...397",
            "r": {
                "birthdate": "294...298",
                "birthlocation": "533...284",
                "citizenship": "894...102",
                "expiry_date": "650...011",
                "facephoto": "870...274",
                "firstname": "656...226",
                "link_secret": "521...922",
                "name": "410...200",
                "uuid": "226...757",
            },
            "rctxt": "774...977",
            "s": "750..893",
            "z": "632...005",
        },
        "revocation": {
            "g": "1 154...813 1 11C...D0D 2 095..8A8",
            "g_dash": "1 1F0...000",
            "h": "1 131...8A8",
            "h0": "1 1AF...8A8",
            "h1": "1 242...8A8",
            "h2": "1 072...8A8",
            "h_cap": "1 196...000",
            "htilde": "1 1D5...8A8",
            "pk": "1 0E7...8A8",
            "u": "1 18E...000",
            "y": "1 068...000",
        },
    },
}


MOCK_SCHEMAS = {
    "CsQY9MGeD3CQP4EyuVFo5m:2:MYCO Biomarker:0.0.3": {"value": {}},
    "FbozHyf7j5q7TDn2s8MXZN:2:MYCO Consent Enablement:0.1.0": {"value": {}},
}

MOCK_CRED_DEFS = {
    "CsQY9MGeD3CQP4EyuVFo5m:3:CL:14951:MYCO_Biomarker": {"value": {}},
    "TUku9MDGa7QALbAJX4oAww:3:CL:531757:MYCO_Consent_Enablement": {"value": {}},
}

MOCK_REV_REG_DEFS = {
    "TUku9MDGa7QALbAJX4oAww:3:TUku9MDGa7QALbAJX4oAww:3:CL:18:tag:CL_ACCUM:0": {
        "txnTime": 1500000000
    }
}

MOCK_REV_REG_DEF = {
    "issuerId": "did:web:example.org",
    "revocDefType": "CL_ACCUM",
    "credDefId": "Gs6cQcvrtWoZKsbBhD3dQJ:3:CL:140384:mctc",
    "tag": "MyCustomCredentialDefinition",
    "value": {
        "publicKeys": {"accumKey": {"z": "1 0BB...386"}},
        "maxCredNum": 666,
        "tailsLocation": "https://my.revocations.tails/tailsfile.txt",
        "tailsHash": "91zvq2cFmBZmHCcLqFyzv7bfehHH5rMhdAG5wTjqy2PE",
    },
}

MOCK_CRED = {
    "schema_id": "Sc886XPwD1gDcHwmmLDeR2:2:degree schema:45.101.94",
    "cred_def_id": "Sc886XPwD1gDcHwmmLDeR2:3:CL:229975:faber.agent.degree_schema",
    "rev_reg_id": "Sc886XPwD1gDcHwmmLDeR2:3:Sc886XPwD1gDcHwmmLDeR2:3:CL:18:tag:CL_ACCUM:0",
    "values": {
        "first_name": {"raw": "Alice", "encoded": "113...335"},
        "last_name": {"raw": "Garcia", "encoded": "532...452"},
        "birthdate_dateint": {"raw": "19981119", "encoded": "19981119"},
    },
    "signature": {
        "p_credential": {
            "m_2": "992...312",
            "a": "548...252",
            "e": "259...199",
            "v": "977...597",
        },
        "r_credential": None,
    },
    "signature_correctness_proof": {"se": "898...935", "c": "935...598"},
    "rev_reg": MOCK_REV_REG_DEF,
    "witness": "977...590",
}
