/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

use std::ffi::OsStr;
use std::ffi::OsString;
use std::fmt;
use std::io::Write as _;
use std::path::Path;
use std::path::PathBuf;
use std::process::Command;

use anyhow::Context as _;
use dupe::Dupe as _;
use pyrefly_python::module_name::ModuleName;
use pyrefly_python::module_path::ModulePathBuf;
use pyrefly_python::sys_info::SysInfo;
use serde::Deserialize;
use starlark_map::small_map::SmallMap;
use starlark_map::small_set::SmallSet;
use tempfile::NamedTempFile;
use vec1::Vec1;
#[allow(unused_imports)]
use vec1::vec1;

use crate::source_db::Target;

pub mod buck;
pub mod custom;

/// An enum representing something that has been included by the build system, and
/// which the build system should query for when building the sourcedb.
#[derive(Debug, PartialEq, Eq, Hash)]
pub(crate) enum Include {
    #[allow(unused)]
    Target(Target),
    Path(ModulePathBuf),
}

impl Include {
    pub fn path(path: ModulePathBuf) -> Self {
        Self::Path(path)
    }

    fn to_cli_arg(&self) -> impl Iterator<Item = &OsStr> {
        match self {
            Include::Target(target) => [OsStr::new("--target"), target.to_os_str()].into_iter(),
            Include::Path(path) => [OsStr::new("--file"), path.as_os_str()].into_iter(),
        }
    }
}

pub trait SourceDbQuerier: Send + Sync + fmt::Debug {
    fn query_source_db(
        &self,
        files: &SmallSet<Include>,
        cwd: &Path,
    ) -> anyhow::Result<TargetManifestDatabase> {
        if files.is_empty() {
            return Ok(TargetManifestDatabase {
                db: SmallMap::new(),
                root: cwd.to_path_buf(),
            });
        }

        let mut argfile =
            NamedTempFile::with_prefix("pyrefly_build_query_").with_context(|| {
                "Failed to create temporary argfile for querying source DB".to_owned()
            })?;
        let mut argfile_args = OsString::from("--");
        files.iter().flat_map(Include::to_cli_arg).for_each(|arg| {
            argfile_args.push("\n");
            argfile_args.push(arg);
        });

        argfile
            .as_file_mut()
            .write_all(argfile_args.as_encoded_bytes())
            .with_context(|| "Could not write to argfile when querying source DB".to_owned())?;

        let mut cmd = self.construct_command();
        cmd.arg(format!("@{}", argfile.path().display()));
        cmd.current_dir(cwd);

        let result = cmd.output()?;
        if !result.status.success() {
            let stdout = String::from_utf8(result.stdout)
                .unwrap_or_else(|_| "<Failed to parse stdout from source DB query>".to_owned());
            let stderr = String::from_utf8(result.stderr).unwrap_or_else(|_| {
                "<Failed to parse stderr from Buck source DB query>".to_owned()
            });

            return Err(anyhow::anyhow!(
                "Source DB query failed...\nSTDOUT: {stdout}\nSTDERR: {stderr}"
            ));
        }

        serde_json::from_slice(&result.stdout).with_context(|| {
            format!(
                "Failed to construct valid `TargetManifestDatabase` from querier result. Command run: `{}`",
                cmd.get_program().display(),
            )
        })
    }

    fn construct_command(&self) -> Command;
}

fn is_path_initfile(path: &Path) -> bool {
    path.ends_with("__init__.py") || path.ends_with("__init__.pyi")
}

#[derive(Debug, PartialEq, Eq, Deserialize, Clone)]
pub(crate) struct PythonLibraryManifest {
    pub deps: SmallSet<Target>,
    pub srcs: SmallMap<ModuleName, Vec1<ModulePathBuf>>,
    #[serde(flatten)]
    pub sys_info: SysInfo,
    pub buildfile_path: PathBuf,
    #[serde(default, skip)]
    pub packages: SmallMap<ModuleName, Vec1<ModulePathBuf>>,
}

impl PythonLibraryManifest {
    fn replace_alias_deps(&mut self, aliases: &SmallMap<Target, Target>) {
        self.deps = self
            .deps
            .iter()
            .map(|t| {
                if let Some(replace) = aliases.get(t) {
                    replace.dupe()
                } else {
                    t.dupe()
                }
            })
            .collect();
    }

    fn rewrite_relative_to_root(&mut self, root: &Path) {
        self.srcs.iter_mut().for_each(|(_, paths)| {
            paths
                .iter_mut()
                .for_each(|p| *p = ModulePathBuf::new(root.join(&**p)))
        });
        self.packages.iter_mut().for_each(|(_, paths)| {
            paths
                .iter_mut()
                .for_each(|p| *p = ModulePathBuf::new(root.join(&**p)))
        });
        self.buildfile_path = root.join(&self.buildfile_path);
    }

    /// Returns a map of package module name to init file or implicit package directory and
    /// path of where to continue upward searching for module names in later steps.
    ///
    /// When a real __init__ file is found on disk, we return `Ok()`, containing a tuple
    /// of all init files pointing to the given module name and path of where to continue
    /// searching from next.
    ///
    /// When no init file is found, we return `Err()`, which is both the directory
    /// where we synthesize an implicit package, as well as the path of where to continue
    /// searching from next.
    fn get_explicit_and_basic_implicit_packages(
        &self,
        target_root: &Path,
        all_dunder_inits: &SmallSet<ModulePathBuf>,
    ) -> SmallMap<ModuleName, Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>> {
        let mut start_packages: SmallMap<
            ModuleName,
            Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>,
        > = SmallMap::new();

        // attempt to push the given init file onto `start_packages` if `all_dunder_inits` knows about
        // it. Return false if we didn't push, true otherwise.
        let push_if_init_exists = |module: &ModuleName,
                                   paths: &Vec1<ModulePathBuf>,
                                   start_packages: &mut SmallMap<
            ModuleName,
            Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>,
        >| {
            let Some(parent) = paths.first().parent() else {
                return false;
            };
            if !parent.starts_with(target_root) {
                return false;
            }
            let Some(parent_parent) = parent.parent() else {
                return false;
            };

            // if one of the paths here is a dunder init, then all paths here will be
            // a dunder init
            let path = paths.first();
            if !all_dunder_inits.contains(path) {
                return false;
            }
            start_packages.insert(
                module.dupe(),
                // `parent_parent` is used here, since we want to continue searching
                // from the package's parent, which is the file's directory's parent.
                Ok((
                    paths.mapped_ref(|p| ModulePathBuf::from_path(p)),
                    ModulePathBuf::from_path(parent_parent),
                )),
            );
            true
        };
        for (module, paths) in &self.srcs {
            if push_if_init_exists(module, paths, &mut start_packages) {
                continue;
            }

            let path = paths.first();
            let Some(parent_path) = path.parent() else {
                continue;
            };

            if push_if_init_exists(
                module,
                &vec1![
                    ModulePathBuf::new(path.join("__init__.py")),
                    ModulePathBuf::new(path.join("__init__.pyi"))
                ],
                &mut start_packages,
            ) {
                continue;
            }

            if !parent_path.starts_with(target_root) {
                continue;
            }
            let Some(parent_module) = module.parent() else {
                continue;
            };

            start_packages
                .entry(parent_module)
                .or_insert(Err(parent_path));
        }

        start_packages
    }

    /// Given a map of modules to real or synthesized packages, synthesize missing packages up to
    /// and including this manifest's build file.
    fn fill_ancestor_synthesized_packages(
        &self,
        start_packages: SmallMap<ModuleName, Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>>,
        target_root: &Path,
    ) -> SmallMap<ModuleName, Vec1<ModulePathBuf>> {
        // the result we're going to use, with all the files we've found so far
        let mut inits: SmallMap<ModuleName, Vec1<ModulePathBuf>> = start_packages
            .iter()
            .map(|(name, path)| {
                let files = match path {
                    Ok((files, _)) => files.clone(),
                    Err(file) => vec1![ModulePathBuf::from_path(file)],
                };
                (name.dupe(), files)
            })
            .collect();

        // fill in implicit packages for parent directories, if there's not an entry already
        for (mut module, path) in start_packages {
            let mut path = match &path {
                Ok((_, next_path)) => &**next_path,
                Err(next_path) => *next_path,
            };
            while let Some(parent_module) = module.parent() {
                let Some(parent_path) = path.parent() else {
                    break;
                };

                if inits.contains_key(&parent_module) || !parent_path.starts_with(target_root) {
                    break;
                }

                inits.insert(
                    parent_module.dupe(),
                    vec1![ModulePathBuf::from_path(parent_path)],
                );
                module = parent_module;
                path = parent_path;
            }
        }

        inits
    }

    /// Get all explicit and implicit dunder inits, preferring explicit. Produces
    /// dunder inits all the way up to the directory containing this manifest's build file.
    fn fill_implicit_packages(&mut self, all_dunder_inits: &SmallSet<ModulePathBuf>) {
        let Some(target_root) = self.buildfile_path.parent() else {
            return;
        };
        let start_packages =
            self.get_explicit_and_basic_implicit_packages(target_root, all_dunder_inits);

        self.packages = self.fill_ancestor_synthesized_packages(start_packages, target_root);
    }
}

#[derive(Debug, PartialEq, Eq, Deserialize, Clone)]
#[serde(untagged)]
pub(crate) enum TargetManifest {
    Library(PythonLibraryManifest),
    Alias { alias: Target },
}

#[derive(Debug, PartialEq, Eq, Deserialize, Clone)]
pub(crate) struct TargetManifestDatabase {
    db: SmallMap<Target, TargetManifest>,
    pub root: PathBuf,
}

impl TargetManifestDatabase {
    pub fn produce_map(mut self) -> SmallMap<Target, PythonLibraryManifest> {
        let mut result = SmallMap::new();
        let aliases: SmallMap<Target, Target> = self
            .db
            .iter()
            .filter_map(|(t, manifest)| match manifest {
                TargetManifest::Alias { alias } => Some((t.dupe(), alias.dupe())),
                _ => None,
            })
            .collect();

        let mut explicit_dunder_inits = SmallSet::new();
        for manifest in self.db.values_mut() {
            match manifest {
                TargetManifest::Alias { .. } => continue,
                TargetManifest::Library(lib) => {
                    lib.replace_alias_deps(&aliases);
                    lib.rewrite_relative_to_root(&self.root);

                    for paths in lib.srcs.values() {
                        if !is_path_initfile(paths.first()) {
                            continue;
                        }
                        for path in paths {
                            explicit_dunder_inits.insert(path.dupe());
                        }
                    }
                }
            }
        }

        for (target, manifest) in self.db {
            match manifest {
                TargetManifest::Alias { .. } => continue,
                TargetManifest::Library(mut lib) => {
                    lib.fill_implicit_packages(&explicit_dunder_inits);
                    result.insert(target, lib);
                }
            }
        }
        result
    }
}

#[cfg(test)]
mod tests {
    use pretty_assertions::assert_eq;
    use pyrefly_python::sys_info::PythonPlatform;
    use pyrefly_python::sys_info::PythonVersion;
    use starlark_map::smallmap;

    use super::*;

    impl TargetManifestDatabase {
        pub fn new(db: SmallMap<Target, TargetManifest>, root: PathBuf) -> Self {
            TargetManifestDatabase { db, root }
        }

        /// This is a simplified sourcedb taken from the BXL output run on pyre/client/log/log.py.
        /// We also add a few extra entries to model some of the behavior around multiple entries
        /// (i.e. multiple file paths corresponding to a module path, multiple module paths in
        /// different targets).
        pub fn get_test_database() -> Self {
            TargetManifestDatabase::new(
                smallmap! {
                    Target::from_string("//colorama:py".to_owned()) => TargetManifest::lib(
                        &[
                        (
                            "colorama",
                            &[
                            "colorama/__init__.py",
                            "colorama/__init__.pyi",
                            ]
                        ),
                        ],
                        &[],
                        "colorama/BUCK",
                        &[],
                    ),
                    Target::from_string("//colorama:colorama".to_owned()) => TargetManifest::alias(
                        "//colorama:py"
                    ),
                    Target::from_string("//click:py".to_owned()) => TargetManifest::lib(
                        &[
                        (
                            "click",
                            &[
                            "click/__init__.pyi",
                            "click/__init__.py",
                            ],
                        )
                        ],
                        &[
                        "//colorama:colorama"
                        ],
                        "click/BUCK",
                        &[],
                    ),
                    Target::from_string("//click:click".to_owned()) => TargetManifest::alias(
                        "//click:py"
                    ),
                    Target::from_string("//pyre/client/log:log".to_owned()) => TargetManifest::lib(
                        &[
                        (
                            "pyre.client.log",
                            &[
                            "pyre/client/log/__init__.py"
                            ]
                        ),
                        (
                            "pyre.client.log.log",
                            &[
                            "pyre/client/log/log.py",
                            "pyre/client/log/log.pyi",
                            ],
                        ),
                        (
                            "pyre.client.log.format",
                            &[
                            "pyre/client/log/format.py",
                            ],
                        ),
                        ],
                        &[
                        "//click:click"
                        ],
                        "pyre/client/log/BUCK",
                        &[],
                    ),
                    Target::from_string("//pyre/client/log:log2".to_owned()) => TargetManifest::lib(
                        &[
                        (
                            "log",
                            &[
                            "pyre/client/log/__init__.py"
                            ]
                        ),
                        (
                            "log.log",
                            &[
                            "pyre/client/log/log.py",
                            "pyre/client/log/log.pyi",
                            ]
                        ),
                        (
                            "log.format",
                            &[
                            "pyre/client/log/format.py",
                            ]
                        ),
                        ],
                        &[
                        "//click:click"
                        ],
                        "pyre/client/log/BUCK",
                        &[],
                    ),
                    Target::from_string("//implicit_package/test:main".to_owned()) => TargetManifest::lib(
                            &[
                            (
                                "implicit_package.main",
                                &[
                                "implicit_package/test/main.py",
                                ],
                            ),
                            ],
                            &[
                            "//implicit_package/test:lib",
                            ],
                            "implicit_package/test/BUCK",
                            &[],
                    ),
                    Target::from_string("//implicit_package/test:lib".to_owned()) => TargetManifest::lib(
                            &[
                            (
                                "implicit_package.lib.utils",
                                &[
                                "implicit_package/test/lib/utils.py",
                                ],
                            ),
                            (
                                "implicit_package.package_boundary_violation",
                                &[
                                "implicit_package/package_boundary_violation.py",
                                ]
                            ),
                            (
                                "implicit_package.deeply.nested.package.file",
                                &[
                                "implicit_package/test/deeply/nested/package/file.py",
                                ],
                            ),
                            ],
                            &["//external:package"],
                            "implicit_package/test/BUCK",
                            &[]
                    ),
                    Target::from_string("//external:package".to_owned()) => TargetManifest::lib(
                        &[
                        (
                            "external_package.main",
                            &[
                            "/path/to/another/repository/package/external_package/main.py",
                            ]
                        ),
                        (
                            "external_package.non_python_file",
                            &[
                            "/path/to/another/repository/package/external_package/non_python_file.thrift",
                            ],
                        ),
                        ],
                        &[],
                        "/path/to/another/repository/package/BUCK",
                        &[]
                    ),
                },
                PathBuf::from("/path/to/this/repository"),
            )
        }
    }

    fn map_srcs(
        srcs: &[(&str, &[&str])],
        prefix_paths: Option<&str>,
    ) -> SmallMap<ModuleName, Vec1<ModulePathBuf>> {
        let prefix = prefix_paths.map(Path::new);
        let map_path = |p| {
            prefix.map_or_else(
                || ModulePathBuf::from_path(Path::new(p)),
                |prefix| ModulePathBuf::new(prefix.join(p)),
            )
        };
        srcs.iter()
            .map(|(n, paths)| {
                (
                    ModuleName::from_str(n),
                    Vec1::try_from_vec(paths.iter().map(map_path).collect()).unwrap(),
                )
            })
            .collect()
    }

    fn map_deps(deps: &[&str]) -> SmallSet<Target> {
        deps.iter()
            .map(|s| Target::from_string((*s).to_owned()))
            .collect()
    }

    fn map_implicit_packages(
        inits: &[(&str, &[&str])],
        prefix_paths: Option<&str>,
    ) -> SmallMap<ModuleName, Vec1<ModulePathBuf>> {
        let prefix = prefix_paths.map(Path::new);
        let map_path = |p| {
            prefix.map_or_else(
                || ModulePathBuf::from_path(Path::new(p)),
                |prefix| ModulePathBuf::new(prefix.join(p)),
            )
        };
        inits
            .iter()
            .map(|(n, paths)| {
                (
                    ModuleName::from_str(n),
                    Vec1::try_from_vec(paths.iter().map(map_path).collect()).unwrap(),
                )
            })
            .collect()
    }

    impl TargetManifest {
        fn alias(target: &str) -> Self {
            TargetManifest::Alias {
                alias: Target::from_string(target.to_owned()),
            }
        }

        pub fn lib(
            srcs: &[(&str, &[&str])],
            deps: &[&str],
            buildfile: &str,
            implicit_packages: &[(&str, &[&str])],
        ) -> Self {
            TargetManifest::Library(PythonLibraryManifest {
                srcs: map_srcs(srcs, None),
                deps: map_deps(deps),
                sys_info: SysInfo::new(PythonVersion::new(3, 12, 0), PythonPlatform::linux()),
                buildfile_path: PathBuf::from(buildfile),
                packages: map_implicit_packages(implicit_packages, None),
            })
        }
    }

    impl PythonLibraryManifest {
        fn new(
            srcs: &[(&str, &[&str])],
            deps: &[&str],
            buildfile: &str,
            inits: &[(&str, &[&str])],
        ) -> Self {
            let root = "/path/to/this/repository";
            Self {
                srcs: map_srcs(srcs, Some(root)),
                deps: map_deps(deps),
                sys_info: SysInfo::new(PythonVersion::new(3, 12, 0), PythonPlatform::linux()),
                buildfile_path: PathBuf::from(root).join(buildfile),
                packages: map_implicit_packages(inits, Some(root)),
            }
        }
    }

    #[test]
    fn example_json_parses() {
        const EXAMPLE_JSON: &str = r#"
{
  "db": {
    "//colorama:py": {
      "srcs": {
        "colorama": [
          "colorama/__init__.py",
          "colorama/__init__.pyi"
        ]
      },
      "deps": [],
      "buildfile_path": "colorama/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//colorama:colorama": {
      "alias": "//colorama:py"
    },
    "//click:py": {
      "srcs": {
        "click": [
          "click/__init__.pyi",
          "click/__init__.py"
        ]
      },
      "deps": [
        "//colorama:colorama"
      ],
      "buildfile_path": "click/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//click:click": {
      "alias": "//click:py"
    },
    "//pyre/client/log:log": {
      "srcs": {
        "pyre.client.log": [
          "pyre/client/log/__init__.py"
        ],
        "pyre.client.log.log": [
          "pyre/client/log/log.py",
          "pyre/client/log/log.pyi"
        ],
        "pyre.client.log.format": [
          "pyre/client/log/format.py"
        ]
      },
      "deps": [
        "//click:click"
      ],
      "buildfile_path": "pyre/client/log/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//pyre/client/log:log2": {
      "srcs": {
        "log": [
          "pyre/client/log/__init__.py"
        ],
        "log.log": [
          "pyre/client/log/log.py",
          "pyre/client/log/log.pyi"
        ],
        "log.format": [
          "pyre/client/log/format.py"
        ]
      },
      "deps": [
        "//click:click"
      ],
      "buildfile_path": "pyre/client/log/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//implicit_package/test:main": {
      "srcs": {
          "implicit_package.main": [
              "implicit_package/test/main.py"
          ]
      },
      "deps": ["//implicit_package/test:lib"],
      "buildfile_path": "implicit_package/test/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//implicit_package/test:lib": {
      "srcs": {
          "implicit_package.lib.utils": [
              "implicit_package/test/lib/utils.py"
          ],
          "implicit_package.package_boundary_violation": [
              "implicit_package/package_boundary_violation.py"
          ],
          "implicit_package.deeply.nested.package.file": [
              "implicit_package/test/deeply/nested/package/file.py"
          ]
      }, 
      "deps": ["//external:package"],
      "buildfile_path": "implicit_package/test/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    },
    "//external:package": {
      "srcs": {
          "external_package.main": [
              "/path/to/another/repository/package/external_package/main.py"
          ],
          "external_package.non_python_file": [
              "/path/to/another/repository/package/external_package/non_python_file.thrift"
          ]
      }, 
      "deps": [],
      "buildfile_path": "/path/to/another/repository/package/BUCK",
      "python_version": "3.12",
      "python_platform": "linux"
    }
  },
  "root": "/path/to/this/repository"
}
        "#;
        let parsed: TargetManifestDatabase = serde_json::from_str(EXAMPLE_JSON).unwrap();
        assert_eq!(parsed, TargetManifestDatabase::get_test_database());
    }

    #[test]
    fn test_produce_db() {
        let expected = smallmap! {
            Target::from_string("//colorama:py".to_owned()) => PythonLibraryManifest::new(
                &[
                    (
                        "colorama",
                        &[
                            "colorama/__init__.py",
                            "colorama/__init__.pyi",
                        ]
                    ),
                ],
                &[],
                "colorama/BUCK",
                &[
                    ("colorama", &[
                        "colorama/__init__.py",
                        "colorama/__init__.pyi",
                    ]),
                ],
            ),
            Target::from_string("//click:py".to_owned()) => PythonLibraryManifest::new(
                &[
                    (
                        "click",
                        &[
                            "click/__init__.pyi",
                            "click/__init__.py",
                        ],
                    )
                ],
                &[
                    "//colorama:py"
                ],
                "click/BUCK",
                &[
                    ("click", &[
                        "click/__init__.pyi",
                        "click/__init__.py",
                    ]),
                ],
            ),
            Target::from_string("//pyre/client/log:log".to_owned()) => PythonLibraryManifest::new(
                &[
                    (
                        "pyre.client.log",
                        &[
                            "pyre/client/log/__init__.py"
                        ]
                    ),
                    (
                        "pyre.client.log.log",
                        &[
                            "pyre/client/log/log.py",
                            "pyre/client/log/log.pyi",
                        ]
                    ),
                    (
                        "pyre.client.log.format",
                        &[
                            "pyre/client/log/format.py",
                        ],
                    ),
                ],
                &[
                    "//click:py"
                ],
                "pyre/client/log/BUCK",
                &[
                    ("pyre.client.log", &[
                     "pyre/client/log/__init__.py",
                    ]),
                ],
            ),
            Target::from_string("//pyre/client/log:log2".to_owned()) => PythonLibraryManifest::new(
                &[
                    (
                        "log",
                        &[
                            "pyre/client/log/__init__.py"
                        ]
                    ),
                    (
                        "log.log",
                        &[
                            "pyre/client/log/log.py",
                            "pyre/client/log/log.pyi",
                        ]
                    ),
                    (
                        "log.format",
                        &[
                            "pyre/client/log/format.py",
                        ],
                    ),
                ],
                &[
                    "//click:py"
                ],
                "pyre/client/log/BUCK",
                &[
                    ("log", &[
                        "pyre/client/log/__init__.py",
                    ]),
                ],
            ),
            Target::from_string("//implicit_package/test:main".to_owned()) => PythonLibraryManifest::new(
                &[
                (
                    "implicit_package.main", &[
                        "implicit_package/test/main.py"
                    ]
                )
                ],
                &[
                    "//implicit_package/test:lib"
                ],
                "implicit_package/test/BUCK",
                &[
                ("implicit_package", &[
                        "implicit_package/test",
                    ],
                )
                ],
            ),
            Target::from_string("//implicit_package/test:lib".to_owned()) => PythonLibraryManifest::new(
                &[
                (
                    "implicit_package.lib.utils", &[
                        "implicit_package/test/lib/utils.py"
                    ],
                ),
                (
                    "implicit_package.package_boundary_violation", &[
                        "implicit_package/package_boundary_violation.py",
                    ],
                ),
                (
                    "implicit_package.deeply.nested.package.file", &[
                        "implicit_package/test/deeply/nested/package/file.py",
                    ],
                ),
                ],
                &["//external:package"],
                "implicit_package/test/BUCK",
                &[
                ("implicit_package", &[
                        "implicit_package/test",
                    ],
                ),
                ("implicit_package.lib", &[
                        "implicit_package/test/lib",
                    ],
                ),
                ("implicit_package.deeply.nested.package", &[
                        "implicit_package/test/deeply/nested/package",
                    ],
                ),
                ("implicit_package.deeply.nested", &[
                        "implicit_package/test/deeply/nested",
                    ],
                ),
                ("implicit_package.deeply", &[
                        "implicit_package/test/deeply",
                    ],
                ),
                ],
            ),
            Target::from_string("//external:package".to_owned()) => PythonLibraryManifest::new(
                &[
                ("external_package.main", &[
                 "/path/to/another/repository/package/external_package/main.py"
                    ]
                ),
                (
                    "external_package.non_python_file", &[
                        "/path/to/another/repository/package/external_package/non_python_file.thrift",
                    ],
                ),
                ],
                &[],
                "/path/to/another/repository/package/BUCK",
                &[
                ("external_package", &[
                 "/path/to/another/repository/package/external_package",
                    ],
                ),
                ],
            ),
        };
        assert_eq!(
            TargetManifestDatabase::get_test_database().produce_map(),
            expected
        );
    }

    #[test]
    fn test_package_finding() {
        let db = TargetManifestDatabase::get_test_database();
        let all_dunder_inits = [
            "colorama/__init__.pyi",
            "colorama/__init__.py",
            "click/__init__.pyi",
            "click/__init__.py",
            "pyre/client/log/__init__.py",
        ]
        .iter()
        .map(|p| ModulePathBuf::from_path(Path::new(p)))
        .collect::<SmallSet<ModulePathBuf>>();

        let result: (
            SmallMap<
                Target,
                SmallMap<ModuleName, Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>>,
            >,
            SmallMap<Target, SmallMap<ModuleName, Vec1<ModulePathBuf>>>,
        ) = db
            .db
            .iter()
            .filter_map(|(t, m)| match m {
                TargetManifest::Library(lib) => Some((t, lib)),
                TargetManifest::Alias { .. } => None,
            })
            .fold(
                (SmallMap::new(), SmallMap::new()),
                |(mut first, mut second), (t, l)| {
                    let root = l.buildfile_path.parent().unwrap();
                    let base_packages =
                        l.get_explicit_and_basic_implicit_packages(root, &all_dunder_inits);
                    first.insert(t.dupe(), base_packages.clone());
                    second.insert(
                        t.dupe(),
                        l.fill_ancestor_synthesized_packages(base_packages, root),
                    );
                    (first, second)
                },
            );

        let expected_start_packages: SmallMap<
            Target,
            SmallMap<ModuleName, Result<(Vec1<ModulePathBuf>, ModulePathBuf), &Path>>,
        > = smallmap! {
            "//colorama:py" => smallmap! {
                "colorama" => Ok((
                    vec1![
                        "colorama/__init__.py",
                        "colorama/__init__.pyi",
                    ],
                    "",
                )),
            },
            "//click:py" => smallmap! {
                "click" => Ok((
                    vec1![
                        "click/__init__.pyi",
                        "click/__init__.py",
                    ],
                    "",
                )),
            },
            "//pyre/client/log:log" => smallmap! {
                "pyre.client.log" => Ok((
                    vec1![
                        "pyre/client/log/__init__.py",
                    ],
                    "pyre/client",
                )),
            },
            "//pyre/client/log:log2" => smallmap! {
                "log" => Ok((
                    vec1![
                        "pyre/client/log/__init__.py",
                    ],
                    "pyre/client",
                )),
            },
            "//implicit_package/test:main" => smallmap! {
                "implicit_package" => Err(
                    "implicit_package/test",
                ),
            },
            "//implicit_package/test:lib" => smallmap! {
                "implicit_package.lib" => Err(
                    "implicit_package/test/lib",
                ),
                "implicit_package.deeply.nested.package" => Err(
                    "implicit_package/test/deeply/nested/package",
                ),
            },
            "//external:package" => smallmap! {
                "external_package" => Err(
                    "/path/to/another/repository/package/external_package",
                ),
            },
        }
        .into_iter()
        .map(|(t, m)| {
            (
                Target::from_string(t.to_owned()),
                m.into_iter()
                    .map(|(name, r)| {
                        (
                            ModuleName::from_str(name),
                            match r {
                                Ok((paths, next)) => Ok((
                                    paths.mapped(|p| ModulePathBuf::from_path(Path::new(p))),
                                    ModulePathBuf::from_path(Path::new(next)),
                                )),
                                Err(next) => Err(Path::new(next)),
                            },
                        )
                    })
                    .collect(),
            )
        })
        .collect();

        assert_eq!(result.0, expected_start_packages);

        let expected_ancestor_synthesized_packages = smallmap! {
            "//colorama:py" => smallmap! {
                "colorama" => vec1![
                        "colorama/__init__.py",
                        "colorama/__init__.pyi",
                ],
            },
            "//click:py" => smallmap! {
                "click" => vec1![
                        "click/__init__.pyi",
                        "click/__init__.py",
                ],
            },
            "//pyre/client/log:log" => smallmap! {
                "pyre.client.log" => vec1![
                        "pyre/client/log/__init__.py",
                ],
            },
            "//pyre/client/log:log2" => smallmap! {
                "log" => vec1![
                        "pyre/client/log/__init__.py",
                ],
            },
            "//implicit_package/test:main" => smallmap! {
                "implicit_package" => vec1![
                    "implicit_package/test",
                ],
            },
            "//implicit_package/test:lib" => smallmap! {
                "implicit_package.lib" => vec1![
                    "implicit_package/test/lib",
                ],
                "implicit_package.deeply.nested.package" => vec1![
                    "implicit_package/test/deeply/nested/package",
                ],
                "implicit_package.deeply.nested" => vec1![
                    "implicit_package/test/deeply/nested",
                ],
                "implicit_package.deeply" => vec1![
                    "implicit_package/test/deeply",
                ],
                "implicit_package" => vec1![
                    "implicit_package/test",
                ],
            },
            "//external:package" => smallmap!{
                "external_package" => vec1![
                    "/path/to/another/repository/package/external_package",
                ],
            }
        }
        .into_iter()
        .map(|(t, m)| {
            (
                Target::from_string(t.to_owned()),
                m.into_iter()
                    .map(|(name, paths)| {
                        (
                            ModuleName::from_str(name),
                            paths.mapped(|p| ModulePathBuf::from_path(Path::new(p))),
                        )
                    })
                    .collect(),
            )
        })
        .collect();

        assert_eq!(result.1, expected_ancestor_synthesized_packages);
    }
}
