/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

//! Things defined as `Foo: _SpecialForm` which have a builtin meaning.

use std::str::FromStr;

use dupe::Dupe;
use parse_display::Display;
use parse_display::FromStr;
use pyrefly_derive::TypeEq;
use pyrefly_derive::Visit;
use pyrefly_derive::VisitMut;
use ruff_python_ast::Expr;
use ruff_python_ast::ExprSubscript;
use ruff_python_ast::name::Name;

use crate::annotation::Qualifier;
use crate::types::NeverStyle;
use crate::types::Type;

#[derive(Debug, Clone, Copy, Dupe, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[derive(Visit, VisitMut, TypeEq, Display, FromStr)]
pub enum SpecialForm {
    Annotated,
    Callable,
    ClassVar,
    Concatenate,
    Final,
    Generic,
    Literal,
    LiteralString,
    Never,
    NoReturn,
    NotRequired,
    Optional,
    Protocol,
    ReadOnly,
    Required,
    #[display("Self")]
    SelfType,
    Tuple,
    Type,
    TypeAlias,
    TypeGuard,
    TypeIs,
    TypedDict,
    Union,
    Unpack,
}

impl SpecialForm {
    pub fn new(name: &Name, annotation: &Expr) -> Option<Self> {
        if name.as_str() == "Generic" {
            if let Expr::Subscript(ExprSubscript {
                value: box Expr::Name(x),
                slice: box Expr::Name(y),
                ..
            }) = annotation
                && x.id == "type"
                && y.id == "_Generic"
            {
                return Some(SpecialForm::Generic);
            }
        } else if !matches!(annotation, Expr::Name(x) if x.id == "_SpecialForm") {
            return None;
        }
        SpecialForm::from_str(name.as_str()).ok()
    }

    pub fn to_type(self) -> Type {
        match self {
            SpecialForm::LiteralString => Type::type_form(Type::LiteralString),
            SpecialForm::Never => Type::type_form(Type::Never(NeverStyle::Never)),
            SpecialForm::NoReturn => Type::type_form(Type::Never(NeverStyle::NoReturn)),
            _ => Type::type_form(Type::SpecialForm(self)),
        }
    }

    /// Keep this in sync with `apply_special_form`
    pub fn can_be_subscripted(self) -> bool {
        match self {
            Self::LiteralString
            | Self::Never
            | Self::NoReturn
            | Self::SelfType
            | Self::TypeAlias
            | Self::TypedDict => false,
            _ => true,
        }
    }

    pub fn to_qualifier(self) -> Option<Qualifier> {
        match self {
            Self::Annotated => Some(Qualifier::Annotated),
            Self::ClassVar => Some(Qualifier::ClassVar),
            Self::Final => Some(Qualifier::Final),
            Self::NotRequired => Some(Qualifier::NotRequired),
            Self::ReadOnly => Some(Qualifier::ReadOnly),
            Self::Required => Some(Qualifier::Required),
            Self::TypeAlias => Some(Qualifier::TypeAlias),
            _ => None,
        }
    }
}

#[cfg(test)]
mod tests {
    use std::str::FromStr;

    use super::*;

    #[test]
    fn test_special_form_display() {
        assert_eq!(SpecialForm::Annotated.to_string(), "Annotated");
        assert_eq!(SpecialForm::Callable.to_string(), "Callable");
        assert_eq!(SpecialForm::SelfType.to_string(), "Self");
    }

    #[test]
    fn test_special_form_from_str() {
        assert_eq!(
            SpecialForm::from_str("Annotated").unwrap(),
            SpecialForm::Annotated
        );
        assert_eq!(
            SpecialForm::from_str("Callable").unwrap(),
            SpecialForm::Callable
        );
        assert_eq!(
            SpecialForm::from_str("Self").unwrap(),
            SpecialForm::SelfType
        );
        assert!(SpecialForm::from_str("NotASpecial").is_err());
    }
}
