from pathlib import Path
from typing import Optional

from PyQt6 import QtWidgets

from pycroglia.core.enums import SkimageCellConnectivity
from pycroglia.core.labeled_cells import SkimageImgLabeling
from pycroglia.ui.widgets.imagefilters.results import FilterResults
from pycroglia.ui.widgets.segmentation.editor import SegmentationEditor
from pycroglia.ui.widgets.segmentation.results import SegmentationResults


class SegmentationEditorStack(QtWidgets.QWidget):
    """Widget that manages a tabbed interface for segmentation editors.

    Creates and manages multiple SegmentationEditor widgets in a tabbed interface,
    allowing users to perform cell segmentation on multiple filtered images
    simultaneously. Each tab corresponds to one filtered image result.

    Attributes:
        tabs (QtWidgets.QTabWidget): Tab widget containing segmentation editors.
        headers_text (Optional[list[str]]): Text for cell list headers.
        rollback_button_text (Optional[str]): Text for rollback button.
        segmentation_button_text (Optional[str]): Text for segmentation button.
        progress_title (Optional[str]): Title for progress dialog.
        progress_cancel_text (Optional[str]): Text for progress cancel button.
    """

    def __init__(
        self,
        headers_text: Optional[list[str]] = None,
        rollback_button_text: Optional[str] = None,
        segmentation_button_text: Optional[str] = None,
        progress_title: Optional[str] = None,
        progress_cancel_text: Optional[str] = None,
        parent: Optional[QtWidgets.QWidget] = None,
    ):
        """Initialize the SegmentationEditorStack widget.

        Args:
            headers_text (Optional[list[str]]): Custom text for cell list headers.
            rollback_button_text (Optional[str]): Custom text for rollback button.
            segmentation_button_text (Optional[str]): Custom text for segmentation button.
            progress_title (Optional[str]): Custom title for progress dialog.
            progress_cancel_text (Optional[str]): Custom text for progress cancel button.
            parent (Optional[QtWidgets.QWidget]): Parent widget.
        """
        super().__init__(parent=parent)

        # Store text parameters
        self.headers_text = headers_text
        self.rollback_button_text = rollback_button_text
        self.segmentation_button_text = segmentation_button_text
        self.progress_title = progress_title
        self.progress_cancel_text = progress_cancel_text

        # Widgets
        self.tabs = QtWidgets.QTabWidget(self)

        # Layout
        layout = QtWidgets.QVBoxLayout(self)
        layout.addWidget(self.tabs)
        self.setLayout(layout)

    def add_tabs(self, results: list[FilterResults]):
        """Clear existing tabs and add a new tab for each filter result.

        Creates a SegmentationEditor for each FilterResults object and adds it
        as a tab. The tab title is derived from the file path.

        Args:
            results (list[FilterResults]): List of filter results to create tabs for.
                Each result will get its own segmentation editor tab.
        """
        self.tabs.clear()

        for elem in results:
            editor = SegmentationEditor(
                img=elem.small_object_filtered_img,
                labeling_strategy=SkimageImgLabeling(SkimageCellConnectivity.CORNERS),
                min_size=elem.min_size,
                with_progress_bar=True,
                headers=self.headers_text,
                rollback_button_text=self.rollback_button_text,
                segmentation_button_text=self.segmentation_button_text,
                progress_title=self.progress_title,
                progress_cancel_text=self.progress_cancel_text,
                parent=self,
            )
            self.tabs.addTab(editor, f"{Path(elem.file_path).name}")

    def get_results(self) -> list[SegmentationResults]:
        """Collect segmentation results from all editor tabs.

        Iterates through all tabs and collects the segmentation results from
        each SegmentationEditor widget.

        Returns:
            list[SegmentationResults]: List containing segmentation results from
                all editor tabs. Each result includes the file path and labeled cells.
        """
        list_of_results = []

        for i in range(self.tabs.count()):
            editor = self.tabs.widget(i)
            file_path = self.tabs.tabText(i)
            if hasattr(editor, "get_results"):
                list_of_results.append(
                    SegmentationResults(file_path, editor.get_results())
                )

        return list_of_results
