from typing import Any, List
from PyQt6 import QtCore

from pycroglia.ui.widgets.imagefilters.stacks import FilterEditorStack
from pycroglia.ui.widgets.io.file_selection_editor import FileSelectionEditor
from pycroglia.ui.widgets.results.stacks import ResultsDashboardStack
from pycroglia.ui.widgets.segmentation.stacks import SegmentationEditorStack
from pycroglia.ui.widgets.analysis.stacks import CellSelectorStack
from pycroglia.ui.widgets.wizard.pages import (
    FileSelectionPage,
    FilterEditorPage,
    SegmentationEditorPage,
    CellSelectionPage,
    DashboardPage,
)

DEFAULT_CONFIG = {
    # File selection page
    "file_selection": {
        "file_headers": ["File Type", "File Path"],
        "delete_button_text": "Delete",
        "open_file_text": "Select Files:",
        "open_button_text": "Open",
        "open_dialog_title": "Open File",
        "file_filters": "All Files (*);;Image Files (*.lsm *.tiff *.tif)",
    },
    # Filter editor page
    "filter_editor": {
        "img_viewer_label": "Image Viewer",
        "read_button_text": "Load Image",
        "channels_label": "Channels:",
        "channel_of_interest_label": "Channel of Interest:",
        "gray_filter_label": "Threshold Image",
        "gray_filter_slider_label": "Threshold (pixel brightness)",
        "small_objects_filter_label": "Noise Filtered Image",
        "small_objects_threshold_label": "Min Size (pixels):",
    },
    # Segmentation editor page
    "segmentation_editor": {
        "segmentation_headers": ["Cell Number", "Cell Size (pixels)"],
        "rollback_button_text": "Roll back segmentation",
        "segmentation_button_text": "Segment Cell",
        "progress_title": "Segmenting cell...",
        "progress_cancel_text": "Cancel",
    },
    # Cell Selector
    "cell_selector": {
        "headers": ["Cell Number", "Cell Size (pixels)"],
        "remove_button_text": "Remove Cell",
        "size_label_text": "Cell Size (pixels)",
        "size_button_text": "Remove smaller than",
        "preview_button_text": "Preview",
        "border_checkbox_text": "Remove border cells",
    },
    # Results Dashboard
    "results_dashboard": {
        "summary_headers": ["Metric", "Value"],
        "cell_headers": ["Metric", "Value"],
        "graphs_text": "Select graphs:",
        "graph_button_txt": "Preview",
        "output_title_txt": "Output formats",
        "output_select_txt": "Destination folder",
        "output_button_txt": "Browse",
        "output_display_txt": "No folder selected",
        "output_dialog_title_txt": "Select a folder",
    },
    # Navigation buttons
    "navigation": {
        "back_button_text": "Back",
        "next_button_text": "Next",
    },
}


def create_wizard_pages(config: dict[str, Any]) -> List[dict[str, Any]]:
    """Create wizard page configurations based on provided config.

    Args:
        config (dict[str, Any]): Configuration dictionary containing text and UI settings
            for all wizard pages. Expected structure:
            - 'file_selection': File selection page configuration
            - 'filter_editor': Filter editor page configuration
            - 'segmentation_editor': Segmentation editor page configuration
            - 'navigation': Navigation button configuration

    Returns:
        List[dict[str, Any]]: List of page configuration dictionaries, each containing:
            - type: Page type identifier
            - widget_class: Widget class to instantiate
            - widget_args: Arguments for widget constructor
            - page_class: Page wrapper class
            - navigation: Navigation button configuration
    """
    page_configs = [
        {
            "type": "file_selection",
            "widget_class": FileSelectionEditor,
            "widget_args": {
                "headers": config["file_selection"]["file_headers"],
                "delete_button_text": config["file_selection"]["delete_button_text"],
                "open_file_text": config["file_selection"]["open_file_text"],
                "open_button_text": config["file_selection"]["open_button_text"],
                "open_dialog_title": config["file_selection"]["open_dialog_title"],
                "open_dialog_default_path": QtCore.QDir.homePath(),
                "file_filters": config["file_selection"]["file_filters"],
            },
            "page_class": FileSelectionPage,
            "navigation": {
                "show_back_btn": False,
                "show_next_btn": True,
                "next_btn_txt": config["navigation"]["next_button_text"],
            },
        },
        {
            "type": "filter_editor",
            "widget_class": FilterEditorStack,
            "widget_args": {
                "img_viewer_label": config["filter_editor"]["img_viewer_label"],
                "read_button_text": config["filter_editor"]["read_button_text"],
                "channels_label": config["filter_editor"]["channels_label"],
                "channel_of_interest_label": config["filter_editor"][
                    "channel_of_interest_label"
                ],
                "gray_filter_label": config["filter_editor"]["gray_filter_label"],
                "gray_filter_slider_label": config["filter_editor"][
                    "gray_filter_slider_label"
                ],
                "small_objects_filter_label": config["filter_editor"][
                    "small_objects_filter_label"
                ],
                "small_objects_threshold_label": config["filter_editor"][
                    "small_objects_threshold_label"
                ],
            },
            "page_class": FilterEditorPage,
            "navigation": {
                "show_back_btn": True,
                "show_next_btn": True,
                "back_btn_txt": config["navigation"]["back_button_text"],
                "next_btn_txt": config["navigation"]["next_button_text"],
            },
        },
        {
            "type": "segmentation_editor",
            "widget_class": SegmentationEditorStack,
            "widget_args": {
                "headers_text": config["segmentation_editor"]["segmentation_headers"],
                "rollback_button_text": config["segmentation_editor"][
                    "rollback_button_text"
                ],
                "segmentation_button_text": config["segmentation_editor"][
                    "segmentation_button_text"
                ],
                "progress_title": config["segmentation_editor"]["progress_title"],
                "progress_cancel_text": config["segmentation_editor"][
                    "progress_cancel_text"
                ],
            },
            "page_class": SegmentationEditorPage,
            "navigation": {
                "show_back_btn": True,
                "show_next_btn": True,
                "back_btn_txt": config["navigation"]["back_button_text"],
                "next_btn_txt": config["navigation"]["next_button_text"],
            },
        },
        {
            "type": "segmentation_editor",
            "widget_class": CellSelectorStack,
            "widget_args": {
                "headers": config["cell_selector"]["headers"],
                "remove_button_text": config["cell_selector"]["remove_button_text"],
                "size_label_text": config["cell_selector"]["size_label_text"],
                "size_button_text": config["cell_selector"]["size_button_text"],
                "preview_button_text": config["cell_selector"]["preview_button_text"],
                "border_checkbox_text": config["cell_selector"]["border_checkbox_text"],
            },
            "page_class": CellSelectionPage,
            "navigation": {
                "show_back_btn": True,
                "show_next_btn": True,
                "back_btn_txt": config["navigation"]["back_button_text"],
                "next_btn_txt": config["navigation"]["next_button_text"],
            },
        },
        {
            "type": "results_dashboard",
            "widget_class": ResultsDashboardStack,
            "widget_args": {
                "summary_headers": config["results_dashboard"]["summary_headers"],
                "cell_headers": config["results_dashboard"]["cell_headers"],
                "graphs_text": config["results_dashboard"]["graphs_text"],
                "graph_button_txt": config["results_dashboard"]["graph_button_txt"],
                "output_title_txt": config["results_dashboard"]["output_title_txt"],
                "output_select_txt": config["results_dashboard"]["output_select_txt"],
                "output_button_txt": config["results_dashboard"]["output_button_txt"],
                "output_display_txt": config["results_dashboard"]["output_display_txt"],
                "output_dialog_title_txt": config["results_dashboard"][
                    "output_dialog_title_txt"
                ],
            },
            "page_class": DashboardPage,
            "navigation": {
                "show_back_btn": True,
                "show_next_btn": False,
                "back_btn_txt": config["navigation"]["back_button_text"],
                "next_btn_txt": config["navigation"]["next_button_text"],
            },
        },
    ]

    return page_configs
