from dataclasses import dataclass
from typing import Dict, List, Optional
import numpy as np
import pandas as pd
from ..config.constants import (
    INFORMATION_GENERATION_RATE,
    COUPLING_CONSTANT,
)
import logging

logger = logging.getLogger(__name__)

@dataclass
class ValidationResult:
    """Stores validation check results."""
    passed: bool
    error: float
    message: str
    details: Dict[str, float]

class HolographicValidationSuite:
    """Comprehensive validation suite for holographic system constraints."""
    
    def __init__(self, tolerance: float = 1e-6):
        self.tolerance = tolerance
        self.validation_history: List[Dict[str, ValidationResult]] = []
        logger.info(f"Initialized validation suite with tolerance {tolerance}")
    
    def validate_state(
        self,
        matter_wavefunction: np.ndarray,
        antimatter_wavefunction: np.ndarray,
        metrics_df: pd.DataFrame,
        dt: float
    ) -> Dict[str, ValidationResult]:
        """Perform comprehensive state validation."""
        try:
            results = {}
            
            # Information bound validation
            results['information_bound'] = self._validate_information_bound(
                matter_wavefunction,
                metrics_df
            )
            
            # Energy conservation
            results['energy_conservation'] = self._validate_energy_conservation(
                metrics_df,
                dt
            )
            
            # Holographic decay
            results['holographic_decay'] = self._validate_holographic_decay(
                metrics_df,
                dt
            )
            
            # Antimatter coherence
            results['antimatter_coherence'] = self._validate_antimatter_coherence(
                antimatter_wavefunction
            )
            
            # Cross-continuum coupling
            results['coupling_strength'] = self._validate_coupling_strength(
                matter_wavefunction,
                antimatter_wavefunction
            )
            
            # Store validation results
            self.validation_history.append(results)
            
            # Log validation summary
            self._log_validation_summary(results)
            
            return results
            
        except Exception as e:
            logger.error(f"Validation failed: {str(e)}")
            raise
    
    def _validate_information_bound(
        self,
        wavefunction: np.ndarray,
        metrics_df: pd.DataFrame
    ) -> ValidationResult:
        """Validate holographic information bounds."""
        try:
            # Calculate current information content
            density = np.abs(wavefunction)**2
            information = -np.sum(density * np.log2(density + 1e-10))
            
            # Calculate maximum allowed information
            max_info = len(wavefunction)  # Holographic bound
            
            # Check bound
            error = max(0, information - max_info)
            passed = error < self.tolerance
            
            return ValidationResult(
                passed=passed,
                error=error,
                message=f"Information bound {'satisfied' if passed else 'violated'}",
                details={
                    'current_information': information,
                    'max_information': max_info,
                    'margin': max_info - information
                }
            )
            
        except Exception as e:
            logger.error(f"Information bound validation failed: {str(e)}")
            raise
    
    def _validate_energy_conservation(
        self,
        metrics_df: pd.DataFrame,
        dt: float
    ) -> ValidationResult:
        """Validate energy conservation with holographic corrections."""
        try:
            if len(metrics_df) < 2:
                return ValidationResult(
                    passed=True,
                    error=0.0,
                    message="Insufficient data for energy validation",
                    details={'time_steps': len(metrics_df)}
                )
            
            # Get energy values
            initial_energy = metrics_df['energy'].iloc[0]
            current_energy = metrics_df['energy'].iloc[-1]
            time_elapsed = metrics_df['time'].iloc[-1]
            
            # Calculate expected decay
            expected_energy = initial_energy * np.exp(-INFORMATION_GENERATION_RATE * time_elapsed)
            
            # Calculate error
            error = np.abs(current_energy - expected_energy) / initial_energy
            passed = error < self.tolerance
            
            return ValidationResult(
                passed=passed,
                error=error,
                message=f"Energy conservation {'satisfied' if passed else 'violated'}",
                details={
                    'initial_energy': initial_energy,
                    'current_energy': current_energy,
                    'expected_energy': expected_energy,
                    'relative_error': error
                }
            )
            
        except Exception as e:
            logger.error(f"Energy conservation validation failed: {str(e)}")
            raise
    
    def _validate_holographic_decay(
        self,
        metrics_df: pd.DataFrame,
        dt: float
    ) -> ValidationResult:
        """Validate holographic decay rates."""
        try:
            if len(metrics_df) < 2:
                return ValidationResult(
                    passed=True,
                    error=0.0,
                    message="Insufficient data for decay validation",
                    details={'time_steps': len(metrics_df)}
                )
            
            # Calculate coherence decay rate
            coherence = metrics_df['coherence'].values
            times = metrics_df['time'].values
            
            # Fit exponential decay
            log_coherence = np.log(coherence + 1e-10)
            decay_rate = np.polyfit(times, log_coherence, 1)[0]
            
            # Compare with expected rate
            error = np.abs(decay_rate + INFORMATION_GENERATION_RATE) / INFORMATION_GENERATION_RATE
            passed = error < self.tolerance
            
            return ValidationResult(
                passed=passed,
                error=error,
                message=f"Holographic decay {'satisfied' if passed else 'violated'}",
                details={
                    'measured_rate': -decay_rate,
                    'expected_rate': INFORMATION_GENERATION_RATE,
                    'relative_error': error
                }
            )
            
        except Exception as e:
            logger.error(f"Holographic decay validation failed: {str(e)}")
            raise
    
    def _validate_antimatter_coherence(
        self,
        antimatter_wavefunction: np.ndarray
    ) -> ValidationResult:
        """Validate antimatter coherence preservation."""
        try:
            # Calculate norm
            norm = np.sum(np.abs(antimatter_wavefunction)**2)
            error = np.abs(norm - 1.0)
            passed = error < self.tolerance
            
            return ValidationResult(
                passed=passed,
                error=error,
                message=f"Antimatter coherence {'preserved' if passed else 'violated'}",
                details={
                    'norm': norm,
                    'deviation': error
                }
            )
            
        except Exception as e:
            logger.error(f"Antimatter coherence validation failed: {str(e)}")
            raise
    
    def _validate_coupling_strength(
        self,
        matter_wavefunction: np.ndarray,
        antimatter_wavefunction: np.ndarray
    ) -> ValidationResult:
        """Validate cross-continuum coupling strength."""
        try:
            # Calculate coupling
            coupling = np.abs(np.vdot(matter_wavefunction, antimatter_wavefunction))
            
            # Check coupling bounds
            error = max(0, coupling - COUPLING_CONSTANT)
            passed = error < self.tolerance
            
            return ValidationResult(
                passed=passed,
                error=error,
                message=f"Coupling strength {'valid' if passed else 'excessive'}",
                details={
                    'coupling': coupling,
                    'max_allowed': COUPLING_CONSTANT,
                    'margin': COUPLING_CONSTANT - coupling
                }
            )
            
        except Exception as e:
            logger.error(f"Coupling strength validation failed: {str(e)}")
            raise
    
    def _log_validation_summary(self, results: Dict[str, ValidationResult]) -> None:
        """Log summary of validation results."""
        failed_checks = [
            name for name, result in results.items() 
            if not result.passed
        ]
        
        if failed_checks:
            logger.warning(
                f"Validation failed for: {', '.join(failed_checks)}"
            )
        else:
            logger.info("All validation checks passed")