import { Cell } from '@jupyterlab/cells';
import { Notebook, NotebookActions } from '@jupyterlab/notebook';
import { NotebookTools } from './NotebookTools';
import { AppStateService } from '../AppState';

/**
 * Class providing cell manipulation tools for notebooks
 */
export class NotebookCellTools {
  private notebookTools: NotebookTools;

  /**
   * Create a new NotebookCellTools instance
   * @param notebookTools The parent NotebookTools instance
   */
  constructor(notebookTools: NotebookTools) {
    this.notebookTools = notebookTools;
  }

  /**
   * Runs a code cell in the specified notebook and returns its formatted output.
   *
   * @param options Configuration options
   * @param options.cell_id Unique identifier of the code cell to execute.
   * @param options.notebook_path Path to the notebook file (optional).
   * @param options.kernel_id Specific kernel ID to use (optional).
   * @param options.is_tracking_id Whether the cell_id is a tracking ID (optional).
   * @returns Formatted output strings from the cell's execution.
   */
  async run_cell(options: {
    cell_id: string;
    notebook_path?: string | null;
    kernel_id?: string | null;
  }): Promise<(string | { error: true; errorText: string })[]> {
    // Find cell by tracking ID or model ID
    let cellInfo = null;
    // Use options.notebook_path directly without reassigning to notebookPath
    console.log(
      'RUNNING CELL - CELL ID:',
      options.cell_id,
      'NOTEBOOK PATH:',
      options.notebook_path
    );

    cellInfo = this.notebookTools.findCellByAnyId(
      options.cell_id,
      options.notebook_path
    );
    console.log('Cell info:', cellInfo);

    if (!cellInfo) {
      throw new Error(
        `Cell with ID ${options.cell_id} not found in notebook ${options.notebook_path || 'current'}`
      );
    }

    const current = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    );
    if (!current) {
      throw new Error(
        `Notebook ${options.notebook_path || 'current'} not found`
      );
    }

    const { notebook } = current;
    const { cell } = cellInfo;

    // Verify that it's a code cell
    if (cell.model.type !== 'code') {
      throw new Error(`Cell with ID ${options.cell_id} is not a code cell`);
    }

    // Activate the cell to run
    this.notebookTools.activateCell(cell);

    // Run the cell - pass the actual cell to run instead of using _findCellById
    await NotebookActions.runCells(
      notebook,
      [cell],
      current.widget?.sessionContext
    );

    // Process outputs in a formatted way
    const outputs: (string | { error: true; errorText: string })[] = [];
    try {
      const codeCell = cell as any; // Code cell needs to be cast to appropriate type
      if (codeCell.outputArea && codeCell.outputArea.model.toJSON) {
        const outputsJson = codeCell.outputArea.model.toJSON();
        for (const output of outputsJson) {
          if (output.output_type === 'stream') {
            const getText = () => {
              // Join the text array into a single string if it's an array
              if (Array.isArray(output.text)) {
                return output.text.join('');
              } else {
                return output.text;
              }
            };

            if (output.name === 'stderr') {
              outputs.push({ error: true, errorText: getText() });
              continue;
            }

            outputs.push(getText());
          } else if (
            output.output_type === 'execute_result' ||
            output.output_type === 'display_data'
          ) {
            if (output.data && output.data['text/plain']) {
              // Handle 'text/plain' data which can be string or array
              if (Array.isArray(output.data['text/plain'])) {
                outputs.push(output.data['text/plain'].join(''));
              } else {
                outputs.push(output.data['text/plain']);
              }
            }
          } else if (output.output_type === 'error') {
            // Format error messages with traceback
            outputs.push({
              error: true,
              errorText: `Error: ${output.ename}: ${output.evalue}`
            });
            if (Array.isArray(output.traceback)) {
              // Join traceback items into a single formatted string
              outputs.push({
                error: true,
                errorText: output.traceback.join('\n')
              });
            } else if (output.traceback) {
              outputs.push({ error: true, errorText: output.traceback });
            }
          }
        }
      }
    } catch (error) {
      console.error('Error collecting cell outputs:', error);
      outputs.push(`Error collecting outputs: ${error}`);
    }

    return outputs;
  }

  /**
   * Adds a new cell (code, markdown, or raw) to the specified notebook at an optional position, returning its unique ID.
   *
   * @param options Configuration options
   * @param options.cell_type Type of cell ("code", "markdown", or "raw").
   * @param options.source Initial source content for the cell.
   * @param options.summary A summary of the cell's content.
   * @param options.notebook_path Path to the notebook file (optional).
   * @param options.position 0-based index for insertion (appends if null).
   * @param options.show_diff Whether to show diff view (defaults to false)
   * @param options.tracking_id Optional tracking ID to reuse
   * @returns The unique ID of the newly created cell
   */
  add_cell(options: {
    cell_type: string;
    source: string;
    summary: string;
    notebook_path?: string | null;
    position?: number | null;
    show_diff?: boolean;
    tracking_id?: string; // Optional tracking ID to reuse
  }): string {
    // Get the notebook - pass options.notebook_path directly
    const notebook = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    )?.notebook;
    if (!notebook) {
      throw new Error(
        `Notebook ${options.notebook_path || 'current'} not found`
      );
    }

    // Normalize the content - ensure we have valid strings
    const content = options.source || '';

    // Determine position for insertion
    let position: number;
    if (options.position !== null && options.position !== undefined) {
      position = options.position;
      // Ensure position is within valid range
      position = Math.min(Math.max(position, 0), notebook.widgets.length);
    } else {
      position = notebook.widgets.length; // Default to end of notebook
    }

    // Handle insertion based on position
    if (position === 0) {
      // Insert at the beginning of notebook
      this.notebookTools.activateCellByIndex(position);
      NotebookActions.insertAbove(notebook);
    } else {
      // Activate the cell above where we want to insert
      this.notebookTools.activateCellByIndex(position - 1);
      // Insert below the selected cell
      NotebookActions.insertBelow(notebook);
    }

    let newCell = notebook.activeCell;
    if (!newCell) {
      throw new Error('Failed to create new cell');
    }

    // Set the cell type to the requested type
    if (options.cell_type === 'markdown' && newCell.model.type !== 'markdown') {
      NotebookActions.changeCellType(notebook, 'markdown');
    } else if (options.cell_type === 'code' && newCell.model.type !== 'code') {
      NotebookActions.changeCellType(notebook, 'code');
    }

    newCell = notebook.activeCell;
    if (!newCell) {
      throw new Error('Failed to create new cell');
    }

    // Add tracking metadata to cell
    const metadata: any = newCell.model.sharedModel.getMetadata() || {};
    const now = new Date().toISOString();
    metadata.cell_tracker = {
      trackingId: options.tracking_id || this.generateTrackingId(notebook),
      createdAt: now,
      lastModified: now,
      origin: 'ai',
      summary: options.summary || ''
    };

    // Add custom metadata with summary
    if (!metadata.custom) {
      metadata.custom = {};
    }
    metadata.custom.summary = options.summary;
    newCell.model.sharedModel.setMetadata(metadata);

    // Only display diff if explicitly requested, otherwise set content directly
    if (options.show_diff) {
      this.notebookTools.display_diff(newCell, '', content, 'add');
    } else {
      // Set the cell content directly
      newCell.model.sharedModel.setSource(content);
    }

    // Return the tracking ID instead of cell.model.id since it's more stable
    const trackingMetadata: any =
      newCell.model.sharedModel.getMetadata().cell_tracker;

    return trackingMetadata.trackingId;
  }

  /**
   * Check if a cell is a plan cell
   */
  public isPlanCell(cell: Cell): boolean {
    const metadata: any = cell?.model?.sharedModel?.getMetadata() || {};
    return metadata.custom?.sage_cell_type === 'plan';
  }

  /**
   * Ensure the first cell of a notebook is a plan cell
   */
  public setFirstCellAsPlan(notebookPath?: string | null): void {
    let current = this.notebookTools.getCurrentNotebook(notebookPath);
    if (!current) return;

    let { notebook } = current;

    const planCell = this.findPlanCell(notebook);
    if (planCell) return;

    let firstCell = notebook.widgets[0];
    if (!firstCell) {
      console.error('Could not find first cell to ensure it is a plan cell');
      return;
    }

    NotebookActions.changeCellType(notebook, 'markdown');

    current = this.notebookTools.getCurrentNotebook(notebookPath);
    if (current && current.notebook) notebook = current.notebook;
    firstCell = notebook.widgets[0]; // Re-fetch after changing type

    const metadata: any = firstCell.model.sharedModel.getMetadata() || {};
    if (!metadata.custom) {
      metadata.custom = {};
    }

    if (metadata && metadata.cell_tracker && metadata.cell_tracker.trackingId)
      metadata.cell_tracker.trackingId = 'planning_cell';

    metadata.custom.sage_cell_type = 'plan';
    firstCell.model.sharedModel.setMetadata(metadata);
    firstCell.model.sharedModel.setSource('');
  }

  /**
   * Find the plan cell in the notebook
   * @param notebookPath Optional notebook path
   * @returns The plan cell or null if not found
   */
  public findPlanCell(notebook: Notebook): Cell | null {
    return notebook.widgets.find(cell => this.isPlanCell(cell)) || null;
  }

  // Generate a unique tracking ID (numerical)
  private generateTrackingId(notebook: any): string {
    // Find the max cell number in use
    let maxNum = 1;
    const cells = notebook.widgets;
    for (let i = 0; i < cells.length; i++) {
      const metadata: any = cells[i].model.sharedModel.getMetadata() || {};
      const trackingId = metadata.cell_tracker?.trackingId;
      if (trackingId && /^cell_(\d+)$/.test(trackingId)) {
        const num = parseInt(trackingId.split('_')[1], 10);
        if (num > maxNum) maxNum = num;
      }
    }
    return `cell_${maxNum}`;
  }

  /**
   * Removes a list of cells from the specified notebook using their IDs.
   *
   * @param options Configuration options
   * @param options.cell_ids A list of unique identifiers of the cells to remove.
   * @param options.notebook_path Path to the notebook file (optional).
   * @returns True if at least one cell was found and removed, False otherwise.
   */
  remove_cells(options: {
    cell_ids: string[];
    notebook_path?: string | null;
    remove_from_notebook?: boolean;
  }): boolean {
    // Determine if the IDs are tracking IDs
    const useTrackingIds = options.cell_ids.every(id => id.startsWith('cell_'));

    // Use options.notebook_path directly
    const current = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    );
    if (!current) {
      return false;
    }

    const { notebook } = current;
    let cellsRemoved = 0;

    // Process each cell ID in the list
    for (const cellId of options.cell_ids) {
      let cellInfo = this.notebookTools.findCellByAnyId(
        cellId,
        options.notebook_path
      );

      if (cellInfo) {
        // Check if this cell is non-deletable
        const metadata: any =
          cellInfo.cell.model.sharedModel.getMetadata() || {};
        const isDeletable = metadata.custom?.deletable !== false;

        if (!isDeletable) {
          console.warn(`Cannot delete non-deletable cell: ${cellId}`);
          continue;
        }

        // Get the cell content before removal to show in diff
        const oldContent = cellInfo.cell.model.sharedModel.getSource();

        // Preserve tracking ID for future reference
        // Type-safe access with explicit cast for cell_tracker and its properties
        const cellTracker = metadata.cell_tracker as
          | { trackingId?: string }
          | undefined;
        const trackingId = cellTracker?.trackingId;

        // Log the diff for cell removal (comparing content to empty)
        if (trackingId) {
          this.notebookTools['diffTools'].logDiff(
            this.notebookTools.normalizeContent,
            oldContent,
            '',
            'remove',
            trackingId // Use tracking ID instead of model ID
          );
        } else {
          this.notebookTools['diffTools'].logDiff(
            this.notebookTools.normalizeContent,
            oldContent,
            '',
            'remove',
            cellId
          );
        }

        if (options.remove_from_notebook) {
          // Activate the cell to remove
          this.notebookTools.activateCell(cellInfo.cell);

          // Delete the cell
          NotebookActions.deleteCells(notebook);
          console.log(
            `[NotebookCellTools] Cell ${trackingId} removed permanently from the notebook`
          );
        }

        cellsRemoved++;
      }
    }

    return cellsRemoved > 0;
  }

  /**
   * Modifies the source content of an existing cell.
   *
   * @param options Configuration options
   * @param options.cell_id Unique identifier of the cell to edit.
   * @param options.new_source New source content for the cell.
   * @param options.summary A summary of the cell's new content.
   * @param options.notebook_path Path to the notebook file (optional).
   * @param options.show_diff Whether to show diff view (defaults to false)
   * @param options.is_tracking_id Whether the cell_id is a tracking ID (optional).
   * @returns True if the cell was found and updated, False otherwise.
   */
  edit_cell(options: {
    cell_id: string;
    new_source: string;
    summary: string;
    notebook_path?: string | null;
    show_diff?: boolean;
  }): boolean {
    // Find cell by tracking ID or model ID - use options.notebook_path directly
    let cellInfo = null;

    cellInfo = this.notebookTools.findCellByAnyId(
      options.cell_id,
      options.notebook_path
    );

    if (!cellInfo) {
      console.error(
        `Cell not found with ID: ${options.cell_id} in notebook ${options.notebook_path || 'current'}`
      );
      return false;
    }

    const current = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    );
    if (!current) {
      return false;
    }

    const { notebook } = current;
    const { cell } = cellInfo;

    // Check if this cell is non-editable
    const metadata: any = cell.model.sharedModel.getMetadata() || {};
    const isEditable = metadata.custom?.editable !== false;

    if (!isEditable) {
      console.warn(`Cannot edit non-editable cell: ${options.cell_id}`);
      return false;
    }

    // Get the old content to compare
    const oldContent = cell.model.sharedModel.getSource();

    // Get tracking ID for consistent reference
    const trackingId =
      metadata.cell_tracker?.trackingId || this.generateTrackingId(notebook);

    // Log the diff between old and new content using tracking ID
    this.notebookTools['diffTools'].logDiff(
      this.notebookTools.normalizeContent,
      oldContent,
      options.new_source,
      'edit',
      trackingId // Use tracking ID instead of model ID
    );

    // Activate the cell to edit
    this.notebookTools.activateCell(cell);

    // Update metadata - last modified time and summary
    if (metadata.cell_tracker) {
      metadata.cell_tracker.lastModified = new Date().toISOString();
      metadata.cell_tracker.summary =
        options.summary || metadata.cell_tracker.summary;
    } else {
      // Create tracking metadata if it doesn't exist
      const now = new Date().toISOString();
      metadata.cell_tracker = {
        trackingId: trackingId,
        createdAt: now,
        lastModified: now,
        origin: 'ai', // Assume AI is editing
        summary: options.summary || ''
      };
    }

    // Update custom summary too
    if (!metadata.custom) {
      metadata.custom = {};
    }
    metadata.custom.summary = options.summary;
    cell.model.sharedModel.setMetadata(metadata);

    // Only display diff if explicitly requested, otherwise set content directly
    if (options.show_diff) {
      this.notebookTools.display_diff(
        cell,
        oldContent,
        options.new_source,
        'edit'
      );
    } else {
      // Set the new content directly
      cell.model.sharedModel.setSource(options.new_source);
    }

    return true;
  }

  stream_edit_plan(options: {
    partial_plan: string;
    notebook_path?: string | null;
  }): boolean {
    const current = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    );
    if (!current) {
      return false;
    }

    const { notebook } = current;

    // Get the first cell (plan cell)
    let planCell = null;

    for (let i = 0; i < notebook.widgets.length; i++) {
      const cell = notebook.widgets[i];
      if (this.isPlanCell(cell)) {
        planCell = cell;
      }
    }

    if (!planCell) {
      console.error('No first cell found for plan update');
      notebook.activeCellIndex = 0;
      NotebookActions.insertAbove(notebook);
      notebook.activeCellIndex = 0;
      this.setFirstCellAsPlan(options.notebook_path);
    }
    const firstCell = notebook.widgets[0];

    const oldContent = firstCell.model.sharedModel.getSource();
    const newContent = options.partial_plan;

    // For edit_cell, we need to properly handle the diff calculation
    // The streaming code might be partial, so we need to preserve the rest of the original content
    let finalContent = oldContent;

    if (oldContent && oldContent.length < newContent.length) {
      // If the streaming code is shorter than original, preserve the remaining content
      finalContent = oldContent + newContent.substring(oldContent.length);
    } else if (oldContent && oldContent.length > newContent.length) {
      // If streaming code is longer than original, use the new code as-is
      finalContent = newContent;
    }
    // If lengths are equal, use the new code as-is

    firstCell.model.sharedModel.setSource(finalContent);

    return true;
  }

  /**
   * Modifies the plan of the notebook by updating the first cell and optionally adding current/next step cells.
   *
   * @param options Configuration options
   * @param options.updated_plan_string Updated plan string in markdown format
   * @param options.current_step_string Current step string (optional)
   * @param options.next_step_string Next step string (optional)
   * @param options.notebook_path Path to the notebook file (optional)
   * @returns True if the plan was updated successfully, False otherwise
   */
  edit_plan(options: {
    updated_plan_string: string;
    current_step_string?: string;
    next_step_string?: string;
    notebook_path?: string | null;
  }): boolean {
    const current = this.notebookTools.getCurrentNotebook(
      options.notebook_path
    );
    if (!current) {
      console.error('No notebook found for edit_plan');
      return false;
    }

    const { notebook } = current;

    // Get the first cell (plan cell)
    let planCell = null;

    for (let i = 0; i < notebook.widgets.length; i++) {
      const cell = notebook.widgets[i];
      if (this.isPlanCell(cell)) {
        planCell = cell;
      }
    }

    if (!planCell) {
      console.error('No first cell found for plan update');
      notebook.activeCellIndex = 0;
      NotebookActions.insertAbove(notebook);
      notebook.activeCellIndex = 0;
      this.setFirstCellAsPlan(options.notebook_path);
    }

    const firstCell = notebook.widgets[0];

    // Update the plan cell content
    const oldContent = firstCell.model.sharedModel.getSource();
    const newContent = options.updated_plan_string;

    // Get tracking ID for consistent reference
    const metadata: any = firstCell.model.sharedModel.getMetadata() || {};
    const trackingId = metadata.cell_tracker?.trackingId || 'planning_cell';

    // Log the diff between old and new content
    this.notebookTools['diffTools'].logDiff(
      this.notebookTools.normalizeContent,
      oldContent,
      newContent,
      'edit_plan',
      trackingId
    );

    // Activate the plan cell
    this.notebookTools.activateCell(firstCell);

    // Update metadata
    if (metadata.cell_tracker) {
      metadata.cell_tracker.lastModified = new Date().toISOString();
      metadata.cell_tracker.summary = 'Updated plan';
    } else {
      const now = new Date().toISOString();
      metadata.cell_tracker = {
        trackingId: trackingId,
        createdAt: now,
        lastModified: now,
        origin: 'ai',
        summary: 'Updated plan'
      };
    }

    if (!metadata.custom) {
      metadata.custom = {};
    }
    metadata.custom.summary = 'Updated plan';
    metadata.custom.current_step_string = options.current_step_string;
    metadata.custom.next_step_string = options.next_step_string;
    firstCell.model.sharedModel.setMetadata(metadata);

    // Set the new plan content
    firstCell.model.sharedModel.setSource(newContent);
    void NotebookActions.runCells(notebook, [firstCell]);

    void AppStateService.getPlanStateDisplay().updatePlan(
      options.current_step_string,
      options.next_step_string,
      options.updated_plan_string
    );

    console.log('Plan cell updated successfully');
    return true;
  }

  /**
   * Get information about all cells in the current notebook
   * @returns Array of cell info objects or null if no notebook
   */
  get_cells_info(notebookPath?: string | null): {
    cells: Array<{ id: string; type: string; content: string }>;
  } | null {
    const current = this.notebookTools.getCurrentNotebook(notebookPath);
    if (!current) return null;

    const { notebook } = current;
    const cells: Array<{ id: string; type: string; content: string }> = [];

    for (let i = 0; i < notebook.widgets.length; i++) {
      const cell = notebook.widgets[i];
      cells.push({
        id: cell.model.id,
        type: cell.model.type,
        content: cell.model.sharedModel.getSource()
      });
    }

    return { cells };
  }

  /**
   * Get detailed information about a specific cell
   * @param options Configuration options
   * @param options.cell_id Unique identifier of the cell
   * @returns Detailed information about the cell or null if not found
   */
  get_cell_info(options: {
    cell_id: string;
    notebook_path?: string | null;
  }): any {
    let cellInfo = this.notebookTools.findCellByAnyId(
      options.cell_id,
      options.notebook_path
    );
    if (!cellInfo) {
      return null;
    }

    const { cell, index } = cellInfo;

    // Get the cell metadata
    const metadata = cell.model.sharedModel.getMetadata();

    // Return detailed information about the cell
    return {
      id: cell.model.id,
      type: cell.model.type,
      content: cell.model.sharedModel.getSource(),
      index: index,
      custom: metadata?.custom || {},
      trackingId: (metadata?.cell_tracker as any)?.trackingId || null
    };
  }

  /**
   * This is just a placeholder method to match the localToolMap.
   * The actual edit_history functionality is handled on the server side.
   *
   * @param options Configuration options
   * @param options.limit Maximum number of history entries to return
   * @returns An empty array as this is just forwarded to the server
   */
  edit_history(options: { limit?: number }): any {
    console.log('Edit history requested with options:', options);
    // This is just a pass-through function - edit_history is handled server-side
    // The ToolService will forward this to the MCP server
    return { forwarded: true };
  }

  /**
   * Read all cells from the notebook with comprehensive information and metadata
   * @param options Configuration options
   * @param options.notebook_path Path to the notebook file (optional)
   * @param options.include_outputs Whether to include cell outputs (optional, default: true)
   * @param options.include_metadata Whether to include cell metadata (optional, default: true)
   * @returns Array of comprehensive cell information or null if no notebook
   */
  read_cells(
    options: {
      notebook_path?: string | null;
      include_outputs?: boolean;
      include_metadata?: boolean;
    } = {}
  ): {
    cells: Array<{
      id: string;
      index: number;
      type: string;
      content: string;
      trackingId?: string;
      metadata?: any;
      outputs?: any[];
      execution_count?: number;
    }>;
    notebook_path?: string;
    total_cells: number;
  } | null {
    const {
      notebook_path = null,
      include_outputs = true,
      include_metadata = true
    } = options;

    const current = this.notebookTools.getCurrentNotebook(notebook_path);
    if (!current) {
      console.log('No current notebook found');
      return null;
    }

    const { notebook, widget } = current;
    const cells: Array<{
      id: string;
      index: number;
      type: string;
      content: string;
      trackingId?: string;
      metadata?: any;
      outputs?: any[];
      execution_count?: number;
    }> = [];

    console.log(`Reading ${notebook.widgets.length} cells from notebook`);

    for (let i = 0; i < notebook.widgets.length; i++) {
      const cell = notebook.widgets[i];
      const cellModel = cell.model;
      const sharedModel = cellModel.sharedModel;

      // Basic cell information
      const cellInfo: {
        id: string;
        index: number;
        type: string;
        content: string;
        trackingId?: string;
        metadata?: any;
        outputs?: any[];
        execution_count?: number;
      } = {
        id: '',
        index: i,
        type: cellModel.type,
        content: sharedModel.getSource()
      };

      // Add metadata if requested
      if (include_metadata) {
        const metadata = sharedModel.getMetadata();
        cellInfo.metadata = metadata || {};

        // Extract tracking ID if available
        const cellTracker = metadata?.cell_tracker as any;
        if (cellTracker?.trackingId) {
          cellInfo.trackingId = cellTracker.trackingId;
        }
      }

      // Add outputs and execution count for code cells if requested
      if (include_outputs && cellModel.type === 'code') {
        try {
          const codeCell = cell as any;
          if (codeCell.outputArea && codeCell.outputArea.model.toJSON) {
            cellInfo.outputs = codeCell.outputArea.model.toJSON();
          }

          // Get execution count
          const executionCount = (cellModel as any).executionCount;
          if (executionCount !== null && executionCount !== undefined) {
            cellInfo.execution_count = executionCount;
          }
        } catch (error) {
          console.warn(`Failed to get outputs for cell ${cellInfo.id}:`, error);
          cellInfo.outputs = [];
        }
      }

      if (cellInfo.outputs && cellInfo.outputs.length > 0) {
        for (const output of cellInfo.outputs) {
          if (output.text && output.text.length > 5000) {
            output.text = output.text.slice(0, 5000);
          }

          if (output.output_type === 'display_data' && output.data) {
            output.data = output.data['text/plain'];
          }
        }
      }

      cells.push(cellInfo);
    }

    const result = {
      cells,
      total_cells: cells.length,
      notebook_path: widget.context.path
    };

    console.log(`Successfully read ${cells.length} cells from notebook`);
    console.log(result);
    return result;
  }
}
