"""
QuerySUTRA v0.2.2 - FIXED MULTI-TABLE CREATION
SUTRA: Structured-Unstructured-Text-Retrieval-Architecture

FIXED: Properly creates multiple entity tables from ANY data
Author: Aditya Batta
License: MIT
Version: 0.2.2
"""

__version__ = "0.2.2"
__author__ = "Aditya Batta"
__title__ = "QuerySUTRA: Structured-Unstructured-Text-Retrieval-Architecture"
__all__ = ["SUTRA", "QueryResult", "quick_start"]

import os
import sqlite3
import pandas as pd
import numpy as np
from typing import Optional, Union, Dict, Any, List
from pathlib import Path
import json
import hashlib
import warnings
import shutil
import datetime
import re
from io import StringIO
warnings.filterwarnings('ignore')

try:
    from openai import OpenAI
    HAS_OPENAI = True
except ImportError:
    HAS_OPENAI = False

try:
    import plotly.express as px
    import plotly.graph_objects as go
    HAS_PLOTLY = True
except ImportError:
    HAS_PLOTLY = False

try:
    import matplotlib.pyplot as plt
    HAS_MATPLOTLIB = True
except ImportError:
    HAS_MATPLOTLIB = False

try:
    import PyPDF2
    HAS_PYPDF2 = True
except ImportError:
    HAS_PYPDF2 = False

try:
    import docx
    HAS_DOCX = True
except ImportError:
    HAS_DOCX = False


class SUTRA:
    """
    SUTRA: Structured-Unstructured-Text-Retrieval-Architecture
    
    Creates multiple structured tables from ANY data source
    """
    
    def __init__(self, api_key: Optional[str] = None, db: str = "sutra.db"):
        print("🚀 Initializing QuerySUTRA v0.2.2 - MULTI-TABLE MODE")
        print("   SUTRA: Structured-Unstructured-Text-Retrieval-Architecture")
        
        if api_key:
            os.environ["OPENAI_API_KEY"] = api_key
        
        self.api_key = os.getenv("OPENAI_API_KEY")
        self.client = OpenAI(api_key=self.api_key) if self.api_key and HAS_OPENAI else None
        
        self.db_path = db
        self.conn = sqlite3.connect(db, check_same_thread=False)
        self.cursor = self.conn.cursor()
        
        self.current_table = None
        self.schema_info = {}
        self.cache = {}
        
        print(f"✅ Ready! Database: {db}")
        if not self.api_key:
            print("⚠️  No API key - smart parsing disabled, use .sql() for direct queries")
    
    # ========================================================================
    # UPLOAD - CREATES MULTIPLE TABLES
    # ========================================================================
    
    def upload(self, data: Union[str, pd.DataFrame], name: Optional[str] = None) -> 'SUTRA':
        """Upload data and create structured tables with AI."""
        print(f"\n📤 Uploading data...")
        
        if isinstance(data, pd.DataFrame):
            name = name or "data"
            self._store_dataframe(data, name)
            return self
        
        path = Path(data)
        if not path.exists():
            raise FileNotFoundError(f"File not found: {data}")
        
        name = name or path.stem.replace(" ", "_").replace("-", "_")
        ext = path.suffix.lower()
        
        print(f"   📄 File: {path.name}")
        
        # Load based on format
        if ext == ".csv":
            df = pd.read_csv(path)
            self._store_dataframe(df, name)
        
        elif ext in [".xlsx", ".xls"]:
            df = pd.read_excel(path)
            self._store_dataframe(df, name)
        
        elif ext == ".json":
            df = pd.read_json(path)
            self._store_dataframe(df, name)
        
        elif ext == ".sql":
            with open(path) as f:
                self.cursor.executescript(f.read())
            self.conn.commit()
            self._refresh_schema()
            print(f"✅ SQL executed!")
        
        elif ext == ".pdf":
            self._smart_upload_pdf(path, name)
        
        elif ext == ".docx":
            self._smart_upload_docx(path, name)
        
        elif ext == ".txt":
            self._smart_upload_txt(path, name)
        
        else:
            raise ValueError(f"Unsupported format: {ext}")
        
        return self
    
    # ========================================================================
    # SMART PARSING - CREATES MULTIPLE TABLES
    # ========================================================================
    
    def _smart_upload_pdf(self, path: Path, base_name: str):
        """Parse PDF and create multiple tables."""
        if not HAS_PYPDF2:
            raise ImportError("PyPDF2 not installed. Run: pip install PyPDF2")
        
        print("   📑 Extracting text from PDF...")
        
        with open(path, 'rb') as file:
            pdf_reader = PyPDF2.PdfReader(file)
            text = ""
            for page_num, page in enumerate(pdf_reader.pages, 1):
                text += page.extract_text() + "\n"
                print(f"      Extracted page {page_num}/{len(pdf_reader.pages)}")
        
        if self.client:
            print("   🧠 AI: Analyzing and extracting structured entities...")
            tables = self._create_tables_with_ai(text, base_name)
            
            if tables and len(tables) > 0:
                print(f"\n✅ Created {len(tables)} structured tables:")
                for tbl_name in tables:
                    count = pd.read_sql_query(f"SELECT COUNT(*) FROM {tbl_name}", self.conn).iloc[0, 0]
                    cols = len(self.schema_info.get(tbl_name, {}))
                    print(f"   📊 {tbl_name}: {count} rows, {cols} columns")
                return
        
        # Fallback: simple text table
        print("   ⚠️  AI not available, creating simple text table")
        df = self._parse_text_simple(text)
        self._store_dataframe(df, base_name)
    
    def _smart_upload_docx(self, path: Path, base_name: str):
        """Parse DOCX and create multiple tables."""
        if not HAS_DOCX:
            raise ImportError("python-docx not installed. Run: pip install python-docx")
        
        print("   📄 Extracting content from DOCX...")
        
        doc = docx.Document(path)
        
        # Check for tables first
        if doc.tables:
            print(f"   📊 Found {len(doc.tables)} table(s)")
            for i, table in enumerate(doc.tables):
                data = [[cell.text.strip() for cell in row.cells] for row in table.rows]
                if data and len(data) > 1:
                    df = pd.DataFrame(data[1:], columns=data[0])
                    table_name = f"{base_name}_table_{i+1}" if len(doc.tables) > 1 else base_name
                    self._store_dataframe(df, table_name)
            return
        
        # Extract text
        text = "\n".join([para.text for para in doc.paragraphs])
        
        if self.client:
            print("   🧠 AI: Analyzing and extracting structured entities...")
            tables = self._create_tables_with_ai(text, base_name)
            
            if tables and len(tables) > 0:
                print(f"\n✅ Created {len(tables)} structured tables:")
                for tbl_name in tables:
                    count = pd.read_sql_query(f"SELECT COUNT(*) FROM {tbl_name}", self.conn).iloc[0, 0]
                    cols = len(self.schema_info.get(tbl_name, {}))
                    print(f"   📊 {tbl_name}: {count} rows, {cols} columns")
                return
        
        df = self._parse_text_simple(text)
        self._store_dataframe(df, base_name)
    
    def _smart_upload_txt(self, path: Path, base_name: str):
        """Parse TXT and create multiple tables."""
        print("   📝 Reading TXT file...")
        
        with open(path, 'r', encoding='utf-8') as file:
            text = file.read()
        
        if self.client:
            print("   🧠 AI: Analyzing and extracting structured entities...")
            tables = self._create_tables_with_ai(text, base_name)
            
            if tables and len(tables) > 0:
                print(f"\n✅ Created {len(tables)} structured tables:")
                for tbl_name in tables:
                    count = pd.read_sql_query(f"SELECT COUNT(*) FROM {tbl_name}", self.conn).iloc[0, 0]
                    cols = len(self.schema_info.get(tbl_name, {}))
                    print(f"   📊 {tbl_name}: {count} rows, {cols} columns")
                return
        
        df = self._parse_text_simple(text)
        self._store_dataframe(df, base_name)
    
    def _create_tables_with_ai(self, text: str, base_name: str) -> List[str]:
        """Use AI to extract entities and create multiple tables."""
        if not self.client:
            return []
        
        try:
            # Step 1: Ask AI to extract ALL data as structured JSON
            extraction_prompt = f"""Extract ALL structured data from this text into separate entity tables.

Text:
{text}

Create these types of tables (if data exists):
1. people - (id, name, address, city, state, zip, email, phone)
2. contacts - (id, person_id, contact_type, value) 
3. events - (id, host_id, description, location, city)
4. organizations - (id, name, address, city)
5. Any other entities you find

Return a JSON object with this EXACT structure:
{{
  "people": [
    {{"id": 1, "name": "John Doe", "address": "123 Main St", "city": "Dallas", "email": "john@email.com", "phone": "555-1234"}},
    ...
  ],
  "contacts": [
    {{"id": 1, "person_id": 1, "email": "john@email.com", "phone": "555-1234"}},
    ...
  ],
  "events": [
    {{"id": 1, "host_id": 1, "description": "Team meeting", "city": "Dallas"}},
    ...
  ]
}}

IMPORTANT:
- Extract ALL people, contacts, events, organizations you find
- Use consistent column names
- Return ONLY valid JSON, no explanations
- If a table type has no data, omit it from JSON"""

            response = self.client.chat.completions.create(
                model="gpt-4o-mini",
                messages=[
                    {"role": "system", "content": "You are a data extraction expert. Extract ALL entities from text into structured JSON tables. Return only valid JSON."},
                    {"role": "user", "content": extraction_prompt}
                ],
                temperature=0
            )
            
            json_text = response.choices[0].message.content.strip()
            json_text = json_text.replace("```json", "").replace("```", "").strip()
            
            extracted_data = json.loads(json_text)
            
            created_tables = []
            
            # Create tables from extracted data
            for entity_type, records in extracted_data.items():
                if records and isinstance(records, list) and len(records) > 0:
                    table_name = f"{base_name}_{entity_type}"
                    
                    try:
                        df = pd.DataFrame(records)
                        if not df.empty:
                            self._store_dataframe(df, table_name, silent=True)
                            created_tables.append(table_name)
                            print(f"      ✅ {entity_type}: {len(df)} records")
                    except Exception as e:
                        print(f"      ⚠️  Failed to create {entity_type}: {e}")
            
            return created_tables
        
        except Exception as e:
            print(f"   ⚠️  AI extraction error: {e}")
            return []
    
    def _parse_text_simple(self, text: str) -> pd.DataFrame:
        """Simple text to DataFrame (fallback)."""
        lines = [line.strip() for line in text.split('\n') if line.strip()]
        
        if not lines:
            return pd.DataFrame({'content': ['No content']})
        
        # Try to detect if it's tabular
        sample = lines[:min(10, len(lines))]
        for delimiter in ['\t', ',', '|', ';']:
            if all(delimiter in line for line in sample):
                try:
                    df = pd.read_csv(StringIO('\n'.join(lines)), sep=delimiter)
                    if len(df.columns) > 1:
                        return df
                except:
                    continue
        
        return pd.DataFrame({
            'line_number': range(1, len(lines) + 1),
            'content': lines
        })
    
    def _store_dataframe(self, df: pd.DataFrame, name: str, silent: bool = False):
        """Store DataFrame in database."""
        df.columns = [str(c).strip().replace(" ", "_").replace("-", "_") for c in df.columns]
        df.to_sql(name, self.conn, if_exists='replace', index=False)
        self.current_table = name
        self._refresh_schema()
        
        if not silent:
            print(f"✅ Uploaded to table: {name}")
            print(f"   📊 {len(df)} rows × {len(df.columns)} columns")
            print(f"   🔤 Columns: {', '.join(df.columns[:10].tolist())}{' ...' if len(df.columns) > 10 else ''}")
    
    # ========================================================================
    # VIEW DATABASE - IMPROVED
    # ========================================================================
    
    def tables(self) -> Dict[str, dict]:
        """Show all tables with details."""
        print("\n" + "="*70)
        print("📋 TABLES IN DATABASE")
        print("="*70)
        
        all_tables = self._get_table_names()
        
        if not all_tables:
            print("❌ No tables found")
            return {}
        
        result = {}
        for i, tbl in enumerate(all_tables, 1):
            count = pd.read_sql_query(f"SELECT COUNT(*) FROM {tbl}", self.conn).iloc[0, 0]
            cols = self.schema_info.get(tbl, {})
            col_list = list(cols.keys())
            
            marker = "👉" if tbl == self.current_table else "  "
            print(f"{marker} {i}. Table: {tbl}")
            print(f"      📊 {count} rows, {len(col_list)} columns")
            print(f"      🔤 Columns: {', '.join(col_list[:8])}{' ...' if len(col_list) > 8 else ''}")
            
            result[tbl] = {
                'rows': count,
                'columns': col_list
            }
        
        print("="*70)
        return result
    
    def schema(self, table: Optional[str] = None) -> dict:
        """Show detailed schema with data types."""
        if not self.schema_info:
            self._refresh_schema()
        
        print("\n" + "="*70)
        print("📋 DATABASE SCHEMA")
        print("="*70)
        
        tables_to_show = [table] if table else self.schema_info.keys()
        
        result = {}
        for tbl in tables_to_show:
            if tbl in self.schema_info:
                count = pd.read_sql_query(f"SELECT COUNT(*) FROM {tbl}", self.conn).iloc[0, 0]
                print(f"\n📊 Table: {tbl}")
                print(f"   Records: {count}")
                print(f"   Columns:")
                
                for col, dtype in self.schema_info[tbl].items():
                    print(f"     - {col:<30} ({dtype})")
                
                result[tbl] = {
                    'records': count,
                    'columns': self.schema_info[tbl]
                }
        
        print("="*70)
        return result
    
    def peek(self, table: Optional[str] = None, n: int = 5) -> pd.DataFrame:
        """View sample data."""
        tbl = table or self.current_table
        if not tbl:
            print("❌ No table specified")
            return pd.DataFrame()
        
        df = pd.read_sql_query(f"SELECT * FROM {tbl} LIMIT {n}", self.conn)
        print(f"\n📊 Sample from '{tbl}' ({n} rows):")
        print(df.to_string(index=False))
        return df
    
    def info(self):
        """Show complete database overview."""
        return self.tables()
    
    # ========================================================================
    # QUERY METHODS
    # ========================================================================
    
    def sql(self, query: str, viz: bool = False) -> 'QueryResult':
        """Execute SQL directly (no API cost)."""
        print(f"\n⚡ Executing SQL...")
        
        try:
            df = pd.read_sql_query(query, self.conn)
            print(f"✅ Success! {len(df)} rows returned")
            
            fig = None
            if viz and not df.empty:
                fig = self._visualize(df, "SQL Query Result")
            
            return QueryResult(True, query, df, fig)
        except Exception as e:
            print(f"❌ Error: {e}")
            return QueryResult(False, query, pd.DataFrame(), None, str(e))
    
    def ask(self, question: str, viz: bool = False, table: Optional[str] = None) -> 'QueryResult':
        """Ask question in natural language."""
        if not self.client:
            print("❌ No API key configured")
            return QueryResult(False, "", pd.DataFrame(), None, "No API key")
        
        print(f"\n🔍 Question: {question}")
        
        tbl = table or self.current_table
        if not tbl:
            all_tables = self._get_table_names()
            if all_tables:
                tbl = all_tables[0]
            else:
                print("❌ No tables found")
                return QueryResult(False, "", pd.DataFrame(), None, "No table")
        
        cache_key = hashlib.md5(f"{question}:{tbl}".encode()).hexdigest()
        if cache_key in self.cache:
            sql_query = self.cache[cache_key]
            print("   💾 From cache")
        else:
            sql_query = self._generate_sql(question, tbl)
            self.cache[cache_key] = sql_query
        
        print(f"   📝 SQL: {sql_query}")
        
        try:
            df = pd.read_sql_query(sql_query, self.conn)
            print(f"✅ Success! {len(df)} rows")
            
            fig = None
            if viz and not df.empty:
                fig = self._visualize(df, question)
            
            return QueryResult(True, sql_query, df, fig)
        except Exception as e:
            print(f"❌ Error: {e}")
            return QueryResult(False, sql_query, pd.DataFrame(), None, str(e))
    
    def interactive(self, question: str) -> 'QueryResult':
        """Ask with interactive visualization prompt."""
        print(f"\n🔍 Question: {question}")
        choice = input("💡 Visualize? (yes/no): ").strip().lower()
        viz = choice in ['yes', 'y', 'yeah', 'yep', 'sure']
        return self.ask(question, viz=viz)
    
    # ========================================================================
    # DATABASE EXPORT
    # ========================================================================
    
    def export_db(self, path: str, format: str = "sqlite"):
        """Export entire database."""
        print(f"\n💾 Exporting to {format}...")
        
        format = format.lower()
        
        if format == "sqlite":
            shutil.copy2(self.db_path, path)
            print(f"✅ Saved to {path}")
        
        elif format == "sql":
            with open(path, 'w', encoding='utf-8') as f:
                for line in self.conn.iterdump():
                    f.write(f'{line}\n')
            print(f"✅ Saved to {path}")
        
        elif format == "json":
            data = {}
            for table in self._get_table_names():
                df = pd.read_sql_query(f"SELECT * FROM {table}", self.conn)
                data[table] = df.to_dict(orient='records')
            
            with open(path, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=2, default=str)
            print(f"✅ Saved to {path}")
        
        elif format == "excel":
            with pd.ExcelWriter(path, engine='openpyxl') as writer:
                for table in self._get_table_names():
                    df = pd.read_sql_query(f"SELECT * FROM {table}", self.conn)
                    df.to_excel(writer, sheet_name=table[:31], index=False)
            print(f"✅ Saved to {path}")
        
        else:
            raise ValueError(f"Unsupported format: {format}")
        
        return self
    
    def save_to_mysql(self, host: str, user: str, password: str, database: str,
                      port: int = 3306, tables: Optional[List[str]] = None):
        """Save to MySQL (local or cloud)."""
        try:
            from sqlalchemy import create_engine
        except ImportError:
            raise ImportError("Run: pip install sqlalchemy mysql-connector-python")
        
        print(f"\n🔄 Connecting to MySQL at {host}:{port}...")
        
        connection_string = f"mysql+mysqlconnector://{user}:{password}@{host}:{port}/{database}"
        engine = create_engine(connection_string)
        
        tables_to_export = tables or self._get_table_names()
        
        print(f"📤 Exporting {len(tables_to_export)} tables...")
        
        for table in tables_to_export:
            df = pd.read_sql_query(f"SELECT * FROM {table}", self.conn)
            df.to_sql(table, engine, if_exists='replace', index=False)
            print(f"   ✅ {table}: {len(df)} rows")
        
        print(f"✅ Complete!")
        return self
    
    def save_to_postgres(self, host: str, user: str, password: str, database: str,
                         port: int = 5432, tables: Optional[List[str]] = None):
        """Save to PostgreSQL (local or cloud)."""
        try:
            from sqlalchemy import create_engine
        except ImportError:
            raise ImportError("Run: pip install sqlalchemy psycopg2-binary")
        
        print(f"\n🔄 Connecting to PostgreSQL at {host}:{port}...")
        
        connection_string = f"postgresql://{user}:{password}@{host}:{port}/{database}"
        engine = create_engine(connection_string)
        
        tables_to_export = tables or self._get_table_names()
        
        print(f"📤 Exporting {len(tables_to_export)} tables...")
        
        for table in tables_to_export:
            df = pd.read_sql_query(f"SELECT * FROM {table}", self.conn)
            df.to_sql(table, engine, if_exists='replace', index=False)
            print(f"   ✅ {table}: {len(df)} rows")
        
        print(f"✅ Complete!")
        return self
    
    def backup(self, backup_path: str = None):
        """Create complete backup."""
        if backup_path:
            backup_dir = Path(backup_path)
            backup_dir.mkdir(parents=True, exist_ok=True)
        else:
            backup_dir = Path(".")
        
        timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
        
        print(f"\n💾 Creating backup...")
        
        db_backup = backup_dir / f"sutra_backup_{timestamp}.db"
        self.export_db(str(db_backup), format="sqlite")
        
        json_backup = backup_dir / f"sutra_data_{timestamp}.json"
        self.export_db(str(json_backup), format="json")
        
        print(f"\n✅ Backup complete!")
        print(f"   📁 Database: {db_backup}")
        print(f"   📊 Data: {json_backup}")
        
        return self
    
    # ========================================================================
    # UTILITIES
    # ========================================================================
    
    def export(self, data: pd.DataFrame, path: str, format: str = "csv"):
        """Export results."""
        fmt = format.lower()
        if fmt == "csv":
            data.to_csv(path, index=False)
        elif fmt in ["excel", "xlsx"]:
            data.to_excel(path, index=False)
        elif fmt == "json":
            data.to_json(path, orient="records", indent=2)
        else:
            raise ValueError(f"Unknown format: {format}")
        
        print(f"✅ Exported to {path}")
        return self
    
    def close(self):
        """Close database."""
        if self.conn:
            self.conn.close()
            print("✅ Closed")
    
    def _get_table_names(self) -> List[str]:
        """Get list of tables."""
        self.cursor.execute("SELECT name FROM sqlite_master WHERE type='table'")
        return [r[0] for r in self.cursor.fetchall()]
    
    def _refresh_schema(self):
        """Refresh schema info."""
        tables = self._get_table_names()
        
        self.schema_info = {}
        for tbl in tables:
            self.cursor.execute(f"PRAGMA table_info({tbl})")
            self.schema_info[tbl] = {r[1]: r[2] for r in self.cursor.fetchall()}
    
    def _generate_sql(self, question: str, table: str) -> str:
        """Generate SQL with OpenAI."""
        schema = self.schema_info.get(table, {})
        sample_df = pd.read_sql_query(f"SELECT * FROM {table} LIMIT 3", self.conn)
        sample = sample_df.to_string(index=False)
        
        schema_str = ", ".join([f"{col} ({dtype})" for col, dtype in schema.items()])
        
        prompt = f"""Convert to SQL.

Database: SQLite
Table: {table}
Columns: {schema_str}

Sample:
{sample}

Question: {question}

Return ONLY SQL. No explanations."""
        
        response = self.client.chat.completions.create(
            model="gpt-4o-mini",
            messages=[
                {"role": "system", "content": "SQL expert. Return only SQL code."},
                {"role": "user", "content": prompt}
            ],
            temperature=0
        )
        
        sql = response.choices[0].message.content.strip()
        return sql.replace("```sql", "").replace("```", "").strip()
    
    def _visualize(self, df: pd.DataFrame, title: str):
        """Create visualization."""
        if not HAS_PLOTLY and not HAS_MATPLOTLIB:
            print("⚠️  Install plotly or matplotlib")
            return None
        
        print("📊 Creating chart...")
        
        if HAS_PLOTLY:
            return self._plotly_viz(df, title)
        else:
            return self._matplotlib_viz(df, title)
    
    def _plotly_viz(self, df: pd.DataFrame, title: str):
        """Plotly chart."""
        try:
            numeric = df.select_dtypes(include=[np.number]).columns.tolist()
            categorical = df.select_dtypes(include=['object']).columns.tolist()
            
            if len(df) == 1 or not numeric:
                fig = go.Figure(data=[go.Table(
                    header=dict(values=list(df.columns)),
                    cells=dict(values=[df[c] for c in df.columns])
                )])
            elif categorical and numeric:
                fig = px.bar(df, x=categorical[0], y=numeric[0], title=title)
            elif len(numeric) >= 2:
                fig = px.line(df, y=numeric[0], title=title)
            else:
                fig = px.bar(df, y=df.columns[0], title=title)
            
            fig.show()
            print("✅ Chart displayed")
            return fig
        except Exception as e:
            print(f"⚠️  Error: {e}")
            return None
    
    def _matplotlib_viz(self, df: pd.DataFrame, title: str):
        """Matplotlib chart."""
        try:
            plt.figure(figsize=(10, 6))
            numeric = df.select_dtypes(include=[np.number]).columns
            
            if len(numeric) > 0:
                df[numeric[0]].plot(kind='bar')
            else:
                df.iloc[:, 0].value_counts().plot(kind='bar')
            
            plt.title(title)
            plt.tight_layout()
            plt.show()
            print("✅ Chart displayed")
            return plt.gcf()
        except Exception as e:
            print(f"⚠️  Error: {e}")
            return None
    
    def __enter__(self):
        return self
    
    def __exit__(self, *args):
        self.close()
    
    def __repr__(self):
        return f"SUTRA(tables={len(self.schema_info)}, current='{self.current_table}')"


class QueryResult:
    """Query result."""
    
    def __init__(self, success: bool, sql: str, data: pd.DataFrame, viz, error: str = None):
        self.success = success
        self.sql = sql
        self.data = data
        self.viz = viz
        self.error = error
    
    def __repr__(self):
        if self.success:
            return f"QueryResult(rows={len(self.data)}, cols={len(self.data.columns)})"
        return f"QueryResult(error='{self.error}')"
    
    def show(self):
        if self.success:
            print(self.data)
        else:
            print(f"Error: {self.error}")
        return self


def quick_start(api_key: str, data_path: str, question: str, viz: bool = False):
    """One-liner."""
    with SUTRA(api_key=api_key) as sutra:
        sutra.upload(data_path)
        return sutra.ask(question, viz=viz)


if __name__ == "__main__":
    print("""
╔══════════════════════════════════════════════════════════════╗
║              QuerySUTRA v0.2.2 - FIXED                       ║
║   Structured-Unstructured-Text-Retrieval-Architecture        ║
╚══════════════════════════════════════════════════════════════╝

Installation: pip install QuerySUTRA
    
Quick Start:
    from sutra import SUTRA
    
    sutra = SUTRA(api_key="your-key")
    sutra.upload("document.pdf")  # Creates multiple tables!
    sutra.tables()                # Show all tables
    sutra.schema()                # Show schema details
    result = sutra.ask("Show people from New York", viz=True)
    
    # Export
    sutra.save_to_mysql("host", "user", "pass", "db")

Supported: CSV, Excel, JSON, SQL, PDF, DOCX, TXT, DataFrame
""")
