# AUTOGENERATED! DO NOT EDIT! File to edit: ../src/helpers.ipynb.

# %% auto 0
__all__ = ['ADDRESS_ZERO', 'MAX_UINT256', 'normalize_address', 'chunk', 'amount_to_k_string', 'format_currency',
           'format_percentage', 'amount_to_m_string', 'float_to_uint256', 'get_future_timestamp', 'apply_slippage',
           'parse_ether', 'get_unique_str', 'get_salt', 'to_bytes32', 'to_bytes32_str', 'Pair', 'find_all_paths',
           'ICACallData', 'hash_ICA_calls', 'serialize_ica_calls', 'Timer', 'time_it', 'atime_it', 'require_supersim']

# %% ../src/helpers.ipynb 2
from json import dumps
from web3 import Web3, constants
from eth_abi import encode
from typing import List, Tuple, Optional, Callable
from decimal import Decimal, getcontext
from datetime import datetime, timedelta
from dataclasses import dataclass
import networkx as nx
import math, time, asyncio, decimal, secrets, socket
from contextlib import contextmanager, asynccontextmanager
from fastcore.test import test_eq

# %% ../src/helpers.ipynb 3
def normalize_address(address: str) -> str: return Web3.to_checksum_address(address.lower())

ADDRESS_ZERO = constants.ADDRESS_ZERO
MAX_UINT256 = Web3.to_int(hexstr='0x' + 'f' * 64)

def chunk(list_to_chunk: List, n: int):
    for i in range(0, len(list_to_chunk), n):
        yield list_to_chunk[i : i + n]


def amount_to_k_string(amount: float) -> str:
    """Turns 2000 to "2K" """
    return f"{round(amount/1000, 2)}K"


def format_currency(value: float, symbol: str = "$", prefix: bool = True) -> str:
    v = "{:0,.2f}".format(value)
    return f"{symbol}{v}" if prefix else f"{v} {symbol}"


def format_percentage(value: float) -> str:
    return "{:0,.2f} %".format(value)


def amount_to_m_string(amount: float) -> str:
    """Turns 2000000 to "2M" """
    return f"{round(amount/1000000, 2)}M"

# %% ../src/helpers.ipynb 4
def float_to_uint256(value: float, decimals: int = 18) -> int:
    """Convert float to uint256 with decimal scaling"""
    # Convert float to Decimal for precision
    amount_decimal = Decimal(str(value))
    # Scale by decimals
    scaled_amount = amount_decimal * Decimal(10 ** decimals)
    # Convert to integer
    return int(scaled_amount)

# %% ../src/helpers.ipynb 6
def get_future_timestamp(deadline_minutes: float) -> int:
    """Convert minutes from now to future unix timestamp"""
    future_time = datetime.now() + timedelta(minutes=deadline_minutes)
    return int(future_time.timestamp())

# %% ../src/helpers.ipynb 8
def apply_slippage(amount: int, slippage: float) -> int:
    if slippage < 0 or slippage > 1: raise ValueError("Slippage must be between 0 and 1")
    return int(math.ceil(amount * (1 - slippage)))

# %% ../src/helpers.ipynb 9
def parse_ether(ether: str) -> int:
    # Set precision high enough to handle 18 decimal places
    getcontext().prec = 50
    
    try:
        # Convert to Decimal for precise arithmetic
        ether_decimal = Decimal(str(ether))
        
        # Convert to wei (multiply by 10^18)
        wei_decimal = ether_decimal * Decimal('1000000000000000000')
        
        # Convert to integer
        return int(wei_decimal)
    
    except (ValueError, TypeError, decimal.InvalidOperation) as e:
        raise ValueError(f"Invalid ether value: {ether}") from e

# %% ../src/helpers.ipynb 10
def get_unique_str(length: int) -> str:
    """
    Generate a cryptographically secure random string of specified length.
    
    This is the Python equivalent of the TypeScript function using crypto.getRandomValues().
    Uses secrets module for cryptographic security.
    
    Args:
        length: Desired length of the random string
        
    Returns:
        str: Random string of specified length containing digits
        
    Examples:
        >>> len(get_unique_str(10))
        10
        >>> get_unique_str(5).isdigit()
        True
    """
    # Generate random bytes and convert to string of digits
    random_bytes = secrets.token_bytes(length)
    return ''.join(str(byte % 10) for byte in random_bytes)[:length]

def get_salt() -> str: return f"0x{get_unique_str(64)}" 

# %% ../src/helpers.ipynb 12
def to_bytes32(val: str) -> bytes: 
    # Remove 0x prefix and pad to 64 hex characters (32 bytes)
    hex_val = val.replace('0x', '').zfill(64)
    return bytes.fromhex(hex_val)

def to_bytes32_str(val: str) -> str: return f"0x{to_bytes32(val).hex()}"

# %% ../src/helpers.ipynb 18
# Claude 3.7 sonnet made this

@dataclass
class Pair: token0: str; token1: str; pool: str

def find_all_paths(pairs: List[Pair], start_token: str, end_token: str, cutoff=3) -> List[List[Tuple]]:
    # MultiGraph required to support parallel edges
    # same tokens can be present in different pools, hence parallel edges
    # specific pool identifier is stored inside edge attribute
    G, complete_paths = nx.MultiGraph(), []
    for pair in pairs: G.add_edge(pair.token0, pair.token1, pool=pair.pool)
    node_paths =  [p for p in nx.all_simple_paths(G, source=start_token, target=end_token, cutoff=cutoff)]
    for path in node_paths:
        edge_path = []
        # For each consecutive pair of nodes in the path
        for i in range(len(path) - 1):
            current = path[i]
            next_node = path[i + 1]
            
            # Get all edges between these nodes
            edges = G.get_edge_data(current, next_node)
            
            # There might be multiple edges (pools) between these nodes
            # Add all possible edges to create different complete paths
            current_paths = [] if not edge_path else edge_path.copy()
            new_edge_paths = []
            
            # If this is the first segment, initialize with empty path
            if not current_paths:
                current_paths = [[]]
                
            # For each possible edge between current and next_node
            for edge_key, edge_attrs in edges.items():
                pool = edge_attrs['pool']
                for current_path in current_paths:
                    # Create a new path that includes this edge
                    new_path = current_path + [(current, next_node, pool)]
                    new_edge_paths.append(new_path)
            
            edge_path = new_edge_paths
        
        # Add all possible edge paths to the complete paths
        complete_paths.extend(edge_path)
    
    # seen is a list of strings that look like this ["pool1-pool2-pool3", ...]
    uniques, seen = [], []

    for path in complete_paths:
        p = '-'.join(map(lambda x: x[2], path))
        if p not in seen:
            uniques.append(path)
            seen.append(p)

    # remove duplicates
    return uniques


# %% ../src/helpers.ipynb 21
# TODO: get rid of ICACallData, use tuples instead
@dataclass(frozen=True)
class ICACallData: to: str; value: int; data: str


def hash_ICA_calls(calls: List[ICACallData], salt: str) -> bytes:
  call_tuples = [(bytes.fromhex(call.to.replace('0x', '')), call.value, bytes.fromhex(call.data.replace('0x', '')))  for call in calls]
  encoded = encode(["(bytes32,uint256,bytes)[]"], [call_tuples])
  return Web3.keccak(hexstr=f"{salt}{encoded.hex()}")

def serialize_ica_calls(calls: List[ICACallData]) -> List[dict]:
    """
    Convert a list of ICACallData to JSON string.
    """
    return list(map(lambda call: {"to": call.to, "value": str(call.value), "data": call.data}, calls))

# %% ../src/helpers.ipynb 25
# Claude 4 sonnet made this

class Timer:
    """Simple timer utility for measuring execution time"""
    
    def __init__(self, name: str = "Operation", precision: int = 4, callback: Optional[Callable] = None):
        self.name = name
        self.precision = precision
        self.callback = callback
        self.start_time: Optional[float] = None
        self.end_time: Optional[float] = None
        self.elapsed: Optional[float] = None
    
    def __enter__(self):
        self.start_time = time.perf_counter()
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.end_time = time.perf_counter()
        self.elapsed = self.end_time - self.start_time
        result = f"{self.name} took {self.elapsed:.{self.precision}f} seconds"
        
        if self.callback:
            self.callback(self.elapsed, result)
        else:
            print(result)
    
    async def __aenter__(self):
        self.start_time = time.perf_counter()
        return self
    
    async def __aexit__(self, exc_type, exc_val, exc_tb):
        self.end_time = time.perf_counter()
        self.elapsed = self.end_time - self.start_time
        result = f"{self.name} took {self.elapsed:.{self.precision}f} seconds"
        
        if self.callback:
            if asyncio.iscoroutinefunction(self.callback):
                await self.callback(self.elapsed, result)
            else:
                self.callback(self.elapsed, result)
        else:
            print(result)

@contextmanager
def time_it(name: str = "Operation", precision: int = 4, callback: Optional[Callable] = None):
    """Context manager for timing synchronous code execution"""
    timer = Timer(name, precision, callback)
    with timer:
        yield timer

@asynccontextmanager
async def atime_it(name: str = "Operation", precision: int = 4, callback: Optional[Callable] = None):
    """Async context manager for timing asynchronous code execution"""
    timer = Timer(name, precision, callback)
    async with timer:
        yield timer

# %% ../src/helpers.ipynb 28
def require_supersim():
    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.settimeout(2)
        result = sock.connect_ex(('127.0.0.1', 4444))
        # are you running supersim?
        test_eq(result, 0)
    except socket.error as err:
        test_eq(err, None)
    finally:
        sock.close()
