import json
import logging
import os
from pathlib import Path
from typing import Any

from dapr.ext.workflow import WorkflowActivityContext
from dapr_agents import OpenAIChatClient
from dapr_agents.workflow.decorators import llm_activity
from dotenv import load_dotenv

from deepbrief.activities.models import PodcastEpisode
from deepbrief.activities.prompts import GENERATE_EPISODE_PROMPT
from deepbrief.services import store_json

logger = logging.getLogger(__name__)

load_dotenv()

# Initializing LLM clients with centralized configuration
llm = OpenAIChatClient(
    api_key=os.getenv("OPENAI_API_KEY"),
    model=os.getenv("OPENAI_API_MODEL"),
    base_url=os.getenv("OPENAI_API_BASE_URL"),
)


@llm_activity(prompt=GENERATE_EPISODE_PROMPT, llm=llm)
def generate_episode_metadata(transcript: str, paper_id: str) -> PodcastEpisode:
    pass


def write_episode_to_file(ctx: WorkflowActivityContext, input_data: dict):
    """
    Process the results and update the article index with podcast episode metadata.

    """
    input_data = input_data or {}
    article_index: dict[str, Any] = input_data.get("article_index") or {}
    results: list[dict[str, Any]] = input_data.get("results") or []
    episodes_directory: str = input_data.get("episodes_directory", "output/episodes")
    episodes_storage_prefix: str = input_data.get("episodes_storage_prefix", "episodes")
    persist_locally = bool(input_data.get("persist_locally", True))
    recordings_metadata = input_data.get("recordings_metadata") or []
    transcripts_metadata = input_data.get("transcripts_metadata") or []

    recordings_lookup = {
        rec.get("article_id"): rec for rec in recordings_metadata if rec.get("article_id")
    }
    transcripts_lookup = {
        rec.get("article_id"): rec for rec in transcripts_metadata if rec.get("article_id")
    }

    outputs = []
    for result in results:
        paper_id = result["paper_id"]
        article_metadata = article_index.get(paper_id)
        if not article_metadata:
            continue

        episode_metadata = {
            "title": result["title"],
            "overview": result["overview"],
            "key_takeaways": result["key_takeaways"],
        }
        recording_entry = recordings_lookup.get(paper_id, {})
        transcript_entry = transcripts_lookup.get(paper_id, {})

        envelope = {
            **article_metadata,
            "podcast_episode": episode_metadata,
            "recording_metadata": recording_entry,
            "search_sources": transcript_entry.get("search_sources", []),
        }

        file_path = None
        if persist_locally:
            Path(episodes_directory).mkdir(parents=True, exist_ok=True)
            output_file = Path(episodes_directory) / f"{paper_id}.json"
            output_file.write_text(json.dumps(envelope, indent=4), encoding="utf-8")
            file_path = str(output_file)

        storage_key = f"{episodes_storage_prefix}/{paper_id}.json"
        store_json(storage_key, envelope)
        logger.info("Saved episode metadata for %s (local=%s)", paper_id, bool(file_path))
        outputs.append({"file_path": file_path, "storage_key": storage_key})
    return outputs
