import logging
import os
from textwrap import shorten
from typing import Any

from dapr.ext.workflow import WorkflowActivityContext
from dapr_agents import OpenAIChatClient
from dapr_agents.workflow.decorators import llm_activity
from dotenv import load_dotenv
from tavily import TavilyClient

from deepbrief.activities.models import ResearchQueries
from deepbrief.activities.prompts import QUERY_GENERATOR_PROMPT

logger = logging.getLogger(__name__)

load_dotenv()

# Initialize clients with centralized configuration
llm = OpenAIChatClient(
    api_key=os.getenv("OPENAI_API_KEY"),
    model=os.getenv("OPENAI_API_MODEL"),
    base_url=os.getenv("OPENAI_API_BASE_URL"),
)
web = TavilyClient(api_key=os.getenv("TAVILY_API_KEY"))

# Search Variables
RESULT_SEPARATOR = "\n" + ("─" * 60) + "\n"
SNIPPET_CHAR_LIMIT = 400
SNIPPET_PLACEHOLDER = " …"


@llm_activity(prompt=QUERY_GENERATOR_PROMPT, llm=llm)
def generate_search_queries(date_time: str, context: str) -> ResearchQueries:
    """
    Generate web search queries based on the paper context.
    """
    pass


def web_search(ctx: WorkflowActivityContext, input_data: dict | None = None) -> dict[str, Any]:
    """Run a Tavily search and return snippets plus aggregated context."""

    input_data = input_data or {}
    query = str(input_data.get("query", "")).strip()
    if not query:
        raise ValueError("A non-empty query string is required.")

    limit = int(input_data.get("limit", 5))
    limit = max(1, min(limit, 20))

    try:
        response = web.search(
            query=query,
            max_results=limit,
            search_depth="basic",
            topic="general",
        )
        hits = response.get("results", [])
        snippets: list[dict[str, Any]] = []
        for item in hits:
            title = item.get("title") or "Untitled"
            url = item.get("url") or "Unknown URL"
            raw_text = item.get("content", "").strip()
            snippet = shorten(
                raw_text,
                width=SNIPPET_CHAR_LIMIT,
                placeholder=SNIPPET_PLACEHOLDER,
            )
            snippets.append({"title": title, "url": url, "snippet": snippet})

        aggregated_text, sources = _summarize_snippets(snippets)
        return {
            "query": query,
            "results": snippets,
            "text": aggregated_text,
            "sources": sources,
        }

    except Exception as exc:
        logging.error("Error in web search for query '%s': %s", query, exc)
        return {"query": query, "results": [], "text": "", "sources": ""}


def _summarize_snippets(snippets: list[dict[str, Any]]) -> tuple[str, str]:
    if not snippets:
        return "", ""

    snippet_text = RESULT_SEPARATOR.join(snippet["snippet"] for snippet in snippets)
    sources = "\n".join(
        f"{idx}. [{item['title']}]({item['url']})" for idx, item in enumerate(snippets, start=1)
    )
    return snippet_text, sources
