import json
import logging
import os
from pathlib import Path
from typing import Any

from dapr.ext.workflow import WorkflowActivityContext
from dapr_agents import OpenAIChatClient
from dapr_agents.workflow.decorators import llm_activity
from dotenv import load_dotenv

from deepbrief.activities.models import PodcastDialogue
from deepbrief.activities.prompts import GENERATE_TRANSCRIPT_PROMPT
from deepbrief.services import store_json

logger = logging.getLogger(__name__)

load_dotenv()

# Initializing LLM clients with centralized configuration
llm = OpenAIChatClient(
    api_key=os.getenv("OPENAI_API_KEY"),
    model=os.getenv("OPENAI_API_MODEL"),
    base_url=os.getenv("OPENAI_API_BASE_URL"),
)


@llm_activity(prompt=GENERATE_TRANSCRIPT_PROMPT, llm=llm)
def generate_transcript(
    podcast_name: str,
    host_name: str,
    prompt: str,
    max_rounds: int,
) -> PodcastDialogue:
    pass


def write_transcript_to_file(ctx: WorkflowActivityContext, input_data: dict) -> dict[str, str]:
    """
    Write the structured podcast dialogue to a specified file in a given directory.

    """
    try:
        input_data = input_data or {}
        podcast_dialogue: dict[str, Any] = input_data.get("podcast_dialogue") or {}
        transcripts_directory: str = input_data.get("transcripts_directory", "output/transcripts")
        file_name: str = input_data.get("file_name", "transcript.json")
        storage_prefix: str = input_data.get("storage_prefix", "transcripts")
        persist_locally = bool(input_data.get("persist_locally", False))

        file_path = None
        if persist_locally:
            Path(transcripts_directory).mkdir(parents=True, exist_ok=True)
            file_path_obj = Path(transcripts_directory) / file_name
            with file_path_obj.open("w", encoding="utf-8") as file:
                json.dump(podcast_dialogue, file, ensure_ascii=False, indent=4)
            file_path = str(file_path_obj)
            logger.info("Podcast dialogue successfully written to %s", file_path)

        storage_key = f"{storage_prefix}/{file_name}"
        store_json(storage_key, podcast_dialogue)
        logger.info("Transcript uploaded to shared storage at %s", storage_key)

        return {"file_path": file_path, "storage_key": storage_key}
    except Exception as e:
        logger.error(f"Error writing podcast dialogue to file: {e}")
        raise
