import asyncio
import threading
from datetime import timedelta

import pytest

from pgqueuer import db
from pgqueuer.models import Job
from pgqueuer.qm import QueueManager
from pgqueuer.queries import Queries
from pgqueuer.types import QueueExecutionMode


@pytest.mark.parametrize("N", (1, 4, 32, 100))
async def test_cancellation_async(
    apgdriver: db.Driver,
    N: int,
) -> None:
    event = asyncio.Event()
    cancel_called_not_cancel_called = list[bool]()
    q = Queries(apgdriver)
    qm = QueueManager(apgdriver, resources={"test_key": "async"})

    @qm.entrypoint("to_be_canceled")
    async def to_be_canceled(job: Job) -> None:
        scope = qm.get_context(job.id).cancellation
        assert qm.get_context(job.id).resources["test_key"] == "async"
        await event.wait()
        cancel_called_not_cancel_called.append(scope.cancel_called)

    job_ids = await q.enqueue(
        ["to_be_canceled"] * N,
        [f"{n}".encode() for n in range(N)],
        [0] * N,
    )

    async def waiter() -> None:
        while sum(x.count for x in await q.queue_size() if x.status == "picked") < N:
            await asyncio.sleep(0.01)

        await q.mark_job_as_cancelled(job_ids)
        event.set()

        qm.shutdown.set()

    await asyncio.gather(
        qm.run(dequeue_timeout=timedelta(seconds=0.0)),
        waiter(),
    )

    assert sum(cancel_called_not_cancel_called) == N

    # Logged as canceled

    assert sum(x.status == "canceled" for x in await q.queue_log()) == N


@pytest.mark.parametrize("N", (1, 4, 32, 100))
async def test_cancellation_sync(
    apgdriver: db.Driver,
    N: int,
) -> None:
    event = threading.Event()
    cancel_called_not_cancel_called = list[bool]()
    q = Queries(apgdriver)
    qm = QueueManager(apgdriver, resources={"test_key": "sync"})

    @qm.entrypoint("to_be_canceled")
    def to_be_canceled(job: Job) -> None:
        nonlocal event
        scope = qm.get_context(job.id).cancellation
        assert qm.get_context(job.id).resources["test_key"] == "sync"
        event.wait()
        cancel_called_not_cancel_called.append(scope.cancel_called)

    job_ids = await q.enqueue(
        ["to_be_canceled"] * N,
        [f"{n}".encode() for n in range(N)],
        [0] * N,
    )

    async def waiter() -> None:
        while sum(x.count for x in await q.queue_size() if x.status == "picked") < N:
            await asyncio.sleep(0)

        await q.mark_job_as_cancelled(job_ids)
        event.set()

        qm.shutdown.set()

    await asyncio.gather(
        qm.run(dequeue_timeout=timedelta(seconds=0.01)),
        waiter(),
    )

    assert sum(cancel_called_not_cancel_called) == N

    # Logged as canceled
    assert sum(x.status == "canceled" for x in await q.queue_log()) == N


@pytest.mark.parametrize("N", (1, 4, 32, 100))
async def test_cancellation_async_context_manager(
    apgdriver: db.Driver,
    N: int,
) -> None:
    event = asyncio.Event()
    cancel_called_not_cancel_called = list[bool]()
    q = Queries(apgdriver)
    qm = QueueManager(apgdriver, resources={"test_key": "async_cm"})

    @qm.entrypoint("to_be_canceled")
    async def to_be_canceled(job: Job) -> None:
        with qm.get_context(job.id).cancellation as scope:
            assert qm.get_context(job.id).resources["test_key"] == "async_cm"
            await event.wait()
            cancel_called_not_cancel_called.append(scope.cancel_called)

    job_ids = await q.enqueue(
        ["to_be_canceled"] * N,
        [f"{n}".encode() for n in range(N)],
        [0] * N,
    )

    async def waiter() -> None:
        while sum(x.count for x in await q.queue_size() if x.status == "picked") < N:
            await asyncio.sleep(0)

        await q.mark_job_as_cancelled(job_ids)
        event.set()

        qm.shutdown.set()

    await asyncio.gather(
        qm.run(dequeue_timeout=timedelta(seconds=0.01)),
        waiter(),
    )

    assert sum(cancel_called_not_cancel_called) == 0

    # Logged as canceled
    assert sum(x.status == "canceled" for x in await q.queue_log()) == N


@pytest.mark.parametrize("N", (1, 4, 32, 100))
async def test_cancellation_sync_context_manager(
    apgdriver: db.Driver,
    N: int,
) -> None:
    with pytest.raises(NotImplementedError):
        raise NotImplementedError(
            "anyio.CancelScope() does not support cancellation in sync functions."
        )


async def test_cancellation_drain_mode(apgdriver: db.Driver) -> None:
    N = 10
    event = asyncio.Event()
    q = Queries(apgdriver)
    qm = QueueManager(apgdriver, resources={"test_key": "drain"})

    @qm.entrypoint("to_be_canceled")
    async def to_be_canceled(job: Job) -> None:
        ctx = qm.get_context(job.id)
        with ctx.cancellation:
            assert ctx.resources["test_key"] == "drain"
            await event.wait()

    jids = await q.enqueue(
        ["to_be_canceled"] * N,
        [f"{n}".encode() for n in range(N)],
        [0] * N,
    )

    async def cancel_after() -> None:
        await asyncio.sleep(0.5)
        await q.mark_job_as_cancelled(jids)

    await asyncio.gather(
        cancel_after(),
        qm.run(mode=QueueExecutionMode.drain),
    )

    assert sum(x.status == "canceled" for x in await q.queue_log()) == N
