from django.conf import settings
from django.contrib.auth.mixins import LoginRequiredMixin
from django.core.exceptions import PermissionDenied
from django.http import HttpResponse, Http404
from django.views import View
from django.views.generic import TemplateView
from django.views.generic.edit import UpdateView, DeleteView
from django.urls import reverse
from minidebconf.forms import register_form_factory
from minidebconf.models import Registration, is_registered


from qrcode import QRCode


class RegistrationMixin:
    @property
    def user(self):
        return self.request.user

    def get_object(self):
        if is_registered(self.user):
            return Registration.objects.get(user=self.user)
        else:
            return Registration(user=self.user)



class RegisterView(LoginRequiredMixin, RegistrationMixin, UpdateView):
    template_name = 'minidebconf/register.html'

    def dispatch(self, request, *args, **kwargs):
        if not settings.WAFER_REGISTRATION_OPEN:
            raise PermissionDenied("Registration is closed")
        return super().dispatch(request, *args, **kwargs)

    def get_form_class(self):
        return register_form_factory()

    def get_success_url(self):
        return reverse('registration_finished')


class UnregisterView(LoginRequiredMixin, RegistrationMixin, DeleteView):

    def get_success_url(self):
        return reverse('wafer_user_profile', args=[self.user.username])


class RegistrationFinishedView(LoginRequiredMixin, TemplateView):
    template_name = 'minidebconf/registration_finished.html'


class RegistrationQRCodeImageView(LoginRequiredMixin, RegistrationMixin, View):
    def get(self, request, *args, **kwargs):
        registration = self.get_object()
        if not registration.id:
            raise Http404()

        url = reverse(
            "admin:minidebconf_registration_change",
            args=(registration.pk,)
        )

        qr = QRCode()
        qr.add_data(url)

        response = HttpResponse(content_type="image/png")

        image = qr.make_image()
        image.save(response)

        return response
