from django.views.generic.edit import FormView, DeleteView
from django.views.generic.base import TemplateView, View
from tom_alerts.alerts import get_service_class, get_service_classes
from django.shortcuts import redirect, get_object_or_404
from django.utils import timezone
from django.urls import reverse, reverse_lazy
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib import messages
from django.core.cache import cache
from guardian.shortcuts import assign_perm
from django_filters.views import FilterView
from django_filters import FilterSet, ChoiceFilter, CharFilter
import json

from tom_alerts.models import BrokerQuery


class BrokerQueryCreateView(LoginRequiredMixin, FormView):
    """
    View for creating a new query to a broker.
    """
    template_name = 'tom_alerts/query_form.html'

    def get_broker_name(self):
        """
        Returns the broker specified in the request

        :returns: Broker name
        :rtype: str
        """
        if self.request.method == 'GET':
            return self.request.GET.get('broker')
        elif self.request.method == 'POST':
            return self.request.POST.get('broker')

    def get_form_class(self):
        """
        Returns the form class to use in this view.
        """
        broker_name = self.get_broker_name()

        if not broker_name:
            raise ValueError('Must provide a broker name')

        return get_service_class(broker_name).form

    def get_form(self):
        """
        Returns the
        """
        form = super().get_form()
        form.helper.form_action = reverse('tom_alerts:create')
        return form

    def get_initial(self):
        initial = super().get_initial()
        initial['broker'] = self.get_broker_name()
        return initial

    def form_valid(self, form):
        form.save()
        return redirect(reverse('tom_alerts:list'))


class BrokerQueryUpdateView(LoginRequiredMixin, FormView):
    template_name = 'tom_alerts/query_form.html'

    def get_object(self):
        return BrokerQuery.objects.get(pk=self.kwargs['pk'])

    def get_form_class(self):
        self.object = self.get_object()
        return get_service_class(self.object.broker).form

    def get_form(self):
        form = super().get_form()
        form.helper.form_action = reverse(
            'tom_alerts:update', kwargs={'pk': self.object.id}
        )
        return form

    def get_initial(self):
        initial = super().get_initial()
        initial.update(self.object.parameters_as_dict)
        initial['broker'] = self.object.broker
        return initial

    def form_valid(self, form):
        form.save(query_id=self.object.id)
        return redirect(reverse('tom_alerts:list'))


class BrokerQueryFilter(FilterSet):
    broker = ChoiceFilter(
        choices=[(k, k) for k in get_service_classes().keys()]
    )
    name = CharFilter(lookup_expr='icontains')

    class Meta:
        model = BrokerQuery
        fields = ['broker', 'name']


class BrokerQueryListView(FilterView):
    model = BrokerQuery
    template_name = 'tom_alerts/brokerquery_list.html'
    filterset_class = BrokerQueryFilter

    def get_context_data(self, *args, **kwargs):
        context = super().get_context_data(*args, **kwargs)
        context['installed_brokers'] = get_service_classes()
        return context


class BrokerQueryDeleteView(LoginRequiredMixin, DeleteView):
    model = BrokerQuery
    success_url = reverse_lazy('tom_alerts:list')


class RunQueryView(TemplateView):
    template_name = 'tom_alerts/query_result.html'

    def get_context_data(self, *args, **kwargs):
        context = super().get_context_data()
        query = get_object_or_404(BrokerQuery, pk=self.kwargs['pk'])
        broker_class = get_service_class(query.broker)()
        alerts = broker_class.fetch_alerts(query.parameters_as_dict)
        context['alerts'] = []
        query.last_run = timezone.now()
        query.save()
        context['query'] = query
        try:
            while len(context['alerts']) < 20:
                alert = next(alerts)
                generic_alert = broker_class.to_generic_alert(alert)
                cache.set('alert_{}'.format(generic_alert.id), json.dumps(alert), 3600)
                context['alerts'].append(generic_alert)
        except StopIteration:
            pass
        return context


class CreateTargetFromAlertView(LoginRequiredMixin, View):
    def post(self, request, *args, **kwargs):
        query_id = self.request.POST['query_id']
        broker_name = self.request.POST['broker']
        broker_class = get_service_class(broker_name)
        alerts = self.request.POST.getlist('alerts')
        if not alerts:
            messages.warning(request, 'Please select at least one alert from which to create a target.')
            return redirect(reverse('tom_alerts:run', kwargs={'pk': query_id}))
        for alert_id in alerts:
            cached_alert = cache.get('alert_{}'.format(alert_id))
            if not cached_alert:
                messages.error(request, 'Could not create targets. Try re running the query again.')
                return redirect(reverse('tom_alerts:run', kwargs={'pk': query_id}))
            generic_alert = broker_class().to_generic_alert(json.loads(cached_alert))
            target = generic_alert.to_target()
            target.save()
            broker_class().process_reduced_data(target, json.loads(cached_alert))
            target.save()
            for group in request.user.groups.all().exclude(name='Public'):
                assign_perm('tom_targets.view_target', group, target)
                assign_perm('tom_targets.change_target', group, target)
                assign_perm('tom_targets.delete_target', group, target)
        if (len(alerts) == 1):
            return redirect(reverse(
                'tom_targets:update', kwargs={'pk': target.id})
            )
        else:
            return redirect(reverse(
                'tom_targets:list')
            )
