"""Simple FastAPI application for KaiserLift.

This module exposes two endpoints:

* ``GET /`` – serves a small HTML form for uploading a FitNotes CSV file.
* ``POST /upload`` – accepts the uploaded CSV, processes it with the core
  KaiserLift utilities and returns the full HTML page generated by
  :func:`gen_html_viewer`.

The application can be started with ``python -m kaiserlift.webapp`` which will
launch a Uvicorn development server.
"""

from __future__ import annotations

import os

from fastapi import FastAPI, File, UploadFile
from fastapi.responses import HTMLResponse

from .pipeline import pipeline
from .running_pipeline import running_pipeline


app = FastAPI()


@app.get("/", response_class=HTMLResponse)
async def index() -> HTMLResponse:
    """Return a simple upload form."""

    return HTMLResponse(
        """
        <!DOCTYPE html>
        <html>
            <head>
                <title>KaiserLift - Data-Driven Workout Optimization</title>
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <style>
                    :root {
                        --bg: #fafafa;
                        --fg: #1a1a1a;
                        --bg-alt: #ffffff;
                        --border: #e5e7eb;
                        --primary: #3b82f6;
                        --primary-hover: #2563eb;
                    }
                    @media (prefers-color-scheme: dark) {
                        :root {
                            --bg: #0f0f0f;
                            --fg: #e5e5e5;
                            --bg-alt: #1a1a1a;
                            --border: #2a2a2a;
                            --primary: #60a5fa;
                            --primary-hover: #3b82f6;
                        }
                    }
                    [data-theme="dark"] {
                        --bg: #0f0f0f;
                        --fg: #e5e5e5;
                        --bg-alt: #1a1a1a;
                        --border: #2a2a2a;
                        --primary: #60a5fa;
                        --primary-hover: #3b82f6;
                    }
                    [data-theme="light"] {
                        --bg: #fafafa;
                        --fg: #1a1a1a;
                        --bg-alt: #ffffff;
                        --border: #e5e7eb;
                        --primary: #3b82f6;
                        --primary-hover: #2563eb;
                    }
                    body {
                        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
                        background-color: var(--bg);
                        color: var(--fg);
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        justify-content: center;
                        min-height: 100vh;
                        padding: 20px;
                        text-align: center;
                        transition: background-color 0.3s ease, color 0.3s ease;
                    }
                    .hero {
                        margin-bottom: 40px;
                    }
                    .hero h1 {
                        font-size: 2.5em;
                        margin-bottom: 10px;
                    }
                    .tagline {
                        font-size: 1.2em;
                        opacity: 0.8;
                        margin-bottom: 20px;
                    }
                    .demo-button {
                        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                        color: white;
                        padding: 14px 28px;
                        border: none;
                        border-radius: 8px;
                        font-size: 1.1em;
                        cursor: pointer;
                        margin: 10px;
                        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
                        transition: all 0.2s ease;
                        font-weight: 600;
                    }
                    .demo-button:hover {
                        transform: translateY(-2px);
                        box-shadow: 0 6px 12px rgba(0, 0, 0, 0.15);
                    }
                    .upload-section {
                        margin-top: 40px;
                    }
                    .upload-section h2 {
                        font-size: 1.5em;
                        margin-bottom: 10px;
                    }
                    form {
                        display: flex;
                        flex-direction: column;
                        gap: 12px;
                        padding: 20px;
                        background-color: var(--bg-alt);
                        border: 1px solid var(--border);
                        border-radius: 8px;
                        margin: 10px;
                        max-width: 400px;
                    }
                    input[type="file"] {
                        color: var(--fg);
                        padding: 8px;
                    }
                    input[type="submit"] {
                        padding: 10px 20px;
                        background-color: var(--primary);
                        color: white;
                        border: none;
                        border-radius: 4px;
                        cursor: pointer;
                        font-weight: 500;
                        transition: background-color 0.2s ease;
                    }
                    input[type="submit"]:hover {
                        background-color: var(--primary-hover);
                    }
                    .loading {
                        display: none;
                        margin: 20px;
                    }
                    .spinner {
                        border: 4px solid var(--border);
                        border-top: 4px solid var(--primary);
                        border-radius: 50%;
                        width: 40px;
                        height: 40px;
                        animation: spin 1s linear infinite;
                        margin: 0 auto;
                    }
                    @keyframes spin {
                        0% { transform: rotate(0deg); }
                        100% { transform: rotate(360deg); }
                    }
                </style>
            </head>
            <body>
                <div class="hero">
                    <h1>🏋️ KaiserLift</h1>
                    <p class="tagline">Never guess your next workout again</p>
                    <button class="demo-button" onclick="loadDemo()">
                        ✨ Try Demo with Sample Data
                    </button>
                </div>

                <div class="upload-section">
                    <h2>Or Upload Your Own Data</h2>

                    <h3>Lifting Data</h3>
                    <form action="/upload" method="post" enctype="multipart/form-data">
                        <input type="file" name="file" accept=".csv" />
                        <input type="submit" value="Upload Lifting Data" />
                    </form>

                    <h3>Running Data</h3>
                    <form action="/upload-running" method="post" enctype="multipart/form-data">
                        <input type="file" name="file" accept=".csv" />
                        <input type="submit" value="Upload Running Data" />
                    </form>
                </div>

                <div class="loading" id="loading">
                    <div class="spinner"></div>
                    <p>Loading demo data...</p>
                </div>

                <div id="result"></div>

                <script>
                async function loadDemo() {
                    const loading = document.getElementById('loading');
                    loading.style.display = 'block';

                    try {
                        const response = await fetch('/demo-data');
                        const html = await response.text();
                        document.body.innerHTML = html;
                    } catch (error) {
                        alert('Failed to load demo data: ' + error.message);
                    } finally {
                        loading.style.display = 'none';
                    }
                }
                </script>
            </body>
        </html>
        """,
    )


@app.post("/upload", response_class=HTMLResponse)
async def upload(file: UploadFile = File(...)) -> str:
    """Process the uploaded CSV via the core pipeline and return HTML.

    All numerical computations occur within :func:`pipeline`. Any JavaScript
    embedded in the resulting HTML is dedicated solely to updating the user
    interface and performs no calculations.
    """

    # ``pipeline`` no longer embeds required JavaScript and CSS assets by
    # default.  Explicitly enable ``embed_assets`` so the returned HTML is a
    # standalone page suitable for direct rendering by the browser.
    return pipeline([file.file], embed_assets=True)


@app.post("/upload-running", response_class=HTMLResponse)
async def upload_running(file: UploadFile = File(...)) -> str:
    """Process the uploaded running CSV via the running pipeline and return HTML.

    Processes running/cardio data with distance and pace metrics.
    Returns interactive HTML with Pareto front analysis and target recommendations.
    """

    return running_pipeline([file.file], embed_assets=True)


@app.get("/demo-data", response_class=HTMLResponse)
async def demo_data() -> str:
    """Serve pre-generated demo HTML with sample lifting data.

    This endpoint loads the example FitNotes CSV file from the test fixtures
    and processes it through the lifting pipeline to generate an interactive
    HTML demo that users can explore without uploading their own data.
    """

    import io
    from pathlib import Path

    # Load example CSV from test fixtures
    example_csv = (
        Path(__file__).parent.parent
        / "tests/example_use/FitNotes_Export_2025_05_21_08_39_11.csv"
    )

    if not example_csv.exists():
        return HTMLResponse(
            "<html><body><h1>Demo data not found</h1>"
            "<p>The example CSV file is not available in this deployment.</p></body></html>",
            status_code=404,
        )

    with open(example_csv, encoding="utf-8") as f:
        csv_content = f.read()

    return pipeline([io.StringIO(csv_content)], embed_assets=True)


def main() -> None:
    """Start a Uvicorn development server."""

    import uvicorn

    port = int(os.getenv("PORT", 8000))
    uvicorn.run("kaiserlift.webapp:app", host="0.0.0.0", port=port, reload=False)


if __name__ == "__main__":  # pragma: no cover - manual server start
    main()
