import argparse
import logging
import sys
from typing import Optional, List, Dict, Union, Tuple
from .prepdir_processor import PrepdirProcessor
from .prepdir_file_entry import PrepdirFileEntry
from .prepdir_output_file import PrepdirOutputFile
from .config import __version__, load_config
from prepdir import prepdir_logging

logger = logging.getLogger(__name__)


def run(
    directory: str = ".",
    extensions: Optional[List[str]] = None,
    specific_files: Optional[List[str]] = None,
    output_file: Optional[str] = None,
    config_path: Optional[str] = None,
    scrub_hyphenated_uuids: Optional[bool] = None,
    scrub_hyphenless_uuids: Optional[bool] = None,
    replacement_uuid: Optional[str] = None,
    use_unique_placeholders: bool = False,
    ignore_exclusions: bool = False,
    include_prepdir_files: bool = False,
    quiet: bool = False,
) -> Union[PrepdirOutputFile, Tuple[str, Dict[str, str], List[PrepdirFileEntry], Dict]]:
    """
    Run prepdir to generate a formatted string of file contents.

    Args:
        directory: Starting directory path.
        extensions: List of file extensions to include (without the dot).
        specific_files: List of specific file paths to process.
        output_file: Path to save the output file.
        config_path: Path to custom config file.
        scrub_hyphenated_uuids: If True, scrub (hyphenated) UUIDs in file contents.
        scrub_hyphenless_uuids: If True, scrub hyphen-less UUIDs.
        replacement_uuid: UUID to replace detected UUIDs with when use_unique_placeholders is False.
        use_unique_placeholders: If True, replace UUIDs with unique placeholders.
        ignore_exclusions: If True, ignore exclusion lists.
        include_prepdir_files: If True, include files previously generated by prepdir.
        quiet: If True, suppress user-facing output.
    Returns:
        PrepdirOutputFile or tuple of (content, UUID mapping, list of PrepdirFileEntry objects, metadata).
    """
    if not quiet:
        print(f"Starting prepdir in {directory}")
    logger.info(f"Starting prepdir in {directory}")
    logger.debug("replacement_uuid is '%s', use_unique_placeholders is %s", replacement_uuid, use_unique_placeholders)

    processor = PrepdirProcessor(
        directory=directory,
        extensions=extensions,
        specific_files=specific_files,
        output_file=output_file,
        config_path=config_path,
        scrub_hyphenated_uuids=scrub_hyphenated_uuids,
        scrub_hyphenless_uuids=scrub_hyphenless_uuids,
        replacement_uuid=replacement_uuid,
        use_unique_placeholders=use_unique_placeholders,
        ignore_exclusions=ignore_exclusions,
        include_prepdir_files=include_prepdir_files,
        quiet=quiet,
    )

    output = processor.generate_output()

    if not output_file:
        print(output.content)
    else:
        processor.save_output(output, output_file)
        if not quiet:
            print(f"Saved output to {output_file}")

    return output


def main():
    """Command-line interface for prepdir."""
    parser = argparse.ArgumentParser(description="Directory traversal utility to prepare project contents for review.")
    parser.add_argument(
        "directory",
        nargs="?",
        default=".",
        help="Directory to process (default: current directory)",
    )
    parser.add_argument(
        "-e",
        "--extensions",
        nargs="*",
        help="File extensions to include (e.g., py txt)",
    )
    parser.add_argument(
        "-f",
        "--files",
        nargs="*",
        help="Specific files to process (relative or absolute paths)",
    )
    parser.add_argument(
        "-o",
        "--output",
        help="Output file name (default: prepped_dir.txt)",
        default="prepped_dir.txt",
    )
    parser.add_argument(
        "--init",
        action="store_true",
        help="Initialize a local config.yaml",
    )
    parser.add_argument(
        "--config",
        help="Path to a custom config file",
    )
    parser.add_argument(
        "--force",
        action="store_true",
        help="Overwrite existing config file with --init",
    )
    parser.add_argument(
        "--no-scrub-uuids",
        action="store_true",
        help="Disable all UUID scrubbing",
    )
    parser.add_argument(
        "--no-scrub-hyphenated-uuids",
        action="store_true",
        help="Disable hyphenated UUID scrubbing",
    )
    parser.add_argument(
        "--no-scrub-hyphenless-uuids",
        action="store_true",
        help="Disable hyphen-less UUID scrubbing",
    )
    parser.add_argument(
        "--replacement-uuid",
        help="Custom UUID to replace detected UUIDs",
    )
    parser.add_argument(
        "--use-unique-placeholders",
        action="store_true",
        help="Replace UUIDs with unique placeholders (e.g., PREPDIR_UUID_PLACEHOLDER_n)",
    )
    parser.add_argument(
        "--all",
        action="store_true",
        help="Include all files, ignoring exclusions",
    )
    parser.add_argument(
        "--include-prepdir-files",
        action="store_true",
        help="Include prepdir-generated files",
    )
    parser.add_argument(
        "-v",
        "--verbose",
        action="count",
        default=0,
        help="Increase verbosity: -v for INFO, -vv for DEBUG",
    )
    parser.add_argument(
        "-q",
        "--quiet",
        action="store_true",
        help="Suppress user-facing output",
    )
    parser.add_argument(
        "--version",
        action="version",
        help="Show version and exit",
        version=f"prepdir {__version__}",
    )

    args = parser.parse_args()

    # Determine logging level: 
    LEVEL_MAP = {0: logging.WARNING, 1: logging.INFO, 2: logging.DEBUG}
    if args.verbose > 0:
        logging_level = LEVEL_MAP.get(args.verbose, logging.WARNING)
    else:
        logging_level = logging.WARNING

    # Configure logging for the 'prepdir' logger to ensure all child loggers inherit
    prepdir_logger = logging.getLogger('prepdir')
    prepdir_logging.configure_logging(prepdir_logger, level=logging_level)
    logging.getLogger("dynaconf").setLevel(logging.CRITICAL)  # Suppress dynaconf logging
    logger.debug("args are: %s", args)

    if args.init:
        try:
            PrepdirProcessor.init_config(config_path=args.config, force=args.force, quiet=args.quiet)
        except SystemExit as e:
            sys.exit(e.code)
        return

    if args.no_scrub_uuids:
        scrub_hyphenated_uuids = False
        scrub_hyphenless_uuids = False
    else:
        scrub_hyphenated_uuids = not args.no_scrub_hyphenated_uuids
        scrub_hyphenless_uuids = not args.no_scrub_hyphenless_uuids

    try:
        output = run(
            directory=args.directory,
            extensions=args.extensions,
            specific_files=args.files,
            output_file=args.output,
            config_path=args.config,
            scrub_hyphenated_uuids=scrub_hyphenated_uuids,
            scrub_hyphenless_uuids=scrub_hyphenless_uuids,
            replacement_uuid=args.replacement_uuid,
            use_unique_placeholders=args.use_unique_placeholders,
            ignore_exclusions=args.all,
            include_prepdir_files=args.include_prepdir_files,
            quiet=args.quiet,
        )
    except ValueError as e:
        print(f"Error: {str(e)}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Unexpected error: {str(e)}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
