from pathlib import Path
from typing import List, Optional, Iterator, Dict
import os
import logging
import re
from datetime import datetime
from io import StringIO
from contextlib import redirect_stdout
from dynaconf import Dynaconf
from prepdir.config import load_config, __version__, init_config
from prepdir.prepdir_file_entry import PrepdirFileEntry
from prepdir.prepdir_output_file import PrepdirOutputFile
from prepdir.scrub_uuids import HYPHENATED_UUID_PATTERN
from prepdir.is_excluded_file import is_excluded_dir, is_excluded_file
from prepdir.glob_translate import glob_translate

logger = logging.getLogger(__name__)


class PrepdirProcessor:
    """Manages generation and parsing of prepdir output files."""

    def __init__(
        self,
        directory: str,
        extensions: Optional[List[str]] = None,
        specific_files: Optional[List[str]] = None,
        output_file: Optional[str] = None,
        config_path: Optional[str] = None,
        scrub_hyphenated_uuids: Optional[bool] = None,
        scrub_hyphenless_uuids: Optional[bool] = None,
        replacement_uuid: Optional[str] = None,
        use_unique_placeholders: bool = False,
        ignore_exclusions: bool = False,
        include_prepdir_files: bool = False,
        quiet: bool = False,
    ):
        """
        Initialize the PrepdirProcessor with configuration options.

        Args:
            directory: Starting directory path.
            extensions: List of file extensions to include (without the dot).
            specific_files: List of specific file paths to process.
            output_file: Path to save the output file.
            config_path: Path to custom config file.
            scrub_hyphenated_uuids: If True, scrub hyphenated UUIDs in file contents.
            scrub_hyphenless_uuids: If True, scrub hyphen-less UUIDs.
            replacement_uuid: UUID to replace detected UUIDs with when use_unique_placeholders is False.
            use_unique_placeholders: If True, replace UUIDs with unique placeholders.
            ignore_exclusions: If True, ignore exclusion lists.
            include_prepdir_files: If True, include files previously generated by prepdir.
            quiet: If True, suppress user-facing output.

        Raises:
            ValueError: If directory is invalid or replacement_uuid is malformed.
        """
        self.logger = logging.getLogger(__name__)
        self.directory = os.path.abspath(directory)
        if not os.path.exists(self.directory):
            raise ValueError(f"Directory '{self.directory}' does not exist")
        if not os.path.isdir(self.directory):
            raise ValueError(f"'{self.directory}' is not a directory")

        self.config = self._load_config(config_path, quiet=quiet)
        self.extensions = extensions or self.config.get("DEFAULT_EXTENSIONS", [])
        self.specific_files = specific_files or []
        self.output_file = output_file or self.config.get("DEFAULT_OUTPUT_FILE", "prepped_dir.txt")
        self.ignore_exclusions = ignore_exclusions or self.config.get("IGNORE_EXCLUSIONS", False)
        self.include_prepdir_files = include_prepdir_files or self.config.get("INCLUDE_PREPDIR_FILES", False)
        self.quiet = quiet

        self.scrub_hyphenated_uuids = (
            scrub_hyphenated_uuids
            if scrub_hyphenated_uuids is not None
            else self.config.get("SCRUB_HYPHENATED_UUIDS", True)
        )
        self.scrub_hyphenless_uuids = (
            scrub_hyphenless_uuids
            if scrub_hyphenless_uuids is not None
            else self.config.get("SCRUB_HYPHENLESS_UUIDS", True)
        )

        if replacement_uuid is not None:
            if not isinstance(replacement_uuid, str):
                self.logger.error(
                    f"Invalid replacement UUID type: '{type(replacement_uuid)}'. Replacement UUID must be a string. Using config default."
                )
                replacement_uuid = None
            elif not re.fullmatch(HYPHENATED_UUID_PATTERN, replacement_uuid):
                self.logger.error(f"Invalid replacement UUID: '{replacement_uuid}'. Using config default.")
                replacement_uuid = None

        self.replacement_uuid = (
            replacement_uuid
            if replacement_uuid is not None
            else self.config.get("REPLACEMENT_UUID", "00000000-0000-0000-0000-000000000000")
        )
        self.use_unique_placeholders = use_unique_placeholders or self.config.get("USE_UNIQUE_PLACEHOLDERS", False)

        self._print_and_log(f"Generated timestamp: {datetime.now().isoformat()}")
        self._print_and_log(f"Traversing directory: {self.directory}")
        self._print_and_log(f"Extensions filter: {self.extensions if self.extensions else 'None'}")
        self._print_and_log(f"Output file: {self.output_file}")
        self._print_and_log(f"Ignoring exclusions: {self.ignore_exclusions}")

        if self.scrub_hyphenated_uuids:
            if self.use_unique_placeholders:
                self._print_and_log(
                    "Regular (hyphenated) UUIDs in file contents will be scrubbed and replaced with unique placeholders (e.g., PREPDIR_UUID_PLACEHOLDER_n)."
                )
            else:
                self._print_and_log(
                    f"Regular (hyphenated) UUIDs in file contents will be scrubbed and replaced with '{self.replacement_uuid}'."
                )
        if self.scrub_hyphenless_uuids:
            if self.use_unique_placeholders:
                self._print_and_log(
                    "Hyphen-less UUIDs in file contents will be scrubbed and replaced with unique placeholders (e.g., PREPDIR_UUID_PLACEHOLDER_n)."
                )

            else:
                self._print_and_log(
                    f"Hyphen-less UUIDs in file contents will be scrubbed and replaced with '{self.replacement_uuid.replace('-', '')}'."
                )

        self.excluded_dir_regexes = [
            re.compile(glob_translate(p)) for p in self.config.get("EXCLUDE", {}).get("DIRECTORIES", [])
        ]
        logger.debug(f"{self.excluded_dir_regexes=}")

        self.excluded_file_regexes = []
        self.excluded_file_recursive_glob_regexes = []
        for p in self.config.get("EXCLUDE", {}).get("FILES", []):
            if "**" in p:
                self.excluded_file_recursive_glob_regexes.append(re.compile(glob_translate(p)))
            else:
                self.excluded_file_regexes.append(re.compile(glob_translate(p)))

        logger.debug(f"{self.excluded_file_regexes=}")
        logger.debug(f"{self.excluded_file_recursive_glob_regexes=}")

    def _print_and_log(self, msg: str):
        """Helper routine to print a message and and log it at the INFO level"""
        self.logger.info(msg)
        if not self.quiet:
            print(msg)

    def _load_config(self, config_path: Optional[str], quiet: bool = False) -> Dynaconf:
        """
        Load configuration using Dynaconf.

        Args:
            config_path: Path to custom config file.
            quiet: If True, suppress user-facing output.

        Returns:
            Dynaconf: Configured Dynaconf instance.
        """
        return load_config("prepdir", config_path, quiet)

    def is_excluded_output_file(self, filename: str, root: str) -> bool:
        """
        Check if a file is an excluded prepdir output file.

        Args:
            filename: Name of the file to check.
            root: Directory containing the file.

        Returns:
            bool: True if the file is an excluded output file, False otherwise.
        """
        full_path = os.path.abspath(os.path.join(root, filename))
        if self.output_file and full_path == os.path.abspath(self.output_file):
            self.logger.debug(f"File {full_path} is excluded since it is the output file for this run")
            return True
        if self.include_prepdir_files:
            return False
        try:
            with open(full_path, "r", encoding="utf-8") as f:
                if PrepdirFileEntry.is_prepdir_outputfile_format(f.read(), file_full_path=full_path):
                    self.logger.debug(f"Found {full_path} is an output file")
                    return True
        except (IOError, UnicodeDecodeError):
            self.logger.debug(f"Could not read {full_path} - assuming it is NOT an output file")
            return False
        except Exception as e:
            self.logger.error(f"Could not read {full_path} - unexpected error")
            raise

        self.logger.debug(f"Found {full_path} is NOT an output file")
        return False

    def is_excluded_dir(self, dirname: str, root: str) -> bool:
        """
        Check if a directory is excluded based on config.

        Args:
            dirname: Name of the directory to check.
            root: Path to the directory.

        Returns:
            bool: True if the directory is excluded, False otherwise.
        """
        if self.ignore_exclusions:
            return False

        relative_path = os.path.relpath(os.path.join(root, dirname), self.directory)
        return is_excluded_dir(relative_path, excluded_dir_regexes=self.excluded_dir_regexes)

    def is_excluded_file(self, filename: str, root: str) -> bool:
        """
        Check if a file is excluded based on config.

        Args:
            filename: Name of the file to check.
            root: Directory containing the file.

        Returns:
            bool: True if the file is excluded, False otherwise.
        """
        if self.ignore_exclusions:
            return False

        relative_path = os.path.relpath(os.path.join(root, filename), self.directory)
        return is_excluded_file(
            relative_path,
            excluded_dir_regexes=self.excluded_dir_regexes,
            excluded_file_regexes=self.excluded_file_regexes,
            excluded_file_recursive_glob_regexes=self.excluded_file_recursive_glob_regexes,
        )

    def generate_output(self) -> PrepdirOutputFile:
        """
        Generate the prepdir output file content by traversing the directory or specific files.

        Returns:
            PrepdirOutputFile: Object containing the generated content, files, and metadata.

        Raises:
            ValueError: If no valid files are found.
        """
        output = StringIO()
        uuid_mapping: Dict[str, str] = {}
        placeholder_counter = 1
        timestamp_to_use = datetime.now().isoformat()
        metadata = {
            "version": __version__,
            "date": timestamp_to_use,
            "base_directory": self.directory,
            "creator": f"prepdir version {__version__} (pip install prepdir)",
        }

        with redirect_stdout(output):
            files_found = False
            print(f"File listing generated {timestamp_to_use} by prepdir version {__version__} (pip install prepdir)")
            print(f"Base directory is '{self.directory}'")
            if self.scrub_hyphenated_uuids:
                if self.use_unique_placeholders:
                    print(
                        "Note: Valid (hyphenated) UUIDs in file contents will be scrubbed and replaced with unique placeholders (e.g., PREPDIR_UUID_PLACEHOLDER_n)."
                    )
                else:
                    print(
                        f"Note: Valid (hyphenated) UUIDs in file contents will be scrubbed and replaced with '{self.replacement_uuid}'."
                    )
            if self.scrub_hyphenless_uuids:
                if self.use_unique_placeholders:
                    print(
                        "Note: Valid hyphen-less UUIDs in file contents will be scrubbed and replaced with unique placeholders (e.g., PREPDIR_UUID_PLACEHOLDER_n)."
                    )
                else:
                    print(
                        f"Note: Valid hyphen-less UUIDs in file contents will be scrubbed and replaced with '{self.replacement_uuid.replace('-', '')}'."
                    )

            file_iterator = self._traverse_specific_files() if self.specific_files else self._traverse_directory()
            for file_path in file_iterator:
                files_found = True
                self.logger.debug(f"adding file {file_path}")
                file_entry, updated_uuid_mapping, placeholder_counter = PrepdirFileEntry.from_file_path(
                    file_path=file_path,
                    base_directory=self.directory,
                    scrub_hyphenated_uuids=self.scrub_hyphenated_uuids,
                    scrub_hyphenless_uuids=self.scrub_hyphenless_uuids,
                    replacement_uuid=self.replacement_uuid,
                    use_unique_placeholders=self.use_unique_placeholders,
                    quiet=self.quiet,
                    placeholder_counter=placeholder_counter,
                    uuid_mapping=uuid_mapping,
                )
                print(file_entry.to_output())
                uuid_mapping.update(updated_uuid_mapping)

            if not files_found:
                if self.specific_files:
                    self._print_and_log("No valid or accessible files found from the provided list.")
                elif self.extensions:
                    self._print_and_log(f"No files with extension(s) {', '.join(self.extensions)} found.")
                else:
                    self._print_and_log("No files found.")

                raise ValueError("No files found!")

        content = output.getvalue()
        return PrepdirOutputFile.from_content(
            content=content,
            path_obj=Path(self.output_file) if self.output_file else None,
            uuid_mapping=uuid_mapping,
            metadata=metadata,
            use_unique_placeholders=self.use_unique_placeholders,
        )

    def _traverse_specific_files(self) -> Iterator[Path]:
        """
        Traverse specific files provided in the configuration.

        Yields:
            Path: Paths to valid files that pass exclusion checks.
        """
        for file_path in self.specific_files:
            path = Path(file_path)
            if not path.is_absolute():
                path = Path(self.directory) / path
            try:
                path = path.resolve()
                if not path.exists():
                    self.logger.warning(f"File '{file_path}' does not exist")
                    continue
                if not path.is_file():
                    self.logger.warning(f"'{file_path}' is not a file")
                    continue
                if not self.ignore_exclusions:
                    if self.is_excluded_dir(path.parent.name, str(path.parent)):
                        self.logger.info(f"Skipping file '{file_path}' (parent directory excluded)")
                        continue
                    if self.is_excluded_file(path.name, str(path.parent)):
                        self.logger.info(f"Skipping file '{file_path}' (excluded in config)")
                        continue

                if self.is_excluded_output_file(path.name, str(path.parent)):
                    self.logger.info(f"Skipping file: {file_path} (excluded prepdir output file)")
                    continue

            except PermissionError as e:
                self.logger.warning(f"Permission denied accessing '{file_path}': {str(e)}")

            except Exception as e:
                logger.exception(f"Issue accessing '{file_path}': {str(e)}")
                continue

            self.logger.debug(f"Will include file at {path}")
            yield path

    def _traverse_directory(self) -> Iterator[Path]:
        self.logger.debug(f"traversing {self.directory}")
        try:
            file_count_checked = 0
            file_count_included = 0
            for root, dirnames, filenames in sorted(os.walk(self.directory)):
                # Check if the current directory is excluded
                relative_root = os.path.relpath(root, self.directory)
                if self.is_excluded_dir(relative_root, root):
                    self.logger.debug(f"Skipping directory: {root} (excluded in config)")
                    dirnames[:] = []  # Prevent further recursion
                    continue
                # Filter subdirectories to avoid recursion into excluded ones
                dirnames[:] = [d for d in dirnames if not self.is_excluded_dir(d, root)]
                for filename in sorted(filenames):
                    file_count_checked += 1
                    #self.logger.debug(f"Processing file {file_count}: {filename}")
                    if self.extensions and not any(filename.endswith(f".{ext}") for ext in self.extensions):
                        self.logger.info(f"Skipping file: {filename} (extension not in {self.extensions})")
                        continue
                    if self.is_excluded_output_file(filename, root):
                        self.logger.info(f"Skipping file: {filename} (excluded output file)")
                        continue
                    if self.is_excluded_file(filename, root):
                        self.logger.info(f"Skipping file: {filename} (excluded in config)")
                        continue
                    path = Path(root) / filename
                    file_count_included += 1
                    self.logger.debug(f"Will include file at {path} (included:{file_count_included}, checked:{file_count_checked})")
                    yield path
        except PermissionError as e:
            self.logger.warning(f"Permission denied traversing directory '{self.directory}': {str(e)}")
            return
        except Exception as e:
            logger.exception(f"Issue accessing '{self.directory}': {str(e)}")
            return

    def save_output(self, output: PrepdirOutputFile, path: Optional[str] = None) -> None:
        """
        Save the prepdir output to a file.

        Args:
            output: PrepdirOutputFile object to save.
            path: Optional path to save the file (overrides self.output_file).

        Raises:
            ValueError: If no output path is specified or saving fails.
        """
        output.save(path)

    def validate_output(
        self,
        content: Optional[str] = None,
        file_path: Optional[str] = None,
        metadata: Optional[Dict[str, str]] = None,
        highest_base_directory: Optional[str] = None,
        validate_files_exist: bool = False,
    ) -> PrepdirOutputFile:
        """
        Validate a prepdir output file or content.

        Args:
            content: Raw content to validate.
            file_path: Path to the output file to validate.
            metadata: Optional metadata to include in the output.
            highest_base_directory: Directory to check file paths against.
            validate_files_exist: If True, check if files exist in the filesystem.

        Returns:
            PrepdirOutputFile: Validated output object.

        Raises:
            ValueError: If content or file_path is invalid or paths are outside the base directory.
        """
        if content is not None and file_path is not None:
            raise ValueError("Cannot provide both content and file_path")
        if content is None and file_path is None:
            raise ValueError("Must provide either content or file_path")
        default_metadata = {
            "base_directory": self.directory,
            "version": __version__,
            "date": datetime.now().isoformat(),
            "creator": f"prepdir version {__version__}",
        }
        if metadata:
            default_metadata.update(metadata)
        highest_base = Path(highest_base_directory or self.directory).resolve()
        try:
            if file_path:
                output = PrepdirOutputFile.from_file(
                    path=Path(file_path),
                    metadata=default_metadata,
                    use_unique_placeholders=self.use_unique_placeholders,
                )
            else:
                output = PrepdirOutputFile.from_content(
                    content=content,
                    path_obj=None,
                    metadata=default_metadata,
                    use_unique_placeholders=self.use_unique_placeholders,
                )
            base_dir = Path(output.metadata["base_directory"]).resolve()
            try:
                base_dir.relative_to(highest_base)
            except ValueError:
                raise ValueError(f"Base directory '{base_dir}' is outside highest base directory '{highest_base}'")
            for entry in output.files.values():
                abs_path = entry.absolute_path.resolve()
                try:
                    abs_path.relative_to(highest_base)
                except ValueError:
                    raise ValueError(f"File path '{abs_path}' is outside highest base directory '{highest_base}'")
                if validate_files_exist and not abs_path.exists():
                    self.logger.warning(f"File {abs_path} does not exist in filesystem")
            return output
        except ValueError as e:
            raise ValueError(f"Invalid prepdir output: {str(e)}") from e
        except Exception as e:
            raise ValueError(f"Unexpected exception: {str(e)}") from e

    @staticmethod
    def init_config(config_path: str = ".prepdir/config.yaml", force: bool = False, quiet: bool = False) -> None:
        """
        Initialize a local config.yaml with the package's default config.

        Args:
            config_path: Path to save the config file.
            force: If True, overwrite existing config file.
            quiet: If True, suppress user-facing output.

        Raises:
            SystemExit: If initialization fails or config file already exists without force=True.
        """
        init_config("prepdir", config_path, force, quiet)
