"""
Minimal configuration for benchmarks.
Only contains human-written metadata that cannot be extracted from code.
Everything else (epochs, temperature, etc.) comes from the actual task definitions.
"""

from dataclasses import dataclass
from functools import lru_cache
import importlib
import importlib.util
import sys
import uuid
from pathlib import Path
from types import ModuleType
from typing import Callable, Iterable, List, Optional

from importlib.metadata import entry_points
import logging

from openbench.utils import BenchmarkMetadata

logger = logging.getLogger(__name__)


def _load_entry_point_benchmarks() -> dict[str, BenchmarkMetadata]:
    """Load benchmarks registered via entry points.

    External packages can register benchmarks by adding to their pyproject.toml:

    [project.entry-points."openbench.benchmarks"]
    my_benchmark = "my_package.benchmarks:get_benchmark_metadata"

    The entry point should return either:
    - A BenchmarkMetadata instance (registered with the entry point name)
    - A dict[str, BenchmarkMetadata] (multiple benchmarks)
    """
    discovered = {}

    try:
        eps = entry_points()
        benchmark_eps = eps.select(group="openbench.benchmarks")

        for ep in benchmark_eps:
            try:
                loaded = ep.load()

                # If the loaded object is callable, call it to get the result
                if callable(loaded):
                    result = loaded()
                else:
                    result = loaded

                if isinstance(result, BenchmarkMetadata):
                    # Single benchmark registered with entry point name
                    discovered[ep.name] = result
                elif isinstance(result, dict):
                    # Multiple benchmarks
                    for key, value in result.items():
                        if not isinstance(value, BenchmarkMetadata):
                            logger.warning(
                                f"Entry point '{ep.name}' returned non-BenchmarkMetadata "
                                f"value for key '{key}', skipping"
                            )
                            continue
                        discovered[key] = value
                else:
                    logger.warning(
                        f"Entry point '{ep.name}' returned unexpected type "
                        f"{type(result).__name__}, expected BenchmarkMetadata or dict"
                    )
            except Exception as e:
                # Log loading errors but don't fail
                logger.warning(
                    f"Failed to load benchmark from entry point '{ep.name}': {e}"
                )
    except Exception as e:
        # If entry_points() itself fails, log but continue
        logger.warning(f"Failed to load entry points: {e}")

    return discovered


@dataclass
class EvalGroup:
    """Group configuration for running multiple benchmarks as a group."""

    name: str  # Human-readable display name
    description: str  # Description of the group
    benchmarks: List[str]  # List of benchmark IDs to run


# Built-in benchmark metadata - minimal, no duplication
_BUILTIN_BENCHMARKS = {
    "mbpp": BenchmarkMetadata(
        name="MBPP",
        description="Mostly Basic Python Problems — code generation tasks with unit test verification",
        category="core",
        tags=["code", "generation", "sandbox", "reasoning"],
        module_path="openbench.evals.mbpp",
        function_name="mbpp",
        is_alpha=False,
    ),
    # Graphwalks benchmarks (alpha)
    "clockbench": BenchmarkMetadata(
        name="ClockBench",
        description="Clock benchmark - time-based reasoning tasks",
        category="community",
        tags=["time", "analog", "clock", "reasoning"],
        module_path="openbench.evals.clockbench",
        function_name="clockbench",
        is_alpha=False,
    ),
    "graphwalks": BenchmarkMetadata(
        name="GraphWalks",
        description="Multi-hop reasoning on graphs - both BFS and parent finding tasks",
        category="core",
        tags=["long-context", "graphs", "reasoning", "alpha"],
        module_path="openbench.evals.graphwalks",
        function_name="graphwalks",
        is_alpha=True,
    ),
    "graphwalks_bfs": BenchmarkMetadata(
        name="GraphWalks BFS",
        description="Multi-hop reasoning on graphs - BFS traversal tasks only",
        category="core",
        tags=["long-context", "graphs", "reasoning", "bfs", "alpha"],
        module_path="openbench.evals.graphwalks",
        function_name="graphwalks_bfs",
        is_alpha=True,
    ),
    "graphwalks_parents": BenchmarkMetadata(
        name="GraphWalks Parents",
        description="Multi-hop reasoning on graphs - parent finding tasks only",
        category="core",
        tags=["long-context", "graphs", "reasoning", "parents", "alpha"],
        module_path="openbench.evals.graphwalks",
        function_name="graphwalks_parents",
        is_alpha=True,
    ),
    # Core benchmarks
    "mmlu": BenchmarkMetadata(
        name="MMLU (cais/mmlu)",
        description="Massive Multitask Language Understanding - 57 academic subjects from the cais/mmlu dataset. Only supports English (EN-US).",
        category="core",
        tags=["multiple-choice", "knowledge", "reasoning", "multitask"],
        module_path="openbench.evals.mmlu",
        function_name="mmlu",
    ),
    "mmlu-pro": BenchmarkMetadata(
        name="MMLU Pro (TIGER-Lab)",
        description="Enhanced version of MMLU with more challenging, reasoning-focused questions.",
        category="core",
        tags=["multiple-choice", "knowledge", "reasoning", "multitask"],
        module_path="openbench.evals.mmlu_pro",
        function_name="mmlu_pro",
    ),
    "mmmlu": BenchmarkMetadata(
        name="MMMLU (openai/MMMLU)",
        description="MMLU translated to 15 languages.",
        category="core",
        tags=["multiple-choice", "knowledge", "reasoning", "multitask"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu",
    ),
    # MMMLU language-specific tasks
    "mmmlu_ar_xy": BenchmarkMetadata(
        name="MMMLU (Arabic)",
        description="MMLU in Arabic (AR_XY)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_ar_xy",
        subtask=True,
    ),
    "mmmlu_bn_bd": BenchmarkMetadata(
        name="MMMLU (Bengali)",
        description="MMLU in Bengali (BN_BD)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_bn_bd",
        subtask=True,
    ),
    "mmmlu_de_de": BenchmarkMetadata(
        name="MMMLU (German)",
        description="MMLU in German (DE_DE)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_de_de",
        subtask=True,
    ),
    "mmmlu_es_la": BenchmarkMetadata(
        name="MMMLU (Spanish)",
        description="MMLU in Spanish Latin America (ES_LA)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_es_la",
        subtask=True,
    ),
    "mmmlu_fr_fr": BenchmarkMetadata(
        name="MMMLU (French)",
        description="MMLU in French (FR_FR)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_fr_fr",
        subtask=True,
    ),
    "mmmlu_hi_in": BenchmarkMetadata(
        name="MMMLU (Hindi)",
        description="MMLU in Hindi (HI_IN)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_hi_in",
        subtask=True,
    ),
    "mmmlu_id_id": BenchmarkMetadata(
        name="MMMLU (Indonesian)",
        description="MMLU in Indonesian (ID_ID)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_id_id",
        subtask=True,
    ),
    "mmmlu_it_it": BenchmarkMetadata(
        name="MMMLU (Italian)",
        description="MMLU in Italian (IT_IT)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_it_it",
        subtask=True,
    ),
    "mmmlu_ja_jp": BenchmarkMetadata(
        name="MMMLU (Japanese)",
        description="MMLU in Japanese (JA_JP)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_ja_jp",
        subtask=True,
    ),
    "mmmlu_ko_kr": BenchmarkMetadata(
        name="MMMLU (Korean)",
        description="MMLU in Korean (KO_KR)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_ko_kr",
        subtask=True,
    ),
    "mmmlu_pt_br": BenchmarkMetadata(
        name="MMMLU (Portuguese)",
        description="MMLU in Portuguese Brazil (PT_BR)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_pt_br",
        subtask=True,
    ),
    "mmmlu_zh_cn": BenchmarkMetadata(
        name="MMMLU (Chinese)",
        description="MMLU in Chinese (ZH_CN)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_zh_cn",
        subtask=True,
    ),
    "mmmlu_sw_ke": BenchmarkMetadata(
        name="MMMLU (Swahili)",
        description="MMLU in Swahili (SW_KE)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_sw_ke",
        subtask=True,
    ),
    "mmmlu_yo_ng": BenchmarkMetadata(
        name="MMMLU (Yoruba)",
        description="MMLU in Yoruba (YO_NG)",
        category="core",
        tags=["multiple-choice", "knowledge", "multilingual", "mmmlu"],
        module_path="openbench.evals.mmmlu",
        function_name="mmmlu_yo_ng",
        subtask=True,
    ),
    "openai_mrcr": BenchmarkMetadata(
        name="OpenAI MRCR (Full)",
        description="Memory-Recall with Contextual Retrieval - long-context evaluation that measures recall of 2, 4, and 8 needles across million-token contexts",
        category="core",
        tags=["long-context", "retrieval", "needle", "sequence-matching"],
        module_path="openbench.evals.mrcr",
        function_name="openai_mrcr",
    ),
    "openai_mrcr_2n": BenchmarkMetadata(
        name="OpenAI MRCR (2 Needles)",
        description="Memory-Recall with Contextual Retrieval - long-context evaluation that measures recall of 2 needles across million-token contexts",
        category="core",
        tags=["long-context", "retrieval", "needle", "sequence-matching"],
        module_path="openbench.evals.mrcr",
        function_name="openai_mrcr_2n",
        subtask=True,
    ),
    "openai_mrcr_4n": BenchmarkMetadata(
        name="OpenAI MRCR (4 Needles)",
        description="Memory-Recall with Contextual Retrieval - long-context evaluation that measures recall of 4 needles across million-token contexts",
        category="core",
        tags=["long-context", "retrieval", "needle", "sequence-matching"],
        module_path="openbench.evals.mrcr",
        function_name="openai_mrcr_4n",
        subtask=True,
    ),
    "openai_mrcr_8n": BenchmarkMetadata(
        name="OpenAI MRCR (8 Needles)",
        description="Memory-Recall with Contextual Retrieval - long-context evaluation that measures recall of 8 needles across million-token contexts",
        category="core",
        tags=["long-context", "retrieval", "needle", "sequence-matching"],
        module_path="openbench.evals.mrcr",
        function_name="openai_mrcr_8n",
        subtask=True,
    ),
    "gpqa_diamond": BenchmarkMetadata(
        name="GPQA Diamond",
        description="Graduate-level Google-Proof Q&A in biology, chemistry, and physics",
        category="core",
        tags=["multiple-choice", "science", "graduate-level"],
        module_path="openbench.evals.gpqa_diamond",
        function_name="gpqa_diamond",
    ),
    "gpqa": BenchmarkMetadata(
        name="GPQA",
        description="Graduate-level science questions (multiple choice) across physics, chemistry, and biology",
        category="core",
        tags=["multiple-choice", "science", "graduate-level", "reasoning"],
        module_path="openbench.evals.gpqa",
        function_name="gpqa",
    ),
    "humaneval": BenchmarkMetadata(
        name="HumanEval",
        description="Code generation benchmark with 164 programming problems",
        category="core",
        tags=["coding", "generation", "execution"],
        module_path="openbench.evals.humaneval",
        function_name="humaneval",
    ),
    # Exercism benchmarks
    "exercism": BenchmarkMetadata(
        name="Exercism",
        description="Multi-language coding benchmark with real-world programming exercises across Python, Go, JavaScript, Java, and Rust",
        category="core",
        tags=["coding", "multi-language", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism",
    ),
    "exercism_python": BenchmarkMetadata(
        name="Exercism (Python)",
        description="Python coding tasks from the Exercism benchmark",
        category="core",
        tags=["coding", "python", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism_python",
        subtask=True,
    ),
    "exercism_javascript": BenchmarkMetadata(
        name="Exercism (JavaScript)",
        description="JavaScript coding tasks from the Exercism benchmark",
        category="core",
        tags=["coding", "javascript", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism_javascript",
        subtask=True,
    ),
    "exercism_go": BenchmarkMetadata(
        name="Exercism (Go)",
        description="Go coding tasks from the Exercism benchmark",
        category="core",
        tags=["coding", "go", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism_go",
        subtask=True,
    ),
    "exercism_java": BenchmarkMetadata(
        name="Exercism (Java)",
        description="Java coding tasks from the Exercism benchmark",
        category="core",
        tags=["coding", "java", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism_java",
        subtask=True,
    ),
    "exercism_rust": BenchmarkMetadata(
        name="Exercism (Rust)",
        description="Rust coding tasks from the Exercism benchmark",
        category="core",
        tags=["coding", "rust", "execution", "docker"],
        module_path="openbench.evals.exercism.exercism",
        function_name="exercism_rust",
        subtask=True,
    ),
    "ifeval": BenchmarkMetadata(
        name="Instruction Following",
        description="Tests ability to follow specific formatting and content constraints with both strict and loose evaluation metrics",
        category="core",
        tags=["instruction-following", "constraints", "formatting"],
        module_path="openbench.evals.ifeval",
        function_name="ifeval",
    ),
    "openbookqa": BenchmarkMetadata(
        name="OpenBookQA",
        description="Elementary-level science questions probing understanding of core facts",
        category="core",
        tags=["multiple-choice", "science", "elementary", "open-book"],
        module_path="openbench.evals.openbookqa",
        function_name="openbookqa",
    ),
    "musr": BenchmarkMetadata(
        name="MuSR",
        description="Testing the Limits of Chain-of-thought with Multistep Soft Reasoning - includes murder mysteries, object placements, and team allocation tasks",
        category="core",
        tags=["multiple-choice", "reasoning", "commonsense", "chain-of-thought"],
        module_path="openbench.evals.musr",
        function_name="musr",
    ),
    "musr_murder_mysteries": BenchmarkMetadata(
        name="MuSR Murder Mysteries",
        description="MuSR murder mystery scenarios - who is the most likely murderer?",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "commonsense",
            "chain-of-thought",
            "murder-mysteries",
        ],
        module_path="openbench.evals.musr",
        function_name="musr_murder_mysteries",
        subtask=True,
    ),
    "musr_object_placements": BenchmarkMetadata(
        name="MuSR Object Placements",
        description="MuSR object placement reasoning - where would someone look for an object?",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "commonsense",
            "chain-of-thought",
            "object-placements",
        ],
        module_path="openbench.evals.musr",
        function_name="musr_object_placements",
        subtask=True,
    ),
    "musr_team_allocation": BenchmarkMetadata(
        name="MuSR Team Allocation",
        description="MuSR team allocation problems - how to allocate people to tasks efficiently?",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "commonsense",
            "chain-of-thought",
            "team-allocation",
        ],
        module_path="openbench.evals.musr",
        function_name="musr_team_allocation",
        subtask=True,
    ),
    "supergpqa": BenchmarkMetadata(
        name="SuperGPQA",
        description="Scaling LLM Evaluation across 285 Graduate Disciplines - 26,529 multiple-choice questions across science, engineering, medicine, economics, and philosophy",
        category="core",
        tags=["multiple-choice", "knowledge", "graduate-level", "multidisciplinary"],
        module_path="openbench.evals.supergpqa",
        function_name="supergpqa",
    ),
    "simpleqa": BenchmarkMetadata(
        name="SimpleQA",
        description="Measuring short-form factuality in large language models with simple Q&A pairs",
        category="core",
        tags=["factuality", "question-answering", "graded"],
        module_path="openbench.evals.simpleqa",
        function_name="simpleqa",
    ),
    "tumlu": BenchmarkMetadata(
        name="TUMLU",
        description="TUMLU is a comprehensive, multilingual, and natively developed language understanding benchmark specifically designed for Turkic languages.",
        category="community",
        tags=["factuality", "question-answering", "multiple-choice", "reasoning"],
        module_path="openbench.evals.tumlu",
        function_name="tumlu",
    ),
    "detailbench": BenchmarkMetadata(
        name="DetailBench",
        description="Tests whether LLMs notify users about wrong facts in a text while they are tasked to translate said text",
        category="community",
        tags=["knowledge", "graded", "instruction-following"],
        module_path="openbench.evals.detailbench",
        function_name="detailbench",
    ),
    "browsecomp": BenchmarkMetadata(
        name="BrowseComp",
        description="A Simple Yet Challenging Benchmark for Browsing Agents - evaluates model performance on browsing-related tasks",
        category="core",
        tags=["browsing", "web", "reasoning", "graded"],
        module_path="openbench.evals.browsecomp",
        function_name="browsecomp",
    ),
    "hle": BenchmarkMetadata(
        name="Humanity's Last Exam",
        description="Multi-modal benchmark at the frontier of human knowledge - 2,500 questions across mathematics, humanities, and natural sciences designed by subject-matter experts globally",
        category="core",
        tags=["knowledge", "reasoning", "multi-modal", "graded", "frontier"],
        module_path="openbench.evals.hle",
        function_name="hle",
    ),
    "hle_text": BenchmarkMetadata(
        name="Humanity's Last Exam (Text-Only)",
        description="Text-only variant of HLE with multi-modal questions filtered out - evaluates models without vision capabilities on text-based questions from the frontier of human knowledge",
        category="core",
        tags=["knowledge", "reasoning", "text-only", "graded", "frontier"],
        module_path="openbench.evals.hle",
        function_name="hle_text",
        subtask=True,
    ),
    "mmstar": BenchmarkMetadata(
        name="MMStar",
        description="MMStar benchmark for measuring multi-modal gain and leakage via coordinated vision and text ablations",
        category="core",
        tags=["vision", "multi-modal", "leakage", "perception", "reasoning"],
        module_path="openbench.evals.mmstar",
        function_name="mmstar",
    ),
    "healthbench": BenchmarkMetadata(
        name="HealthBench",
        description="Medical dialogue evaluation using physician-created rubrics for assessing healthcare conversations",
        category="core",
        tags=["medical", "dialogue", "graded", "rubric-based"],
        module_path="openbench.evals.healthbench",
        function_name="healthbench",
    ),
    "healthbench_hard": BenchmarkMetadata(
        name="HealthBench Hard",
        description="Most challenging medical dialogue cases from HealthBench requiring nuanced medical knowledge",
        category="core",
        tags=["medical", "dialogue", "graded", "rubric-based", "hard"],
        module_path="openbench.evals.healthbench",
        function_name="healthbench_hard",
        subtask=True,
    ),
    "healthbench_consensus": BenchmarkMetadata(
        name="HealthBench Consensus",
        description="Medical dialogue cases with strong physician consensus on appropriate responses",
        category="core",
        tags=["medical", "dialogue", "graded", "rubric-based", "consensus"],
        module_path="openbench.evals.healthbench",
        function_name="healthbench_consensus",
        subtask=True,
    ),
    "mgsm": BenchmarkMetadata(
        name="MGSM",
        description="Multilingual Grade School Math benchmark across 11 languages for testing mathematical reasoning",
        category="core",
        tags=["math", "multilingual", "reasoning", "chain-of-thought"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm",
    ),
    "mgsm_en": BenchmarkMetadata(
        name="MGSM English",
        description="Grade school math problems in English for testing mathematical reasoning",
        category="core",
        tags=["math", "english", "reasoning", "chain-of-thought"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_en",
        subtask=True,
    ),
    "mgsm_latin": BenchmarkMetadata(
        name="MGSM Latin Script",
        description="Grade school math problems in Latin script languages (German, English, Spanish, French, Swahili)",
        category="core",
        tags=["math", "multilingual", "latin-script", "reasoning", "chain-of-thought"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_latin",
        subtask=True,
    ),
    "mgsm_non_latin": BenchmarkMetadata(
        name="MGSM Non-Latin Script",
        description="Grade school math problems in non-Latin script languages (Bengali, Japanese, Russian, Telugu, Thai, Chinese)",
        category="core",
        tags=[
            "math",
            "multilingual",
            "non-latin-script",
            "reasoning",
            "chain-of-thought",
        ],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_non_latin",
        subtask=True,
    ),
    "mgsm_de": BenchmarkMetadata(
        name="MGSM (German)",
        description="Multilingual Grade School Math in German",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_de",
        subtask=True,
    ),
    "mgsm_es": BenchmarkMetadata(
        name="MGSM (Spanish)",
        description="Multilingual Grade School Math in Spanish",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_es",
        subtask=True,
    ),
    "mgsm_fr": BenchmarkMetadata(
        name="MGSM (French)",
        description="Multilingual Grade School Math in French",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_fr",
        subtask=True,
    ),
    "mgsm_sw": BenchmarkMetadata(
        name="MGSM (Swahili)",
        description="Multilingual Grade School Math in Swahili",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_sw",
        subtask=True,
    ),
    "mgsm_bn": BenchmarkMetadata(
        name="MGSM (Bengali)",
        description="Multilingual Grade School Math in Bengali",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_bn",
        subtask=True,
    ),
    "mgsm_ja": BenchmarkMetadata(
        name="MGSM (Japanese)",
        description="Multilingual Grade School Math in Japanese",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_ja",
        subtask=True,
    ),
    "mgsm_ru": BenchmarkMetadata(
        name="MGSM (Russian)",
        description="Multilingual Grade School Math in Russian",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_ru",
        subtask=True,
    ),
    "mgsm_te": BenchmarkMetadata(
        name="MGSM (Telugu)",
        description="Multilingual Grade School Math in Telugu",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_te",
        subtask=True,
    ),
    "mgsm_th": BenchmarkMetadata(
        name="MGSM (Thai)",
        description="Multilingual Grade School Math in Thai",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_th",
        subtask=True,
    ),
    "mgsm_zh": BenchmarkMetadata(
        name="MGSM (Chinese)",
        description="Multilingual Grade School Math in Chinese",
        category="math",
        tags=["math", "reasoning", "multilingual", "mgsm"],
        module_path="openbench.evals.mgsm",
        function_name="mgsm_zh",
        subtask=True,
    ),
    "drop": BenchmarkMetadata(
        name="DROP",
        description="Reading comprehension benchmark requiring discrete reasoning over paragraphs (arithmetic, counting, sorting)",
        category="core",
        tags=[
            "reading-comprehension",
            "reasoning",
            "arithmetic",
            "counting",
            "sorting",
        ],
        module_path="openbench.evals.drop",
        function_name="drop",
    ),
    "math": BenchmarkMetadata(
        name="MATH",
        description="Measuring Mathematical Problem Solving - 5000 competition math problems across 7 subjects and 5 difficulty levels",
        category="core",
        tags=["math", "problem-solving", "reasoning", "competition", "graded"],
        module_path="openbench.evals.math",
        function_name="math",
    ),
    "math_500": BenchmarkMetadata(
        name="MATH-500",
        description="500-problem subset of MATH dataset for faster evaluation of mathematical problem solving",
        category="core",
        tags=[
            "math",
            "problem-solving",
            "reasoning",
            "competition",
            "graded",
            "subset",
        ],
        module_path="openbench.evals.math",
        function_name="math_500",
        subtask=True,
    ),
    # Math competitions
    # MathArena family aggregate
    "aime_2023_I": BenchmarkMetadata(
        name="AIME 2023 I",
        description="American Invitational Mathematics Examination 2023 (First)",
        category="math",
        tags=["math", "competition", "aime", "2023"],
        module_path="openbench.evals.matharena.aime_2023_I.aime_2023_I",
        function_name="aime_2023_I",
        subtask=True,
    ),
    "aime_2023_II": BenchmarkMetadata(
        name="AIME 2023 II",
        description="American Invitational Mathematics Examination 2023 (Second)",
        category="math",
        tags=["math", "competition", "aime", "2023"],
        module_path="openbench.evals.matharena.aime_2023_II.aime_2023_II",
        function_name="aime_2023_II",
        subtask=True,
    ),
    "aime_2024": BenchmarkMetadata(
        name="AIME 2024",
        description="American Invitational Mathematics Examination 2024 (Combined I & II)",
        category="math",
        tags=["math", "competition", "aime", "2024", "combined"],
        module_path="openbench.evals.matharena.aime_2024.aime_2024",
        function_name="aime_2024",
        subtask=True,
    ),
    "aime_2024_I": BenchmarkMetadata(
        name="AIME 2024 I",
        description="American Invitational Mathematics Examination 2024 (First)",
        category="math",
        tags=["math", "competition", "aime", "2024"],
        module_path="openbench.evals.matharena.aime_2024_I.aime_2024_I",
        function_name="aime_2024_I",
        subtask=True,
    ),
    "aime_2024_II": BenchmarkMetadata(
        name="AIME 2024 II",
        description="American Invitational Mathematics Examination 2024 (Second)",
        category="math",
        tags=["math", "competition", "aime", "2024"],
        module_path="openbench.evals.matharena.aime_2024_II.aime_2024_II",
        function_name="aime_2024_II",
        subtask=True,
    ),
    "aime_2025": BenchmarkMetadata(
        name="AIME 2025",
        description="American Invitational Mathematics Examination 2025",
        category="math",
        tags=["math", "competition", "aime", "2025"],
        module_path="openbench.evals.matharena.aime_2025.aime_2025",
        function_name="aime_2025",
        subtask=True,
    ),
    "aime_2025_II": BenchmarkMetadata(
        name="AIME 2025 II",
        description="American Invitational Mathematics Examination 2025 (Second)",
        category="math",
        tags=["math", "competition", "aime", "2025"],
        module_path="openbench.evals.matharena.aime_2025_II.aime_2025_II",
        function_name="aime_2025_II",
        subtask=True,
    ),
    "brumo_2025": BenchmarkMetadata(
        name="BRUMO 2025",
        description="Bruno Mathematical Olympiad 2025",
        category="math",
        tags=["math", "competition", "olympiad", "2025"],
        module_path="openbench.evals.matharena.brumo_2025.brumo_2025",
        function_name="brumo_2025",
        subtask=True,
    ),
    "hmmt_feb_2023": BenchmarkMetadata(
        name="HMMT Feb 2023",
        description="Harvard-MIT Mathematics Tournament February 2023",
        category="math",
        tags=["math", "competition", "hmmt", "2023"],
        module_path="openbench.evals.matharena.hmmt_feb_2023.hmmt_feb_2023",
        function_name="hmmt_feb_2023",
        subtask=True,
    ),
    "hmmt_feb_2024": BenchmarkMetadata(
        name="HMMT Feb 2024",
        description="Harvard-MIT Mathematics Tournament February 2024",
        category="math",
        tags=["math", "competition", "hmmt", "2024"],
        module_path="openbench.evals.matharena.hmmt_feb_2024.hmmt_feb_2024",
        function_name="hmmt_feb_2024",
        subtask=True,
    ),
    "hmmt_feb_2025": BenchmarkMetadata(
        name="HMMT Feb 2025",
        description="Harvard-MIT Mathematics Tournament February 2025",
        category="math",
        tags=["math", "competition", "hmmt", "2025"],
        module_path="openbench.evals.matharena.hmmt_feb_2025.hmmt_feb_2025",
        function_name="hmmt_feb_2025",
        subtask=True,
    ),
    "global_mmlu": BenchmarkMetadata(
        name="Global-MMLU (42 Languages)",
        description="Culturally adapted multilingual MMLU with 42 languages",
        category="core",
        tags=["multiple-choice", "multilingual", "cultural-sensitivity", "mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu",
    ),
    # Global-MMLU individual language tasks (42 total)
    "global_mmlu_amharic": BenchmarkMetadata(
        name="Global-MMLU: Amharic",
        description="Global-MMLU culturally adapted MMLU for Amharic (am)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_amharic",
        subtask=True,
    ),
    "global_mmlu_arabic": BenchmarkMetadata(
        name="Global-MMLU: Arabic",
        description="Global-MMLU culturally adapted MMLU for Arabic (ar)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_arabic",
        subtask=True,
    ),
    "global_mmlu_bengali": BenchmarkMetadata(
        name="Global-MMLU: Bengali",
        description="Global-MMLU culturally adapted MMLU for Bengali (bn)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_bengali",
        subtask=True,
    ),
    "global_mmlu_czech": BenchmarkMetadata(
        name="Global-MMLU: Czech",
        description="Global-MMLU culturally adapted MMLU for Czech (cs)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_czech",
        subtask=True,
    ),
    "global_mmlu_german": BenchmarkMetadata(
        name="Global-MMLU: German",
        description="Global-MMLU culturally adapted MMLU for German (de)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_german",
        subtask=True,
    ),
    "global_mmlu_greek": BenchmarkMetadata(
        name="Global-MMLU: Greek",
        description="Global-MMLU culturally adapted MMLU for Greek (el)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_greek",
        subtask=True,
    ),
    "global_mmlu_english": BenchmarkMetadata(
        name="Global-MMLU: English",
        description="Global-MMLU culturally adapted MMLU for English (en)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_english",
        subtask=True,
    ),
    "global_mmlu_spanish": BenchmarkMetadata(
        name="Global-MMLU: Spanish",
        description="Global-MMLU culturally adapted MMLU for Spanish (es)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_spanish",
        subtask=True,
    ),
    "global_mmlu_persian": BenchmarkMetadata(
        name="Global-MMLU: Persian",
        description="Global-MMLU culturally adapted MMLU for Persian (fa)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_persian",
        subtask=True,
    ),
    "global_mmlu_filipino": BenchmarkMetadata(
        name="Global-MMLU: Filipino",
        description="Global-MMLU culturally adapted MMLU for Filipino (fil)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_filipino",
        subtask=True,
    ),
    "global_mmlu_french": BenchmarkMetadata(
        name="Global-MMLU: French",
        description="Global-MMLU culturally adapted MMLU for French (fr)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_french",
        subtask=True,
    ),
    "global_mmlu_hausa": BenchmarkMetadata(
        name="Global-MMLU: Hausa",
        description="Global-MMLU culturally adapted MMLU for Hausa (ha)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_hausa",
        subtask=True,
    ),
    "global_mmlu_hebrew": BenchmarkMetadata(
        name="Global-MMLU: Hebrew",
        description="Global-MMLU culturally adapted MMLU for Hebrew (he)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_hebrew",
        subtask=True,
    ),
    "global_mmlu_hindi": BenchmarkMetadata(
        name="Global-MMLU: Hindi",
        description="Global-MMLU culturally adapted MMLU for Hindi (hi)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_hindi",
        subtask=True,
    ),
    "global_mmlu_indonesian": BenchmarkMetadata(
        name="Global-MMLU: Indonesian",
        description="Global-MMLU culturally adapted MMLU for Indonesian (id)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_indonesian",
        subtask=True,
    ),
    "global_mmlu_igbo": BenchmarkMetadata(
        name="Global-MMLU: Igbo",
        description="Global-MMLU culturally adapted MMLU for Igbo (ig)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_igbo",
        subtask=True,
    ),
    "global_mmlu_italian": BenchmarkMetadata(
        name="Global-MMLU: Italian",
        description="Global-MMLU culturally adapted MMLU for Italian (it)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_italian",
        subtask=True,
    ),
    "global_mmlu_japanese": BenchmarkMetadata(
        name="Global-MMLU: Japanese",
        description="Global-MMLU culturally adapted MMLU for Japanese (ja)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_japanese",
        subtask=True,
    ),
    "global_mmlu_korean": BenchmarkMetadata(
        name="Global-MMLU: Korean",
        description="Global-MMLU culturally adapted MMLU for Korean (ko)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_korean",
        subtask=True,
    ),
    "global_mmlu_kyrgyz": BenchmarkMetadata(
        name="Global-MMLU: Kyrgyz",
        description="Global-MMLU culturally adapted MMLU for Kyrgyz (ky)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_kyrgyz",
        subtask=True,
    ),
    "global_mmlu_lithuanian": BenchmarkMetadata(
        name="Global-MMLU: Lithuanian",
        description="Global-MMLU culturally adapted MMLU for Lithuanian (lt)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_lithuanian",
        subtask=True,
    ),
    "global_mmlu_malagasy": BenchmarkMetadata(
        name="Global-MMLU: Malagasy",
        description="Global-MMLU culturally adapted MMLU for Malagasy (mg)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_malagasy",
        subtask=True,
    ),
    "global_mmlu_malay": BenchmarkMetadata(
        name="Global-MMLU: Malay",
        description="Global-MMLU culturally adapted MMLU for Malay (ms)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_malay",
        subtask=True,
    ),
    "global_mmlu_nepali": BenchmarkMetadata(
        name="Global-MMLU: Nepali",
        description="Global-MMLU culturally adapted MMLU for Nepali (ne)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_nepali",
        subtask=True,
    ),
    "global_mmlu_dutch": BenchmarkMetadata(
        name="Global-MMLU: Dutch",
        description="Global-MMLU culturally adapted MMLU for Dutch (nl)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_dutch",
        subtask=True,
    ),
    "global_mmlu_chichewa": BenchmarkMetadata(
        name="Global-MMLU: Chichewa",
        description="Global-MMLU culturally adapted MMLU for Chichewa (ny)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_chichewa",
        subtask=True,
    ),
    "global_mmlu_polish": BenchmarkMetadata(
        name="Global-MMLU: Polish",
        description="Global-MMLU culturally adapted MMLU for Polish (pl)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_polish",
        subtask=True,
    ),
    "global_mmlu_portuguese": BenchmarkMetadata(
        name="Global-MMLU: Portuguese",
        description="Global-MMLU culturally adapted MMLU for Portuguese (pt)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_portuguese",
        subtask=True,
    ),
    "global_mmlu_romanian": BenchmarkMetadata(
        name="Global-MMLU: Romanian",
        description="Global-MMLU culturally adapted MMLU for Romanian (ro)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_romanian",
        subtask=True,
    ),
    "global_mmlu_russian": BenchmarkMetadata(
        name="Global-MMLU: Russian",
        description="Global-MMLU culturally adapted MMLU for Russian (ru)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_russian",
        subtask=True,
    ),
    "global_mmlu_sinhala": BenchmarkMetadata(
        name="Global-MMLU: Sinhala",
        description="Global-MMLU culturally adapted MMLU for Sinhala (si)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_sinhala",
        subtask=True,
    ),
    "global_mmlu_shona": BenchmarkMetadata(
        name="Global-MMLU: Shona",
        description="Global-MMLU culturally adapted MMLU for Shona (sn)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_shona",
        subtask=True,
    ),
    "global_mmlu_somali": BenchmarkMetadata(
        name="Global-MMLU: Somali",
        description="Global-MMLU culturally adapted MMLU for Somali (so)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_somali",
        subtask=True,
    ),
    "global_mmlu_serbian": BenchmarkMetadata(
        name="Global-MMLU: Serbian",
        description="Global-MMLU culturally adapted MMLU for Serbian (sr)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_serbian",
        subtask=True,
    ),
    "global_mmlu_swedish": BenchmarkMetadata(
        name="Global-MMLU: Swedish",
        description="Global-MMLU culturally adapted MMLU for Swedish (sv)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_swedish",
        subtask=True,
    ),
    "global_mmlu_swahili": BenchmarkMetadata(
        name="Global-MMLU: Swahili",
        description="Global-MMLU culturally adapted MMLU for Swahili (sw)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_swahili",
        subtask=True,
    ),
    "global_mmlu_telugu": BenchmarkMetadata(
        name="Global-MMLU: Telugu",
        description="Global-MMLU culturally adapted MMLU for Telugu (te)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_telugu",
        subtask=True,
    ),
    "global_mmlu_turkish": BenchmarkMetadata(
        name="Global-MMLU: Turkish",
        description="Global-MMLU culturally adapted MMLU for Turkish (tr)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_turkish",
        subtask=True,
    ),
    "global_mmlu_ukrainian": BenchmarkMetadata(
        name="Global-MMLU: Ukrainian",
        description="Global-MMLU culturally adapted MMLU for Ukrainian (uk)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_ukrainian",
        subtask=True,
    ),
    "global_mmlu_vietnamese": BenchmarkMetadata(
        name="Global-MMLU: Vietnamese",
        description="Global-MMLU culturally adapted MMLU for Vietnamese (vi)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_vietnamese",
        subtask=True,
    ),
    "global_mmlu_yoruba": BenchmarkMetadata(
        name="Global-MMLU: Yoruba",
        description="Global-MMLU culturally adapted MMLU for Yoruba (yo)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_yoruba",
        subtask=True,
    ),
    "global_mmlu_chinese": BenchmarkMetadata(
        name="Global-MMLU: Chinese",
        description="Global-MMLU culturally adapted MMLU for Chinese (zh)",
        category="global-mmlu",
        tags=["multiple-choice", "multilingual", "cultural-adaptation", "global-mmlu"],
        module_path="openbench.evals.global_mmlu",
        function_name="global_mmlu_chinese",
        subtask=True,
    ),
    # BIG-Bench family aggregate
    # BIG-Bench individual tasks
    "bigbench_anachronisms": BenchmarkMetadata(
        name="BigBench: Anachronisms",
        description="BigBench MCQ task: anachronisms",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_anachronisms",
        subtask=True,
    ),
    "bigbench_analogical_similarity": BenchmarkMetadata(
        name="BigBench: Analogical Similarity",
        description="BigBench MCQ task: analogical_similarity",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_analogical_similarity",
        subtask=True,
    ),
    "bigbench_analytic_entailment": BenchmarkMetadata(
        name="BigBench: Analytic Entailment",
        description="BigBench MCQ task: analytic_entailment",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_analytic_entailment",
        subtask=True,
    ),
    "bigbench_arithmetic": BenchmarkMetadata(
        name="BigBench: Arithmetic",
        description="BigBench MCQ task: arithmetic",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_arithmetic",
        subtask=True,
    ),
    "bigbench_authorship_verification": BenchmarkMetadata(
        name="BigBench: Authorship Verification",
        description="BigBench MCQ task: authorship_verification",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_authorship_verification",
        subtask=True,
    ),
    "bigbench_bbq_lite_json": BenchmarkMetadata(
        name="BigBench: Bbq Lite Json",
        description="BigBench MCQ task: bbq_lite_json",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_bbq_lite_json",
        subtask=True,
    ),
    "bigbench_causal_judgment": BenchmarkMetadata(
        name="BigBench: Causal Judgment",
        description="BigBench MCQ task: causal_judgment",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_causal_judgment",
        subtask=True,
    ),
    "bigbench_cause_and_effect": BenchmarkMetadata(
        name="BigBench: Cause And Effect",
        description="BigBench MCQ task: cause_and_effect",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_cause_and_effect",
        subtask=True,
    ),
    "bigbench_checkmate_in_one": BenchmarkMetadata(
        name="BigBench: Checkmate In One",
        description="BigBench MCQ task: checkmate_in_one",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_checkmate_in_one",
        subtask=True,
    ),
    "bigbench_cifar10_classification": BenchmarkMetadata(
        name="BigBench: Cifar10 Classification",
        description="BigBench MCQ task: cifar10_classification",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_cifar10_classification",
        subtask=True,
    ),
    "bigbench_code_line_description": BenchmarkMetadata(
        name="BigBench: Code Line Description",
        description="BigBench MCQ task: code_line_description",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_code_line_description",
        subtask=True,
    ),
    "bigbench_color": BenchmarkMetadata(
        name="BigBench: Color",
        description="BigBench MCQ task: color",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_color",
        subtask=True,
    ),
    "bigbench_common_morpheme": BenchmarkMetadata(
        name="BigBench: Common Morpheme",
        description="BigBench MCQ task: common_morpheme",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_common_morpheme",
        subtask=True,
    ),
    "bigbench_conceptual_combinations": BenchmarkMetadata(
        name="BigBench: Conceptual Combinations",
        description="BigBench MCQ task: conceptual_combinations",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_conceptual_combinations",
        subtask=True,
    ),
    "bigbench_contextual_parametric_knowledge_conflicts": BenchmarkMetadata(
        name="BigBench: Contextual Parametric Knowledge Conflicts",
        description="BigBench MCQ task: contextual_parametric_knowledge_conflicts",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_contextual_parametric_knowledge_conflicts",
        subtask=True,
    ),
    "bigbench_crash_blossom": BenchmarkMetadata(
        name="BigBench: Crash Blossom",
        description="BigBench MCQ task: crash_blossom",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_crash_blossom",
        subtask=True,
    ),
    "bigbench_crass_ai": BenchmarkMetadata(
        name="BigBench: Crass Ai",
        description="BigBench MCQ task: crass_ai",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_crass_ai",
        subtask=True,
    ),
    "bigbench_cryobiology_spanish": BenchmarkMetadata(
        name="BigBench: Cryobiology Spanish",
        description="BigBench MCQ task: cryobiology_spanish",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_cryobiology_spanish",
        subtask=True,
    ),
    "bigbench_cs_algorithms": BenchmarkMetadata(
        name="BigBench: Cs Algorithms",
        description="BigBench MCQ task: cs_algorithms",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_cs_algorithms",
        subtask=True,
    ),
    "bigbench_dark_humor_detection": BenchmarkMetadata(
        name="BigBench: Dark Humor Detection",
        description="BigBench MCQ task: dark_humor_detection",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_dark_humor_detection",
        subtask=True,
    ),
    "bigbench_date_understanding": BenchmarkMetadata(
        name="BigBench: Date Understanding",
        description="BigBench MCQ task: date_understanding",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_date_understanding",
        subtask=True,
    ),
    "bigbench_disambiguation_qa": BenchmarkMetadata(
        name="BigBench: Disambiguation Qa",
        description="BigBench MCQ task: disambiguation_qa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_disambiguation_qa",
        subtask=True,
    ),
    "bigbench_discourse_marker_prediction": BenchmarkMetadata(
        name="BigBench: Discourse Marker Prediction",
        description="BigBench MCQ task: discourse_marker_prediction",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_discourse_marker_prediction",
        subtask=True,
    ),
    "bigbench_dyck_languages": BenchmarkMetadata(
        name="BigBench: Dyck Languages",
        description="BigBench MCQ task: dyck_languages",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_dyck_languages",
        subtask=True,
    ),
    "bigbench_elementary_math_qa": BenchmarkMetadata(
        name="BigBench: Elementary Math Qa",
        description="BigBench MCQ task: elementary_math_qa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_elementary_math_qa",
        subtask=True,
    ),
    "bigbench_emoji_movie": BenchmarkMetadata(
        name="BigBench: Emoji Movie",
        description="BigBench MCQ task: emoji_movie",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_emoji_movie",
        subtask=True,
    ),
    "bigbench_emojis_emotion_prediction": BenchmarkMetadata(
        name="BigBench: Emojis Emotion Prediction",
        description="BigBench MCQ task: emojis_emotion_prediction",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_emojis_emotion_prediction",
        subtask=True,
    ),
    "bigbench_empirical_judgments": BenchmarkMetadata(
        name="BigBench: Empirical Judgments",
        description="BigBench MCQ task: empirical_judgments",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_empirical_judgments",
        subtask=True,
    ),
    "bigbench_english_proverbs": BenchmarkMetadata(
        name="BigBench: English Proverbs",
        description="BigBench MCQ task: english_proverbs",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_english_proverbs",
        subtask=True,
    ),
    "bigbench_english_russian_proverbs": BenchmarkMetadata(
        name="BigBench: English Russian Proverbs",
        description="BigBench MCQ task: english_russian_proverbs",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_english_russian_proverbs",
        subtask=True,
    ),
    "bigbench_entailed_polarity": BenchmarkMetadata(
        name="BigBench: Entailed Polarity",
        description="BigBench MCQ task: entailed_polarity",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_entailed_polarity",
        subtask=True,
    ),
    "bigbench_entailed_polarity_hindi": BenchmarkMetadata(
        name="BigBench: Entailed Polarity Hindi",
        description="BigBench MCQ task: entailed_polarity_hindi",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_entailed_polarity_hindi",
        subtask=True,
    ),
    "bigbench_epistemic_reasoning": BenchmarkMetadata(
        name="BigBench: Epistemic Reasoning",
        description="BigBench MCQ task: epistemic_reasoning",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_epistemic_reasoning",
        subtask=True,
    ),
    "bigbench_evaluating_information_essentiality": BenchmarkMetadata(
        name="BigBench: Evaluating Information Essentiality",
        description="BigBench MCQ task: evaluating_information_essentiality",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_evaluating_information_essentiality",
        subtask=True,
    ),
    "bigbench_fact_checker": BenchmarkMetadata(
        name="BigBench: Fact Checker",
        description="BigBench MCQ task: fact_checker",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_fact_checker",
        subtask=True,
    ),
    "bigbench_fantasy_reasoning": BenchmarkMetadata(
        name="BigBench: Fantasy Reasoning",
        description="BigBench MCQ task: fantasy_reasoning",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_fantasy_reasoning",
        subtask=True,
    ),
    "bigbench_figure_of_speech_detection": BenchmarkMetadata(
        name="BigBench: Figure Of Speech Detection",
        description="BigBench MCQ task: figure_of_speech_detection",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_figure_of_speech_detection",
        subtask=True,
    ),
    "bigbench_formal_fallacies_syllogisms_negation": BenchmarkMetadata(
        name="BigBench: Formal Fallacies Syllogisms Negation",
        description="BigBench MCQ task: formal_fallacies_syllogisms_negation",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_formal_fallacies_syllogisms_negation",
        subtask=True,
    ),
    "bigbench_general_knowledge": BenchmarkMetadata(
        name="BigBench: General Knowledge",
        description="BigBench MCQ task: general_knowledge",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_general_knowledge",
        subtask=True,
    ),
    "bigbench_geometric_shapes": BenchmarkMetadata(
        name="BigBench: Geometric Shapes",
        description="BigBench MCQ task: geometric_shapes",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_geometric_shapes",
        subtask=True,
    ),
    "bigbench_goal_step_wikihow": BenchmarkMetadata(
        name="BigBench: Goal Step Wikihow",
        description="BigBench MCQ task: goal_step_wikihow",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_goal_step_wikihow",
        subtask=True,
    ),
    "bigbench_gre_reading_comprehension": BenchmarkMetadata(
        name="BigBench: Gre Reading Comprehension",
        description="BigBench MCQ task: gre_reading_comprehension",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_gre_reading_comprehension",
        subtask=True,
    ),
    "bigbench_hhh_alignment": BenchmarkMetadata(
        name="BigBench: Hhh Alignment",
        description="BigBench MCQ task: hhh_alignment",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_hhh_alignment",
        subtask=True,
    ),
    "bigbench_hindu_knowledge": BenchmarkMetadata(
        name="BigBench: Hindu Knowledge",
        description="BigBench MCQ task: hindu_knowledge",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_hindu_knowledge",
        subtask=True,
    ),
    "bigbench_hinglish_toxicity": BenchmarkMetadata(
        name="BigBench: Hinglish Toxicity",
        description="BigBench MCQ task: hinglish_toxicity",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_hinglish_toxicity",
        subtask=True,
    ),
    "bigbench_human_organs_senses": BenchmarkMetadata(
        name="BigBench: Human Organs Senses",
        description="BigBench MCQ task: human_organs_senses",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_human_organs_senses",
        subtask=True,
    ),
    "bigbench_hyperbaton": BenchmarkMetadata(
        name="BigBench: Hyperbaton",
        description="BigBench MCQ task: hyperbaton",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_hyperbaton",
        subtask=True,
    ),
    "bigbench_identify_math_theorems": BenchmarkMetadata(
        name="BigBench: Identify Math Theorems",
        description="BigBench MCQ task: identify_math_theorems",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_identify_math_theorems",
        subtask=True,
    ),
    "bigbench_identify_odd_metaphor": BenchmarkMetadata(
        name="BigBench: Identify Odd Metaphor",
        description="BigBench MCQ task: identify_odd_metaphor",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_identify_odd_metaphor",
        subtask=True,
    ),
    "bigbench_implicatures": BenchmarkMetadata(
        name="BigBench: Implicatures",
        description="BigBench MCQ task: implicatures",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_implicatures",
        subtask=True,
    ),
    "bigbench_implicit_relations": BenchmarkMetadata(
        name="BigBench: Implicit Relations",
        description="BigBench MCQ task: implicit_relations",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_implicit_relations",
        subtask=True,
    ),
    "bigbench_indic_cause_and_effect": BenchmarkMetadata(
        name="BigBench: Indic Cause And Effect",
        description="BigBench MCQ task: indic_cause_and_effect",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_indic_cause_and_effect",
        subtask=True,
    ),
    "bigbench_intent_recognition": BenchmarkMetadata(
        name="BigBench: Intent Recognition",
        description="BigBench MCQ task: intent_recognition",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_intent_recognition",
        subtask=True,
    ),
    "bigbench_international_phonetic_alphabet_nli": BenchmarkMetadata(
        name="BigBench: International Phonetic Alphabet Nli",
        description="BigBench MCQ task: international_phonetic_alphabet_nli",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_international_phonetic_alphabet_nli",
        subtask=True,
    ),
    "bigbench_intersect_geometry": BenchmarkMetadata(
        name="BigBench: Intersect Geometry",
        description="BigBench MCQ task: intersect_geometry",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_intersect_geometry",
        subtask=True,
    ),
    "bigbench_irony_identification": BenchmarkMetadata(
        name="BigBench: Irony Identification",
        description="BigBench MCQ task: irony_identification",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_irony_identification",
        subtask=True,
    ),
    "bigbench_kanji_ascii": BenchmarkMetadata(
        name="BigBench: Kanji Ascii",
        description="BigBench MCQ task: kanji_ascii",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_kanji_ascii",
        subtask=True,
    ),
    "bigbench_kannada": BenchmarkMetadata(
        name="BigBench: Kannada",
        description="BigBench MCQ task: kannada",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_kannada",
        subtask=True,
    ),
    "bigbench_key_value_maps": BenchmarkMetadata(
        name="BigBench: Key Value Maps",
        description="BigBench MCQ task: key_value_maps",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_key_value_maps",
        subtask=True,
    ),
    "bigbench_known_unknowns": BenchmarkMetadata(
        name="BigBench: Known Unknowns",
        description="BigBench MCQ task: known_unknowns",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_known_unknowns",
        subtask=True,
    ),
    "bigbench_language_identification": BenchmarkMetadata(
        name="BigBench: Language Identification",
        description="BigBench MCQ task: language_identification",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_language_identification",
        subtask=True,
    ),
    "bigbench_logic_grid_puzzle": BenchmarkMetadata(
        name="BigBench: Logic Grid Puzzle",
        description="BigBench MCQ task: logic_grid_puzzle",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_logic_grid_puzzle",
        subtask=True,
    ),
    "bigbench_logical_args": BenchmarkMetadata(
        name="BigBench: Logical Args",
        description="BigBench MCQ task: logical_args",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_logical_args",
        subtask=True,
    ),
    "bigbench_logical_deduction": BenchmarkMetadata(
        name="BigBench: Logical Deduction",
        description="BigBench MCQ task: logical_deduction",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_logical_deduction",
        subtask=True,
    ),
    "bigbench_logical_fallacy_detection": BenchmarkMetadata(
        name="BigBench: Logical Fallacy Detection",
        description="BigBench MCQ task: logical_fallacy_detection",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_logical_fallacy_detection",
        subtask=True,
    ),
    "bigbench_logical_sequence": BenchmarkMetadata(
        name="BigBench: Logical Sequence",
        description="BigBench MCQ task: logical_sequence",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_logical_sequence",
        subtask=True,
    ),
    "bigbench_mathematical_induction": BenchmarkMetadata(
        name="BigBench: Mathematical Induction",
        description="BigBench MCQ task: mathematical_induction",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_mathematical_induction",
        subtask=True,
    ),
    "bigbench_medical_questions_russian": BenchmarkMetadata(
        name="BigBench: Medical Questions Russian",
        description="BigBench MCQ task: medical_questions_russian",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_medical_questions_russian",
        subtask=True,
    ),
    "bigbench_metaphor_boolean": BenchmarkMetadata(
        name="BigBench: Metaphor Boolean",
        description="BigBench MCQ task: metaphor_boolean",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_metaphor_boolean",
        subtask=True,
    ),
    "bigbench_metaphor_understanding": BenchmarkMetadata(
        name="BigBench: Metaphor Understanding",
        description="BigBench MCQ task: metaphor_understanding",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_metaphor_understanding",
        subtask=True,
    ),
    "bigbench_minute_mysteries_qa": BenchmarkMetadata(
        name="BigBench: Minute Mysteries Qa",
        description="BigBench MCQ task: minute_mysteries_qa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_minute_mysteries_qa",
        subtask=True,
    ),
    "bigbench_misconceptions": BenchmarkMetadata(
        name="BigBench: Misconceptions",
        description="BigBench MCQ task: misconceptions",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_misconceptions",
        subtask=True,
    ),
    "bigbench_misconceptions_russian": BenchmarkMetadata(
        name="BigBench: Misconceptions Russian",
        description="BigBench MCQ task: misconceptions_russian",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_misconceptions_russian",
        subtask=True,
    ),
    "bigbench_mnist_ascii": BenchmarkMetadata(
        name="BigBench: Mnist Ascii",
        description="BigBench MCQ task: mnist_ascii",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_mnist_ascii",
        subtask=True,
    ),
    "bigbench_moral_permissibility": BenchmarkMetadata(
        name="BigBench: Moral Permissibility",
        description="BigBench MCQ task: moral_permissibility",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_moral_permissibility",
        subtask=True,
    ),
    "bigbench_movie_dialog_same_or_different": BenchmarkMetadata(
        name="BigBench: Movie Dialog Same Or Different",
        description="BigBench MCQ task: movie_dialog_same_or_different",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_movie_dialog_same_or_different",
        subtask=True,
    ),
    "bigbench_movie_recommendation": BenchmarkMetadata(
        name="BigBench: Movie Recommendation",
        description="BigBench MCQ task: movie_recommendation",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_movie_recommendation",
        subtask=True,
    ),
    "bigbench_navigate": BenchmarkMetadata(
        name="BigBench: Navigate",
        description="BigBench MCQ task: navigate",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_navigate",
        subtask=True,
    ),
    "bigbench_nonsense_words_grammar": BenchmarkMetadata(
        name="BigBench: Nonsense Words Grammar",
        description="BigBench MCQ task: nonsense_words_grammar",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_nonsense_words_grammar",
        subtask=True,
    ),
    "bigbench_novel_concepts": BenchmarkMetadata(
        name="BigBench: Novel Concepts",
        description="BigBench MCQ task: novel_concepts",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_novel_concepts",
        subtask=True,
    ),
    "bigbench_odd_one_out": BenchmarkMetadata(
        name="BigBench: Odd One Out",
        description="BigBench MCQ task: odd_one_out",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_odd_one_out",
        subtask=True,
    ),
    "bigbench_parsinlu_qa": BenchmarkMetadata(
        name="BigBench: Parsinlu Qa",
        description="BigBench MCQ task: parsinlu_qa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_parsinlu_qa",
        subtask=True,
    ),
    "bigbench_penguins_in_a_table": BenchmarkMetadata(
        name="BigBench: Penguins In A Table",
        description="BigBench MCQ task: penguins_in_a_table",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_penguins_in_a_table",
        subtask=True,
    ),
    "bigbench_periodic_elements": BenchmarkMetadata(
        name="BigBench: Periodic Elements",
        description="BigBench MCQ task: periodic_elements",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_periodic_elements",
        subtask=True,
    ),
    "bigbench_persian_idioms": BenchmarkMetadata(
        name="BigBench: Persian Idioms",
        description="BigBench MCQ task: persian_idioms",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_persian_idioms",
        subtask=True,
    ),
    "bigbench_phrase_relatedness": BenchmarkMetadata(
        name="BigBench: Phrase Relatedness",
        description="BigBench MCQ task: phrase_relatedness",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_phrase_relatedness",
        subtask=True,
    ),
    "bigbench_physical_intuition": BenchmarkMetadata(
        name="BigBench: Physical Intuition",
        description="BigBench MCQ task: physical_intuition",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_physical_intuition",
        subtask=True,
    ),
    "bigbench_physics": BenchmarkMetadata(
        name="BigBench: Physics",
        description="BigBench MCQ task: physics",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_physics",
        subtask=True,
    ),
    "bigbench_play_dialog_same_or_different": BenchmarkMetadata(
        name="BigBench: Play Dialog Same Or Different",
        description="BigBench MCQ task: play_dialog_same_or_different",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_play_dialog_same_or_different",
        subtask=True,
    ),
    "bigbench_presuppositions_as_nli": BenchmarkMetadata(
        name="BigBench: Presuppositions As Nli",
        description="BigBench MCQ task: presuppositions_as_nli",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_presuppositions_as_nli",
        subtask=True,
    ),
    "bigbench_question_selection": BenchmarkMetadata(
        name="BigBench: Question Selection",
        description="BigBench MCQ task: question_selection",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_question_selection",
        subtask=True,
    ),
    "bigbench_real_or_fake_text": BenchmarkMetadata(
        name="BigBench: Real Or Fake Text",
        description="BigBench MCQ task: real_or_fake_text",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_real_or_fake_text",
        subtask=True,
    ),
    "bigbench_reasoning_about_colored_objects": BenchmarkMetadata(
        name="BigBench: Reasoning About Colored Objects",
        description="BigBench MCQ task: reasoning_about_colored_objects",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_reasoning_about_colored_objects",
        subtask=True,
    ),
    "bigbench_rhyming": BenchmarkMetadata(
        name="BigBench: Rhyming",
        description="BigBench MCQ task: rhyming",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_rhyming",
        subtask=True,
    ),
    "bigbench_riddle_sense": BenchmarkMetadata(
        name="BigBench: Riddle Sense",
        description="BigBench MCQ task: riddle_sense",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_riddle_sense",
        subtask=True,
    ),
    "bigbench_ruin_names": BenchmarkMetadata(
        name="BigBench: Ruin Names",
        description="BigBench MCQ task: ruin_names",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_ruin_names",
        subtask=True,
    ),
    "bigbench_salient_translation_error_detection": BenchmarkMetadata(
        name="BigBench: Salient Translation Error Detection",
        description="BigBench MCQ task: salient_translation_error_detection",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_salient_translation_error_detection",
        subtask=True,
    ),
    "bigbench_sentence_ambiguity": BenchmarkMetadata(
        name="BigBench: Sentence Ambiguity",
        description="BigBench MCQ task: sentence_ambiguity",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_sentence_ambiguity",
        subtask=True,
    ),
    "bigbench_similarities_abstraction": BenchmarkMetadata(
        name="BigBench: Similarities Abstraction",
        description="BigBench MCQ task: similarities_abstraction",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_similarities_abstraction",
        subtask=True,
    ),
    "bigbench_simple_ethical_questions": BenchmarkMetadata(
        name="BigBench: Simple Ethical Questions",
        description="BigBench MCQ task: simple_ethical_questions",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_simple_ethical_questions",
        subtask=True,
    ),
    "bigbench_snarks": BenchmarkMetadata(
        name="BigBench: Snarks",
        description="BigBench MCQ task: snarks",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_snarks",
        subtask=True,
    ),
    "bigbench_social_iqa": BenchmarkMetadata(
        name="BigBench: Social Iqa",
        description="BigBench MCQ task: social_iqa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_social_iqa",
        subtask=True,
    ),
    "bigbench_social_support": BenchmarkMetadata(
        name="BigBench: Social Support",
        description="BigBench MCQ task: social_support",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_social_support",
        subtask=True,
    ),
    "bigbench_sports_understanding": BenchmarkMetadata(
        name="BigBench: Sports Understanding",
        description="BigBench MCQ task: sports_understanding",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_sports_understanding",
        subtask=True,
    ),
    "bigbench_strange_stories": BenchmarkMetadata(
        name="BigBench: Strange Stories",
        description="BigBench MCQ task: strange_stories",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_strange_stories",
        subtask=True,
    ),
    "bigbench_strategyqa": BenchmarkMetadata(
        name="BigBench: Strategyqa",
        description="BigBench MCQ task: strategyqa",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_strategyqa",
        subtask=True,
    ),
    "bigbench_suicide_risk": BenchmarkMetadata(
        name="BigBench: Suicide Risk",
        description="BigBench MCQ task: suicide_risk",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_suicide_risk",
        subtask=True,
    ),
    "bigbench_swahili_english_proverbs": BenchmarkMetadata(
        name="BigBench: Swahili English Proverbs",
        description="BigBench MCQ task: swahili_english_proverbs",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_swahili_english_proverbs",
        subtask=True,
    ),
    "bigbench_swedish_to_german_proverbs": BenchmarkMetadata(
        name="BigBench: Swedish To German Proverbs",
        description="BigBench MCQ task: swedish_to_german_proverbs",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_swedish_to_german_proverbs",
        subtask=True,
    ),
    "bigbench_symbol_interpretation": BenchmarkMetadata(
        name="BigBench: Symbol Interpretation",
        description="BigBench MCQ task: symbol_interpretation",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_symbol_interpretation",
        subtask=True,
    ),
    "bigbench_temporal_sequences": BenchmarkMetadata(
        name="BigBench: Temporal Sequences",
        description="BigBench MCQ task: temporal_sequences",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_temporal_sequences",
        subtask=True,
    ),
    "bigbench_timedial": BenchmarkMetadata(
        name="BigBench: Timedial",
        description="BigBench MCQ task: timedial",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_timedial",
        subtask=True,
    ),
    "bigbench_tracking_shuffled_objects": BenchmarkMetadata(
        name="BigBench: Tracking Shuffled Objects",
        description="BigBench MCQ task: tracking_shuffled_objects",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_tracking_shuffled_objects",
        subtask=True,
    ),
    "bigbench_understanding_fables": BenchmarkMetadata(
        name="BigBench: Understanding Fables",
        description="BigBench MCQ task: understanding_fables",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_understanding_fables",
        subtask=True,
    ),
    "bigbench_undo_permutation": BenchmarkMetadata(
        name="BigBench: Undo Permutation",
        description="BigBench MCQ task: undo_permutation",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_undo_permutation",
        subtask=True,
    ),
    "bigbench_unit_conversion": BenchmarkMetadata(
        name="BigBench: Unit Conversion",
        description="BigBench MCQ task: unit_conversion",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_unit_conversion",
        subtask=True,
    ),
    "bigbench_unit_interpretation": BenchmarkMetadata(
        name="BigBench: Unit Interpretation",
        description="BigBench MCQ task: unit_interpretation",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_unit_interpretation",
        subtask=True,
    ),
    "bigbench_vitaminc_fact_verification": BenchmarkMetadata(
        name="BigBench: Vitaminc Fact Verification",
        description="BigBench MCQ task: vitaminc_fact_verification",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_vitaminc_fact_verification",
        subtask=True,
    ),
    "bigbench_what_is_the_tao": BenchmarkMetadata(
        name="BigBench: What Is The Tao",
        description="BigBench MCQ task: what_is_the_tao",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_what_is_the_tao",
        subtask=True,
    ),
    "bigbench_which_wiki_edit": BenchmarkMetadata(
        name="BigBench: Which Wiki Edit",
        description="BigBench MCQ task: which_wiki_edit",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_which_wiki_edit",
        subtask=True,
    ),
    "bigbench_winowhy": BenchmarkMetadata(
        name="BigBench: Winowhy",
        description="BigBench MCQ task: winowhy",
        category="bigbench",
        tags=["multiple-choice", "reasoning", "bigbench"],
        module_path="openbench.evals.bigbench",
        function_name="bigbench_winowhy",
        subtask=True,
    ),
    # BIG-Bench Hard family aggregate
    # BBH individual tasks
    "bbh_causal_judgment": BenchmarkMetadata(
        name="BBH: Causal Judgment",
        description="BigBench Hard - Causal judgment reasoning",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_causal_judgment",
        subtask=True,
    ),
    "bbh_date_understanding": BenchmarkMetadata(
        name="BBH: Date Understanding",
        description="BigBench Hard - Understanding and reasoning about dates",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_date_understanding",
        subtask=True,
    ),
    "bbh_disambiguation_qa": BenchmarkMetadata(
        name="BBH: Disambiguation QA",
        description="BigBench Hard - Pronoun disambiguation in questions",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_disambiguation_qa",
        subtask=True,
    ),
    "bbh_geometric_shapes": BenchmarkMetadata(
        name="BBH: Geometric Shapes",
        description="BigBench Hard - Reasoning about geometric shapes",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "geometry",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_geometric_shapes",
        subtask=True,
    ),
    "bbh_logical_deduction_five_objects": BenchmarkMetadata(
        name="BBH: Logical Deduction (5 Objects)",
        description="BigBench Hard - Logical deduction with five objects",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought", "logic"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_logical_deduction_five_objects",
        subtask=True,
    ),
    "bbh_logical_deduction_seven_objects": BenchmarkMetadata(
        name="BBH: Logical Deduction (7 Objects)",
        description="BigBench Hard - Logical deduction with seven objects",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought", "logic"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_logical_deduction_seven_objects",
        subtask=True,
    ),
    "bbh_logical_deduction_three_objects": BenchmarkMetadata(
        name="BBH: Logical Deduction (3 Objects)",
        description="BigBench Hard - Logical deduction with three objects",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought", "logic"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_logical_deduction_three_objects",
        subtask=True,
    ),
    "bbh_movie_recommendation": BenchmarkMetadata(
        name="BBH: Movie Recommendation",
        description="BigBench Hard - Movie recommendation reasoning",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_movie_recommendation",
        subtask=True,
    ),
    "bbh_navigate": BenchmarkMetadata(
        name="BBH: Navigate",
        description="BigBench Hard - Spatial navigation reasoning",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "spatial",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_navigate",
        subtask=True,
    ),
    "bbh_reasoning_about_colored_objects": BenchmarkMetadata(
        name="BBH: Reasoning About Colored Objects",
        description="BigBench Hard - Reasoning about colored objects",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_reasoning_about_colored_objects",
        subtask=True,
    ),
    "bbh_ruin_names": BenchmarkMetadata(
        name="BBH: Ruin Names",
        description="BigBench Hard - Word manipulation and reasoning",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "wordplay",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_ruin_names",
        subtask=True,
    ),
    "bbh_salient_translation_error_detection": BenchmarkMetadata(
        name="BBH: Salient Translation Error Detection",
        description="BigBench Hard - Detecting translation errors",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "translation",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_salient_translation_error_detection",
        subtask=True,
    ),
    "bbh_snarks": BenchmarkMetadata(
        name="BBH: Snarks",
        description="BigBench Hard - Understanding sarcasm and irony",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "sarcasm",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_snarks",
        subtask=True,
    ),
    "bbh_sports_understanding": BenchmarkMetadata(
        name="BBH: Sports Understanding",
        description="BigBench Hard - Sports knowledge and reasoning",
        category="core",
        tags=["multiple-choice", "reasoning", "bigbench", "chain-of-thought", "sports"],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_sports_understanding",
        subtask=True,
    ),
    "bbh_temporal_sequences": BenchmarkMetadata(
        name="BBH: Temporal Sequences",
        description="BigBench Hard - Understanding temporal sequences",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "temporal",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_temporal_sequences",
        subtask=True,
    ),
    "bbh_tracking_shuffled_objects_five_objects": BenchmarkMetadata(
        name="BBH: Tracking Shuffled Objects (5 Objects)",
        description="BigBench Hard - Tracking five shuffled objects",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "tracking",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_tracking_shuffled_objects_five_objects",
        subtask=True,
    ),
    "bbh_tracking_shuffled_objects_seven_objects": BenchmarkMetadata(
        name="BBH: Tracking Shuffled Objects (7 Objects)",
        description="BigBench Hard - Tracking seven shuffled objects",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "tracking",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_tracking_shuffled_objects_seven_objects",
        subtask=True,
    ),
    "bbh_tracking_shuffled_objects_three_objects": BenchmarkMetadata(
        name="BBH: Tracking Shuffled Objects (3 Objects)",
        description="BigBench Hard - Tracking three shuffled objects",
        category="core",
        tags=[
            "multiple-choice",
            "reasoning",
            "bigbench",
            "chain-of-thought",
            "tracking",
        ],
        module_path="openbench.evals.bigbench_hard",
        function_name="bbh_tracking_shuffled_objects_three_objects",
        subtask=True,
    ),
    # BIG-Bench Lite (alias for BBH)
    "medmcqa": BenchmarkMetadata(
        name="MedMCQA",
        description="Medical multiple-choice questions from Indian medical entrance exams (AIIMS & NEET PG)",
        category="core",
        tags=["multiple-choice", "medical", "healthcare", "medicine"],
        module_path="openbench.evals.medmcqa",
        function_name="medmcqa",
    ),
    "medqa": BenchmarkMetadata(
        name="MedQA",
        description="US Medical Licensing Exam (USMLE) questions for medical reasoning",
        category="core",
        tags=["multiple-choice", "medical", "healthcare", "medicine", "clinical"],
        module_path="openbench.evals.medqa",
        function_name="medqa",
    ),
    "pubmedqa": BenchmarkMetadata(
        name="PubMedQA",
        description="Biomedical question answering from PubMed abstracts",
        category="core",
        tags=["multiple-choice", "medical", "biomedical", "research", "literature"],
        module_path="openbench.evals.pubmedqa",
        function_name="pubmedqa",
    ),
    "headqa": BenchmarkMetadata(
        name="HeadQA",
        description="Spanish healthcare specialization exam questions (Spanish and English)",
        category="core",
        tags=["multiple-choice", "medical", "healthcare", "multilingual"],
        module_path="openbench.evals.headqa",
        function_name="headqa",
    ),
    "headqa_en": BenchmarkMetadata(
        name="HeadQA (English)",
        description="Spanish healthcare specialization exam questions in English",
        category="core",
        tags=["multiple-choice", "medical", "healthcare", "english"],
        module_path="openbench.evals.headqa",
        function_name="headqa_en",
        subtask=True,
    ),
    "headqa_es": BenchmarkMetadata(
        name="HeadQA (Spanish)",
        description="Spanish healthcare specialization exam questions in Spanish",
        category="core",
        tags=["multiple-choice", "medical", "healthcare", "spanish"],
        module_path="openbench.evals.headqa",
        function_name="headqa_es",
        subtask=True,
    ),
    # ARC family aggregate
    "arc_easy": BenchmarkMetadata(
        name="ARC-Easy",
        description="AI2 Reasoning Challenge - Easy questions from grade-school science exams",
        category="core",
        tags=["multiple-choice", "science", "commonsense-reasoning"],
        module_path="openbench.evals.arc",
        function_name="arc_easy",
        subtask=True,
    ),
    "arc_challenge": BenchmarkMetadata(
        name="ARC-Challenge",
        description="AI2 Reasoning Challenge - Challenging questions from grade-school science exams",
        category="core",
        tags=["multiple-choice", "science", "commonsense-reasoning"],
        module_path="openbench.evals.arc",
        function_name="arc_challenge",
        subtask=True,
    ),
    "boolq": BenchmarkMetadata(
        name="BoolQ",
        description="BoolQ: A Question Answering Dataset for Boolean Reasoning",
        category="core",
        tags=["boolean-reasoning", "question-answering"],
        module_path="openbench.evals.boolq",
        function_name="boolq",
    ),
    "hellaswag": BenchmarkMetadata(
        name="HellaSwag",
        description="Adversarially-filtered sentence completion benchmark for commonsense reasoning",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "sentence-completion"],
        module_path="openbench.evals.hellaswag",
        function_name="hellaswag",
    ),
    "piqa": BenchmarkMetadata(
        name="PIQA",
        description="Physical Interaction Question Answering - commonsense about physical situations",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "physical-reasoning"],
        module_path="openbench.evals.piqa",
        function_name="piqa",
    ),
    "prost": BenchmarkMetadata(
        name="PROST",
        description="Physical Reasoning about Objects through Space and Time",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "physical-reasoning"],
        module_path="openbench.evals.prost",
        function_name="prost",
    ),
    "swag": BenchmarkMetadata(
        name="SWAG",
        description="Situations With Adversarial Generations - grounded commonsense inference",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "video-captions"],
        module_path="openbench.evals.swag",
        function_name="swag",
    ),
    "winogrande": BenchmarkMetadata(
        name="WinoGrande",
        description="Large-scale Winograd Schema Challenge for commonsense pronoun resolution",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "pronoun-resolution"],
        module_path="openbench.evals.winogrande",
        function_name="winogrande",
    ),
    "wsc273": BenchmarkMetadata(
        name="WSC273",
        description="Original Winograd Schema Challenge with 273 expert-crafted questions",
        category="core",
        tags=["multiple-choice", "commonsense-reasoning", "pronoun-resolution"],
        module_path="openbench.evals.wsc273",
        function_name="wsc273",
    ),
    "scicode": BenchmarkMetadata(
        name="SciCode",
        description="Scientific computing and programming challenges",
        category="core",
        tags=["code-generation", "science", "alpha"],
        module_path="openbench.evals.scicode",
        function_name="scicode",
        is_alpha=True,
    ),
    "rootly_gmcq": BenchmarkMetadata(
        name="GMCQ",
        description="GitHub Multiple Choice Questions",
        category="core",
        tags=["code-understanding"],
        module_path="openbench.evals.rootly_gmcq",
        function_name="rootly_gmcq",
    ),
    "rootly_terraform": BenchmarkMetadata(
        name="Terraform",
        description="Terraform Multiple Choice Questions",
        category="core",
        tags=["code-understanding"],
        module_path="openbench.evals.rootly_terraform",
        function_name="rootly_terraform",
    ),
    "jsonschemabench": BenchmarkMetadata(
        name="JSONSchemaBench",
        description="JSON Schema generation benchmark with ~10K real-world schemas from GitHub, Kubernetes, and other sources for evaluating constrained decoding",
        category="core",
        tags=["json", "jsonschema", "generation", "constrained-decoding"],
        module_path="openbench.evals.jsonschemabench",
        function_name="jsonschemabench",
        is_alpha=False,
    ),
    "mmmu": BenchmarkMetadata(
        name="MMMU",
        description="Massive Multi-discipline Multimodal Understanding and Reasoning Benchmark with 11.5K questions across 30 subjects from college exams, quizzes, and textbooks",
        category="core",
        tags=["multimodal", "multiple-choice", "reasoning", "college-level", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu",
        is_alpha=False,
    ),
    "mmmu_art": BenchmarkMetadata(
        name="MMMU Art",
        description="MMMU Art subset focusing on art and visual design questions",
        category="core",
        tags=["multimodal", "multiple-choice", "art", "visual-design", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_art",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_biology": BenchmarkMetadata(
        name="MMMU Biology",
        description="MMMU Biology subset focusing on biological sciences",
        category="core",
        tags=["multimodal", "multiple-choice", "biology", "science", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_biology",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_chemistry": BenchmarkMetadata(
        name="MMMU Chemistry",
        description="MMMU Chemistry subset focusing on chemical sciences",
        category="core",
        tags=["multimodal", "multiple-choice", "chemistry", "science", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_chemistry",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_math": BenchmarkMetadata(
        name="MMMU Math",
        description="MMMU Mathematics subset focusing on mathematical reasoning",
        category="math",
        tags=["multimodal", "multiple-choice", "mathematics", "reasoning", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_math",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_physics": BenchmarkMetadata(
        name="MMMU Physics",
        description="MMMU Physics subset focusing on physics and physical sciences",
        category="core",
        tags=["multimodal", "multiple-choice", "physics", "science", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_physics",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_accounting": BenchmarkMetadata(
        name="MMMU Accounting",
        description="MMMU Accounting subset focusing on accounting principles and practices",
        category="core",
        tags=["multimodal", "multiple-choice", "accounting", "business", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_accounting",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_agriculture": BenchmarkMetadata(
        name="MMMU Agriculture",
        description="MMMU Agriculture subset focusing on agricultural sciences and practices",
        category="core",
        tags=["multimodal", "multiple-choice", "agriculture", "science", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_agriculture",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_architecture_and_engineering": BenchmarkMetadata(
        name="MMMU Architecture and Engineering",
        description="MMMU Architecture and Engineering subset focusing on engineering design and architecture",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "architecture",
            "engineering",
            "design",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_architecture_and_engineering",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_art_theory": BenchmarkMetadata(
        name="MMMU Art Theory",
        description="MMMU Art Theory subset focusing on art history and theoretical concepts",
        category="core",
        tags=["multimodal", "multiple-choice", "art", "theory", "history", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_art_theory",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_basic_medical_science": BenchmarkMetadata(
        name="MMMU Basic Medical Science",
        description="MMMU Basic Medical Science subset focusing on fundamental medical knowledge",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "medicine",
            "science",
            "health",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_basic_medical_science",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_clinical_medicine": BenchmarkMetadata(
        name="MMMU Clinical Medicine",
        description="MMMU Clinical Medicine subset focusing on clinical medical practice",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "medicine",
            "clinical",
            "health",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_clinical_medicine",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_design": BenchmarkMetadata(
        name="MMMU Design",
        description="MMMU Design subset focusing on design principles and practices",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "design",
            "visual",
            "creative",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_design",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_diagnostics_and_laboratory_medicine": BenchmarkMetadata(
        name="MMMU Diagnostics and Laboratory Medicine",
        description="MMMU Diagnostics and Laboratory Medicine subset focusing on medical diagnostics",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "medicine",
            "diagnostics",
            "laboratory",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_diagnostics_and_laboratory_medicine",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_electronics": BenchmarkMetadata(
        name="MMMU Electronics",
        description="MMMU Electronics subset focusing on electronic systems and circuits",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "electronics",
            "engineering",
            "technology",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_electronics",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_energy_and_power": BenchmarkMetadata(
        name="MMMU Energy and Power",
        description="MMMU Energy and Power subset focusing on energy systems and power generation",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "energy",
            "power",
            "engineering",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_energy_and_power",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_finance": BenchmarkMetadata(
        name="MMMU Finance",
        description="MMMU Finance subset focusing on financial concepts and analysis",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "finance",
            "business",
            "economics",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_finance",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_geography": BenchmarkMetadata(
        name="MMMU Geography",
        description="MMMU Geography subset focusing on geographical knowledge and analysis",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "geography",
            "earth-science",
            "spatial",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_geography",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_history": BenchmarkMetadata(
        name="MMMU History",
        description="MMMU History subset focusing on historical knowledge and analysis",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "history",
            "humanities",
            "culture",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_history",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_literature": BenchmarkMetadata(
        name="MMMU Literature",
        description="MMMU Literature subset focusing on literary analysis and knowledge",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "literature",
            "humanities",
            "language",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_literature",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_manage": BenchmarkMetadata(
        name="MMMU Management",
        description="MMMU Management subset focusing on management principles and practices",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "management",
            "business",
            "leadership",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_manage",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_marketing": BenchmarkMetadata(
        name="MMMU Marketing",
        description="MMMU Marketing subset focusing on marketing strategies and concepts",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "marketing",
            "business",
            "communication",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_marketing",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_materials": BenchmarkMetadata(
        name="MMMU Materials",
        description="MMMU Materials subset focusing on materials science and engineering",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "materials",
            "science",
            "engineering",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_materials",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_mechanical_engineering": BenchmarkMetadata(
        name="MMMU Mechanical Engineering",
        description="MMMU Mechanical Engineering subset focusing on mechanical systems and design",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "mechanical",
            "engineering",
            "design",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_mechanical_engineering",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_music": BenchmarkMetadata(
        name="MMMU Music",
        description="MMMU Music subset focusing on music theory and analysis",
        category="core",
        tags=["multimodal", "multiple-choice", "music", "arts", "theory", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_music",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_pharmacy": BenchmarkMetadata(
        name="MMMU Pharmacy",
        description="MMMU Pharmacy subset focusing on pharmaceutical sciences and practice",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "pharmacy",
            "medicine",
            "health",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_pharmacy",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_public_health": BenchmarkMetadata(
        name="MMMU Public Health",
        description="MMMU Public Health subset focusing on public health concepts and practices",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "public-health",
            "health",
            "population",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_public_health",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_sociology": BenchmarkMetadata(
        name="MMMU Sociology",
        description="MMMU Sociology subset focusing on sociological concepts and analysis",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "sociology",
            "social-science",
            "society",
            "images",
        ],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_sociology",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_mcq": BenchmarkMetadata(
        name="MMMU MCQ",
        description="MMMU MCQ subset focusing on multiple choice questions",
        category="core",
        tags=["multimodal", "multiple-choice", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_mcq",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_open": BenchmarkMetadata(
        name="MMMU Open",
        description="MMMU Open subset focusing on open-ended questions",
        category="core",
        tags=["multimodal", "open-ended", "images"],
        module_path="openbench.evals.mmmu",
        function_name="mmmu_open",
        is_alpha=False,
        subtask=True,
    ),
    "mmmu_pro": BenchmarkMetadata(
        name="MMMU-Pro",
        description="Enhanced multimodal MMMU-Pro benchmark with multiple-choice across many options and images",
        category="core",
        tags=[
            "multimodal",
            "multiple-choice",
            "reasoning",
            "images",
            "mmmu-pro",
        ],
        module_path="openbench.evals.mmmu_pro",
        function_name="mmmu_pro",
        is_alpha=False,
    ),
    "mmmu_pro_vision": BenchmarkMetadata(
        name="MMMU-Pro (Vision)",
        description="MMMU-Pro vision subset with images and multiple-choice questions",
        category="core",
        tags=["multimodal", "vision", "multiple-choice", "images", "mmmu-pro"],
        module_path="openbench.evals.mmmu_pro",
        function_name="mmmu_pro_vision",
        is_alpha=False,
        subtask=True,
    ),
    "arc_agi": BenchmarkMetadata(
        name="ARC-AGI",
        description="Abstraction and Reasoning Corpus for Artificial General Intelligence; specify version with -T version=1 or version=2",
        category="core",
        tags=[
            "reasoning",
            "pattern-recognition",
            "abstract-reasoning",
            "visual",
            "logic",
            "agi",
        ],
        module_path="openbench.evals.arc_agi",
        function_name="arc_agi",
    ),
    "arc_agi_1": BenchmarkMetadata(
        name="ARC-AGI-1",
        description="ARC-AGI dataset version 1",
        category="core",
        tags=[
            "reasoning",
            "pattern-recognition",
            "abstract-reasoning",
            "visual",
            "logic",
            "agi",
        ],
        module_path="openbench.evals.arc_agi",
        function_name="arc_agi_1",
        subtask=True,
    ),
    "arc_agi_2": BenchmarkMetadata(
        name="ARC-AGI-2",
        description="ARC-AGI dataset version 2",
        category="core",
        tags=[
            "reasoning",
            "pattern-recognition",
            "abstract-reasoning",
            "visual",
            "logic",
            "agi",
        ],
        module_path="openbench.evals.arc_agi",
        function_name="arc_agi_2",
        subtask=True,
    ),
    "livemcpbench": BenchmarkMetadata(
        name="LiveMCPBench",
        description="Benchmark for evaluating LLM agents on real-world tasks using the Model Context Protocol (MCP) - 95 tasks across different categories",
        category="core",
        tags=["mcp", "agents", "real-world", "tools", "graded"],
        module_path="openbench.evals.livemcpbench",
        function_name="livemcpbench",
        is_alpha=False,
    ),  # GLUE/SuperGLUE benchmarks
    "anli": BenchmarkMetadata(
        name="ANLI (All Rounds)",
        description="Adversarial Natural Language Inference - challenging NLI benchmark",
        category="glue",
        tags=["multiple-choice", "nli", "adversarial", "reasoning"],
        module_path="openbench.evals.anli",
        function_name="anli",
    ),
    "anli_r1": BenchmarkMetadata(
        name="ANLI Round 1",
        description="Adversarial NLI Round 1",
        category="glue",
        tags=["multiple-choice", "nli", "adversarial", "reasoning"],
        module_path="openbench.evals.anli",
        function_name="anli_r1",
        subtask=True,
    ),
    "anli_r2": BenchmarkMetadata(
        name="ANLI Round 2",
        description="Adversarial NLI Round 2",
        category="glue",
        tags=["multiple-choice", "nli", "adversarial", "reasoning"],
        module_path="openbench.evals.anli",
        function_name="anli_r2",
        subtask=True,
    ),
    "anli_r3": BenchmarkMetadata(
        name="ANLI Round 3",
        description="Adversarial NLI Round 3",
        category="glue",
        tags=["multiple-choice", "nli", "adversarial", "reasoning"],
        module_path="openbench.evals.anli",
        function_name="anli_r3",
        subtask=True,
    ),
    "copa": BenchmarkMetadata(
        name="COPA",
        description="Choice of Plausible Alternatives for causal reasoning",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="copa",
        subtask=True,
    ),
    "rte": BenchmarkMetadata(
        name="RTE (SuperGLUE)",
        description="Recognizing Textual Entailment from SuperGLUE",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="rte",
        subtask=True,
    ),
    "wic": BenchmarkMetadata(
        name="WiC",
        description="Word in Context - word sense disambiguation",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "wsd", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="wic",
        subtask=True,
    ),
    "wsc": BenchmarkMetadata(
        name="WSC",
        description="Winograd Schema Challenge - coreference resolution",
        category="glue",
        tags=[
            "multiple-choice",
            "superglue",
            "nli",
            "reasoning",
            "coreference-resolution",
        ],
        module_path="openbench.evals.glue",
        function_name="wsc",
        subtask=True,
    ),
    "cb": BenchmarkMetadata(
        name="CommitmentBank",
        description="Natural language inference with commitment",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="cb",
        subtask=True,
    ),
    "multirc": BenchmarkMetadata(
        name="MultiRC",
        description="Multi-Sentence Reading Comprehension",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="multirc",
        subtask=True,
    ),
    "superglue": BenchmarkMetadata(
        name="SuperGLUE (All Tasks)",
        description="SuperGLUE benchmark suite - run any subset by name (boolq, cb, copa, multirc, rte, wic, wsc)",
        category="glue",
        tags=["multiple-choice", "superglue", "nli", "reasoning"],
        module_path="openbench.evals.glue",
        function_name="superglue",
    ),
    "glue": BenchmarkMetadata(
        name="GLUE (All Tasks)",
        description="General Language Understanding Evaluation benchmark suite",
        category="glue",
        tags=["multiple-choice", "glue", "nli", "sentiment", "similarity"],
        module_path="openbench.evals.glue_standard",
        function_name="glue",
    ),
    "glue_cola": BenchmarkMetadata(
        name="GLUE: CoLA",
        description="Corpus of Linguistic Acceptability",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_cola",
        subtask=True,
    ),
    "glue_sst2": BenchmarkMetadata(
        name="GLUE: SST-2",
        description="Stanford Sentiment Treebank",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_sst2",
        subtask=True,
    ),
    "glue_mrpc": BenchmarkMetadata(
        name="GLUE: MRPC",
        description="Microsoft Research Paraphrase Corpus",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_mrpc",
        subtask=True,
    ),
    "glue_qqp": BenchmarkMetadata(
        name="GLUE: QQP",
        description="Quora Question Pairs",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_qqp",
        subtask=True,
    ),
    "glue_stsb": BenchmarkMetadata(
        name="GLUE: STS-B",
        description="Semantic Textual Similarity Benchmark",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_stsb",
        subtask=True,
    ),
    "glue_mnli": BenchmarkMetadata(
        name="GLUE: MNLI",
        description="Multi-Genre Natural Language Inference",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_mnli",
        subtask=True,
    ),
    "glue_mnli_mismatched": BenchmarkMetadata(
        name="GLUE: MNLI-MM",
        description="MNLI Mismatched",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_mnli_mismatched",
        subtask=True,
    ),
    "glue_qnli": BenchmarkMetadata(
        name="GLUE: QNLI",
        description="Question Natural Language Inference",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_qnli",
        subtask=True,
    ),
    "glue_rte": BenchmarkMetadata(
        name="GLUE: RTE",
        description="Recognizing Textual Entailment",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_rte",
        subtask=True,
    ),
    "glue_wnli": BenchmarkMetadata(
        name="GLUE: WNLI",
        description="Winograd Natural Language Inference",
        category="glue",
        tags=["multiple-choice", "glue", "nli"],
        module_path="openbench.evals.glue_standard",
        function_name="glue_wnli",
        subtask=True,
    ),
    # Cross-lingual Understanding benchmarks
    "xcopa": BenchmarkMetadata(
        name="XCOPA (11 Languages)",
        description="Cross-lingual Choice of Plausible Alternatives for causal commonsense reasoning",
        category="cross-lingual",
        tags=["multiple-choice", "causal-reasoning", "commonsense", "multilingual"],
        module_path="openbench.evals.xcopa",
        function_name="xcopa",
    ),
    "xcopa_et": BenchmarkMetadata(
        name="XCOPA: Estonian",
        description="XCOPA causal reasoning for Estonian (et)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_et",
        subtask=True,
    ),
    "xcopa_ht": BenchmarkMetadata(
        name="XCOPA: Haitian Creole",
        description="XCOPA causal reasoning for Haitian Creole (ht)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_ht",
        subtask=True,
    ),
    "xcopa_id": BenchmarkMetadata(
        name="XCOPA: Indonesian",
        description="XCOPA causal reasoning for Indonesian (id)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_id",
        subtask=True,
    ),
    "xcopa_it": BenchmarkMetadata(
        name="XCOPA: Italian",
        description="XCOPA causal reasoning for Italian (it)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_it",
        subtask=True,
    ),
    "xcopa_qu": BenchmarkMetadata(
        name="XCOPA: Quechua",
        description="XCOPA causal reasoning for Quechua (qu)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_qu",
        subtask=True,
    ),
    "xcopa_sw": BenchmarkMetadata(
        name="XCOPA: Swahili",
        description="XCOPA causal reasoning for Swahili (sw)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_sw",
        subtask=True,
    ),
    "xcopa_ta": BenchmarkMetadata(
        name="XCOPA: Tamil",
        description="XCOPA causal reasoning for Tamil (ta)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_ta",
        subtask=True,
    ),
    "xcopa_th": BenchmarkMetadata(
        name="XCOPA: Thai",
        description="XCOPA causal reasoning for Thai (th)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_th",
        subtask=True,
    ),
    "xcopa_tr": BenchmarkMetadata(
        name="XCOPA: Turkish",
        description="XCOPA causal reasoning for Turkish (tr)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_tr",
        subtask=True,
    ),
    "xcopa_vi": BenchmarkMetadata(
        name="XCOPA: Vietnamese",
        description="XCOPA causal reasoning for Vietnamese (vi)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_vi",
        subtask=True,
    ),
    "xcopa_zh": BenchmarkMetadata(
        name="XCOPA: Chinese",
        description="XCOPA causal reasoning for Chinese (zh)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "causal-reasoning",
            "commonsense",
            "multilingual",
            "xcopa",
        ],
        module_path="openbench.evals.xcopa",
        function_name="xcopa_zh",
        subtask=True,
    ),
    "xstorycloze": BenchmarkMetadata(
        name="XStoryCloze (11 Languages)",
        description="Cross-lingual story completion for commonsense reasoning",
        category="cross-lingual",
        tags=["multiple-choice", "story-completion", "commonsense", "multilingual"],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze",
    ),
    "xstorycloze_en": BenchmarkMetadata(
        name="XStoryCloze: English",
        description="XStoryCloze story completion for English (en)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_en",
        subtask=True,
    ),
    "xstorycloze_ru": BenchmarkMetadata(
        name="XStoryCloze: Russian",
        description="XStoryCloze story completion for Russian (ru)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_ru",
        subtask=True,
    ),
    "xstorycloze_zh": BenchmarkMetadata(
        name="XStoryCloze: Chinese",
        description="XStoryCloze story completion for Chinese (zh)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_zh",
        subtask=True,
    ),
    "xstorycloze_es": BenchmarkMetadata(
        name="XStoryCloze: Spanish",
        description="XStoryCloze story completion for Spanish (es)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_es",
        subtask=True,
    ),
    "xstorycloze_ar": BenchmarkMetadata(
        name="XStoryCloze: Arabic",
        description="XStoryCloze story completion for Arabic (ar)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_ar",
        subtask=True,
    ),
    "xstorycloze_hi": BenchmarkMetadata(
        name="XStoryCloze: Hindi",
        description="XStoryCloze story completion for Hindi (hi)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_hi",
        subtask=True,
    ),
    "xstorycloze_id": BenchmarkMetadata(
        name="XStoryCloze: Indonesian",
        description="XStoryCloze story completion for Indonesian (id)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_id",
        subtask=True,
    ),
    "xstorycloze_te": BenchmarkMetadata(
        name="XStoryCloze: Telugu",
        description="XStoryCloze story completion for Telugu (te)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_te",
        subtask=True,
    ),
    "xstorycloze_sw": BenchmarkMetadata(
        name="XStoryCloze: Swahili",
        description="XStoryCloze story completion for Swahili (sw)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_sw",
        subtask=True,
    ),
    "xstorycloze_eu": BenchmarkMetadata(
        name="XStoryCloze: Basque",
        description="XStoryCloze story completion for Basque (eu)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_eu",
        subtask=True,
    ),
    "xstorycloze_my": BenchmarkMetadata(
        name="XStoryCloze: Burmese",
        description="XStoryCloze story completion for Burmese (my)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "story-completion",
            "commonsense",
            "multilingual",
            "xstorycloze",
        ],
        module_path="openbench.evals.xstorycloze",
        function_name="xstorycloze_my",
        subtask=True,
    ),
    "xwinograd": BenchmarkMetadata(
        name="XWinograd (6 Languages)",
        description="Cross-lingual Winograd Schema Challenge for pronoun resolution",
        category="cross-lingual",
        tags=["multiple-choice", "pronoun-resolution", "commonsense", "multilingual"],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd",
    ),
    "xwinograd_en": BenchmarkMetadata(
        name="XWinograd: English",
        description="XWinograd pronoun resolution for English (en)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_en",
        subtask=True,
    ),
    "xwinograd_fr": BenchmarkMetadata(
        name="XWinograd: French",
        description="XWinograd pronoun resolution for French (fr)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_fr",
        subtask=True,
    ),
    "xwinograd_jp": BenchmarkMetadata(
        name="XWinograd: Japanese",
        description="XWinograd pronoun resolution for Japanese (jp)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_jp",
        subtask=True,
    ),
    "xwinograd_pt": BenchmarkMetadata(
        name="XWinograd: Portuguese",
        description="XWinograd pronoun resolution for Portuguese (pt)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_pt",
        subtask=True,
    ),
    "xwinograd_ru": BenchmarkMetadata(
        name="XWinograd: Russian",
        description="XWinograd pronoun resolution for Russian (ru)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_ru",
        subtask=True,
    ),
    "xwinograd_zh": BenchmarkMetadata(
        name="XWinograd: Chinese",
        description="XWinograd pronoun resolution for Chinese (zh)",
        category="cross-lingual",
        tags=[
            "multiple-choice",
            "pronoun-resolution",
            "commonsense",
            "multilingual",
            "xwinograd",
        ],
        module_path="openbench.evals.xwinograd",
        function_name="xwinograd_zh",
        subtask=True,
    ),  # Knowledge QA benchmarks
    "logiqa": BenchmarkMetadata(
        name="LogiQA",
        description="Logical reasoning dataset from Chinese civil service exam questions - tests deductive reasoning skills",
        category="knowledge-qa",
        tags=["multiple-choice", "logical-reasoning", "deduction", "critical-thinking"],
        module_path="openbench.evals.logiqa",
        function_name="logiqa",
    ),
    "math_qa": BenchmarkMetadata(
        name="MathQA",
        description="Mathematical word problems with multiple-choice answers and solution rationales",
        category="knowledge-qa",
        tags=["multiple-choice", "mathematics", "word-problems", "reasoning"],
        module_path="openbench.evals.mathqa",
        function_name="math_qa",
    ),
    "sciq": BenchmarkMetadata(
        name="SciQ",
        description="Science exam questions covering Physics, Chemistry, Biology, and other scientific domains",
        category="knowledge-qa",
        tags=["multiple-choice", "science", "physics", "chemistry", "biology"],
        module_path="openbench.evals.sciq",
        function_name="sciq",
    ),
    "truthfulqa": BenchmarkMetadata(
        name="TruthfulQA",
        description="Tests if models generate truthful answers to questions that humans often answer falsely due to misconceptions",
        category="knowledge-qa",
        tags=["multiple-choice", "truthfulness", "misconceptions", "factuality"],
        module_path="openbench.evals.truthfulqa",
        function_name="truthfulqa",
    ),
    # BLiMP: Benchmark of Linguistic Minimal Pairs
    "blimp": BenchmarkMetadata(
        name="BLiMP (67 Linguistic Phenomena)",
        description="Benchmark of Linguistic Minimal Pairs testing grammatical knowledge through minimal pair comparisons",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "morphology"],
        module_path="openbench.evals.blimp",
        function_name="blimp",
    ),
    "blimp_adjunct_island": BenchmarkMetadata(
        name="BLiMP: Adjunct island effects",
        description="BLiMP Adjunct island effects",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_adjunct_island",
        subtask=True,
    ),
    "blimp_anaphor_gender_agreement": BenchmarkMetadata(
        name="BLiMP: Anaphor gender agreement",
        description="BLiMP Anaphor gender agreement",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_anaphor_gender_agreement",
        subtask=True,
    ),
    "blimp_anaphor_number_agreement": BenchmarkMetadata(
        name="BLiMP: Anaphor number agreement",
        description="BLiMP Anaphor number agreement",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_anaphor_number_agreement",
        subtask=True,
    ),
    "blimp_animate_subject_passive": BenchmarkMetadata(
        name="BLiMP: Animate subject in passive constructions",
        description="BLiMP Animate subject in passive constructions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_animate_subject_passive",
        subtask=True,
    ),
    "blimp_animate_subject_trans": BenchmarkMetadata(
        name="BLiMP: Animate subject in transitive constructions",
        description="BLiMP Animate subject in transitive constructions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_animate_subject_trans",
        subtask=True,
    ),
    "blimp_causative": BenchmarkMetadata(
        name="BLiMP: Causative constructions",
        description="BLiMP Causative constructions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_causative",
        subtask=True,
    ),
    "blimp_complex_NP_island": BenchmarkMetadata(
        name="BLiMP: Complex NP island effects",
        description="BLiMP Complex NP island effects",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_complex_NP_island",
        subtask=True,
    ),
    "blimp_coordinate_structure_constraint_complex_left_branch": BenchmarkMetadata(
        name="BLiMP: Coordinate structure constraint - complex left branch",
        description="BLiMP Coordinate structure constraint - complex left branch",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_coordinate_structure_constraint_complex_left_branch",
        subtask=True,
    ),
    "blimp_coordinate_structure_constraint_object_extraction": BenchmarkMetadata(
        name="BLiMP: Coordinate structure constraint - object extraction",
        description="BLiMP Coordinate structure constraint - object extraction",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_coordinate_structure_constraint_object_extraction",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_1": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement (1)",
        description="BLiMP Determiner-noun agreement (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_1",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_2": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement (2)",
        description="BLiMP Determiner-noun agreement (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_2",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_irregular_1": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with irregular nouns (1)",
        description="BLiMP Determiner-noun agreement with irregular nouns (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_irregular_1",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_irregular_2": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with irregular nouns (2)",
        description="BLiMP Determiner-noun agreement with irregular nouns (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_irregular_2",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_with_adj_2": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with adjective (2)",
        description="BLiMP Determiner-noun agreement with adjective (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_with_adj_2",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_with_adj_irregular_1": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with adjective and irregular nouns (1)",
        description="BLiMP Determiner-noun agreement with adjective and irregular nouns (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_with_adj_irregular_1",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_with_adj_irregular_2": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with adjective and irregular nouns (2)",
        description="BLiMP Determiner-noun agreement with adjective and irregular nouns (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_with_adj_irregular_2",
        subtask=True,
    ),
    "blimp_determiner_noun_agreement_with_adjective_1": BenchmarkMetadata(
        name="BLiMP: Determiner-noun agreement with adjective (1)",
        description="BLiMP Determiner-noun agreement with adjective (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_determiner_noun_agreement_with_adjective_1",
        subtask=True,
    ),
    "blimp_distractor_agreement_relational_noun": BenchmarkMetadata(
        name="BLiMP: Distractor agreement with relational nouns",
        description="BLiMP Distractor agreement with relational nouns",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_distractor_agreement_relational_noun",
        subtask=True,
    ),
    "blimp_distractor_agreement_relative_clause": BenchmarkMetadata(
        name="BLiMP: Distractor agreement in relative clauses",
        description="BLiMP Distractor agreement in relative clauses",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_distractor_agreement_relative_clause",
        subtask=True,
    ),
    "blimp_drop_argument": BenchmarkMetadata(
        name="BLiMP: Dropped argument",
        description="BLiMP Dropped argument",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_drop_argument",
        subtask=True,
    ),
    "blimp_ellipsis_n_bar_1": BenchmarkMetadata(
        name="BLiMP: N-bar ellipsis (1)",
        description="BLiMP N-bar ellipsis (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_ellipsis_n_bar_1",
        subtask=True,
    ),
    "blimp_ellipsis_n_bar_2": BenchmarkMetadata(
        name="BLiMP: N-bar ellipsis (2)",
        description="BLiMP N-bar ellipsis (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_ellipsis_n_bar_2",
        subtask=True,
    ),
    "blimp_existential_there_object_raising": BenchmarkMetadata(
        name="BLiMP: Existential 'there' with object raising",
        description="BLiMP Existential 'there' with object raising",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_existential_there_object_raising",
        subtask=True,
    ),
    "blimp_existential_there_quantifiers_1": BenchmarkMetadata(
        name="BLiMP: Existential 'there' with quantifiers (1)",
        description="BLiMP Existential 'there' with quantifiers (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_existential_there_quantifiers_1",
        subtask=True,
    ),
    "blimp_existential_there_quantifiers_2": BenchmarkMetadata(
        name="BLiMP: Existential 'there' with quantifiers (2)",
        description="BLiMP Existential 'there' with quantifiers (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_existential_there_quantifiers_2",
        subtask=True,
    ),
    "blimp_existential_there_subject_raising": BenchmarkMetadata(
        name="BLiMP: Existential 'there' with subject raising",
        description="BLiMP Existential 'there' with subject raising",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_existential_there_subject_raising",
        subtask=True,
    ),
    "blimp_expletive_it_object_raising": BenchmarkMetadata(
        name="BLiMP: Expletive 'it' with object raising",
        description="BLiMP Expletive 'it' with object raising",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_expletive_it_object_raising",
        subtask=True,
    ),
    "blimp_inchoative": BenchmarkMetadata(
        name="BLiMP: Inchoative constructions",
        description="BLiMP Inchoative constructions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_inchoative",
        subtask=True,
    ),
    "blimp_intransitive": BenchmarkMetadata(
        name="BLiMP: Intransitive verbs",
        description="BLiMP Intransitive verbs",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_intransitive",
        subtask=True,
    ),
    "blimp_irregular_past_participle_adjectives": BenchmarkMetadata(
        name="BLiMP: Irregular past participles as adjectives",
        description="BLiMP Irregular past participles as adjectives",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_irregular_past_participle_adjectives",
        subtask=True,
    ),
    "blimp_irregular_past_participle_verbs": BenchmarkMetadata(
        name="BLiMP: Irregular past participles in verbs",
        description="BLiMP Irregular past participles in verbs",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_irregular_past_participle_verbs",
        subtask=True,
    ),
    "blimp_irregular_plural_subject_verb_agreement_1": BenchmarkMetadata(
        name="BLiMP: Subject-verb agreement with irregular plurals (1)",
        description="BLiMP Subject-verb agreement with irregular plurals (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_irregular_plural_subject_verb_agreement_1",
        subtask=True,
    ),
    "blimp_irregular_plural_subject_verb_agreement_2": BenchmarkMetadata(
        name="BLiMP: Subject-verb agreement with irregular plurals (2)",
        description="BLiMP Subject-verb agreement with irregular plurals (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_irregular_plural_subject_verb_agreement_2",
        subtask=True,
    ),
    "blimp_left_branch_island_echo_question": BenchmarkMetadata(
        name="BLiMP: Left branch island effects in echo questions",
        description="BLiMP Left branch island effects in echo questions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_left_branch_island_echo_question",
        subtask=True,
    ),
    "blimp_left_branch_island_simple_question": BenchmarkMetadata(
        name="BLiMP: Left branch island effects in simple questions",
        description="BLiMP Left branch island effects in simple questions",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_left_branch_island_simple_question",
        subtask=True,
    ),
    "blimp_matrix_question_npi_licensor_present": BenchmarkMetadata(
        name="BLiMP: Matrix question NPI licensor present",
        description="BLiMP Matrix question NPI licensor present",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_matrix_question_npi_licensor_present",
        subtask=True,
    ),
    "blimp_npi_present_1": BenchmarkMetadata(
        name="BLiMP: Negative polarity items present (1)",
        description="BLiMP Negative polarity items present (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_npi_present_1",
        subtask=True,
    ),
    "blimp_npi_present_2": BenchmarkMetadata(
        name="BLiMP: Negative polarity items present (2)",
        description="BLiMP Negative polarity items present (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_npi_present_2",
        subtask=True,
    ),
    "blimp_only_npi_licensor_present": BenchmarkMetadata(
        name="BLiMP: 'Only' as NPI licensor",
        description="BLiMP 'Only' as NPI licensor",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_only_npi_licensor_present",
        subtask=True,
    ),
    "blimp_only_npi_scope": BenchmarkMetadata(
        name="BLiMP: 'Only' NPI scope",
        description="BLiMP 'Only' NPI scope",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_only_npi_scope",
        subtask=True,
    ),
    "blimp_passive_1": BenchmarkMetadata(
        name="BLiMP: Passive constructions (1)",
        description="BLiMP Passive constructions (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_passive_1",
        subtask=True,
    ),
    "blimp_passive_2": BenchmarkMetadata(
        name="BLiMP: Passive constructions (2)",
        description="BLiMP Passive constructions (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_passive_2",
        subtask=True,
    ),
    "blimp_principle_A_c_command": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - c-command",
        description="BLiMP Binding Principle A - c-command",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_c_command",
        subtask=True,
    ),
    "blimp_principle_A_case_1": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - case (1)",
        description="BLiMP Binding Principle A - case (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_case_1",
        subtask=True,
    ),
    "blimp_principle_A_case_2": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - case (2)",
        description="BLiMP Binding Principle A - case (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_case_2",
        subtask=True,
    ),
    "blimp_principle_A_domain_1": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - domain (1)",
        description="BLiMP Binding Principle A - domain (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_domain_1",
        subtask=True,
    ),
    "blimp_principle_A_domain_2": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - domain (2)",
        description="BLiMP Binding Principle A - domain (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_domain_2",
        subtask=True,
    ),
    "blimp_principle_A_domain_3": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - domain (3)",
        description="BLiMP Binding Principle A - domain (3)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_domain_3",
        subtask=True,
    ),
    "blimp_principle_A_reconstruction": BenchmarkMetadata(
        name="BLiMP: Binding Principle A - reconstruction",
        description="BLiMP Binding Principle A - reconstruction",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_principle_A_reconstruction",
        subtask=True,
    ),
    "blimp_regular_plural_subject_verb_agreement_1": BenchmarkMetadata(
        name="BLiMP: Subject-verb agreement with regular plurals (1)",
        description="BLiMP Subject-verb agreement with regular plurals (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_regular_plural_subject_verb_agreement_1",
        subtask=True,
    ),
    "blimp_regular_plural_subject_verb_agreement_2": BenchmarkMetadata(
        name="BLiMP: Subject-verb agreement with regular plurals (2)",
        description="BLiMP Subject-verb agreement with regular plurals (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_regular_plural_subject_verb_agreement_2",
        subtask=True,
    ),
    "blimp_sentential_negation_npi_licensor_present": BenchmarkMetadata(
        name="BLiMP: Sentential negation as NPI licensor",
        description="BLiMP Sentential negation as NPI licensor",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_sentential_negation_npi_licensor_present",
        subtask=True,
    ),
    "blimp_sentential_negation_npi_scope": BenchmarkMetadata(
        name="BLiMP: Sentential negation NPI scope",
        description="BLiMP Sentential negation NPI scope",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_sentential_negation_npi_scope",
        subtask=True,
    ),
    "blimp_sentential_subject_island": BenchmarkMetadata(
        name="BLiMP: Sentential subject island effects",
        description="BLiMP Sentential subject island effects",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_sentential_subject_island",
        subtask=True,
    ),
    "blimp_superlative_quantifiers_1": BenchmarkMetadata(
        name="BLiMP: Superlative quantifiers (1)",
        description="BLiMP Superlative quantifiers (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_superlative_quantifiers_1",
        subtask=True,
    ),
    "blimp_superlative_quantifiers_2": BenchmarkMetadata(
        name="BLiMP: Superlative quantifiers (2)",
        description="BLiMP Superlative quantifiers (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_superlative_quantifiers_2",
        subtask=True,
    ),
    "blimp_tough_vs_raising_1": BenchmarkMetadata(
        name="BLiMP: Tough vs raising constructions (1)",
        description="BLiMP Tough vs raising constructions (1)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_tough_vs_raising_1",
        subtask=True,
    ),
    "blimp_tough_vs_raising_2": BenchmarkMetadata(
        name="BLiMP: Tough vs raising constructions (2)",
        description="BLiMP Tough vs raising constructions (2)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_tough_vs_raising_2",
        subtask=True,
    ),
    "blimp_transitive": BenchmarkMetadata(
        name="BLiMP: Transitive verbs",
        description="BLiMP Transitive verbs",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_transitive",
        subtask=True,
    ),
    "blimp_wh_island": BenchmarkMetadata(
        name="BLiMP: Wh-island effects",
        description="BLiMP Wh-island effects",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_island",
        subtask=True,
    ),
    "blimp_wh_questions_object_gap": BenchmarkMetadata(
        name="BLiMP: Wh-questions with object gap",
        description="BLiMP Wh-questions with object gap",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_questions_object_gap",
        subtask=True,
    ),
    "blimp_wh_questions_subject_gap": BenchmarkMetadata(
        name="BLiMP: Wh-questions with subject gap",
        description="BLiMP Wh-questions with subject gap",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_questions_subject_gap",
        subtask=True,
    ),
    "blimp_wh_questions_subject_gap_long_distance": BenchmarkMetadata(
        name="BLiMP: Wh-questions with long-distance subject gap",
        description="BLiMP Wh-questions with long-distance subject gap",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_questions_subject_gap_long_distance",
        subtask=True,
    ),
    "blimp_wh_vs_that_no_gap": BenchmarkMetadata(
        name="BLiMP: Wh vs that complementizers without gap",
        description="BLiMP Wh vs that complementizers without gap",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_vs_that_no_gap",
        subtask=True,
    ),
    "blimp_wh_vs_that_no_gap_long_distance": BenchmarkMetadata(
        name="BLiMP: Wh vs that complementizers without gap (long-distance)",
        description="BLiMP Wh vs that complementizers without gap (long-distance)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_vs_that_no_gap_long_distance",
        subtask=True,
    ),
    "blimp_wh_vs_that_with_gap": BenchmarkMetadata(
        name="BLiMP: Wh vs that complementizers with gap",
        description="BLiMP Wh vs that complementizers with gap",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_vs_that_with_gap",
        subtask=True,
    ),
    "blimp_wh_vs_that_with_gap_long_distance": BenchmarkMetadata(
        name="BLiMP: Wh vs that complementizers with gap (long-distance)",
        description="BLiMP Wh vs that complementizers with gap (long-distance)",
        category="linguistic",
        tags=["multiple-choice", "linguistics", "grammar", "syntax", "blimp"],
        module_path="openbench.evals.blimp",
        function_name="blimp_wh_vs_that_with_gap_long_distance",
        subtask=True,
    ),
    # Reading Comprehension benchmarks
    # RACE family aggregate
    "race": BenchmarkMetadata(
        name="RACE",
        description="Reading comprehension from middle and high school English exams (combined)",
        category="reading-comprehension",
        tags=["multiple-choice", "reading-comprehension", "english-exam"],
        module_path="openbench.evals.race",
        function_name="race",
    ),
    "race_high": BenchmarkMetadata(
        name="RACE-High",
        description="High school level reading comprehension from English exams for Chinese students - passages with multiple questions",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "english-exam",
            "high-school",
        ],
        module_path="openbench.evals.race",
        function_name="race_high",
        subtask=True,
    ),
    "race_middle": BenchmarkMetadata(
        name="RACE-Middle",
        description="Middle school level reading comprehension from English exams for Chinese students",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "english-exam",
            "middle-school",
        ],
        module_path="openbench.evals.race",
        function_name="race_middle",
        subtask=True,
    ),
    "qa4mre": BenchmarkMetadata(
        name="QA4MRE (All Years)",
        description="Question Answering for Machine Reading Evaluation - CLEF shared tasks 2011-2013",
        category="reading-comprehension",
        tags=["multiple-choice", "reading-comprehension", "clef", "machine-reading"],
        module_path="openbench.evals.qa4mre",
        function_name="qa4mre",
    ),
    "qa4mre_2011": BenchmarkMetadata(
        name="QA4MRE 2011",
        description="Question Answering for Machine Reading Evaluation (English, 2011)",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "clef",
            "machine-reading",
            "qa4mre",
        ],
        module_path="openbench.evals.qa4mre",
        function_name="qa4mre_2011",
        subtask=True,
    ),
    "qa4mre_2012": BenchmarkMetadata(
        name="QA4MRE 2012",
        description="Question Answering for Machine Reading Evaluation (English, 2012)",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "clef",
            "machine-reading",
            "qa4mre",
        ],
        module_path="openbench.evals.qa4mre",
        function_name="qa4mre_2012",
        subtask=True,
    ),
    "qa4mre_2013": BenchmarkMetadata(
        name="QA4MRE 2013",
        description="Question Answering for Machine Reading Evaluation (English, 2013)",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "clef",
            "machine-reading",
            "qa4mre",
        ],
        module_path="openbench.evals.qa4mre",
        function_name="qa4mre_2013",
        subtask=True,
    ),
    "qasper_ll": BenchmarkMetadata(
        name="QASPER (Binary MCQ)",
        description="Question Answering on Scientific Papers - binary yes/no questions on research paper abstracts",
        category="reading-comprehension",
        tags=[
            "multiple-choice",
            "reading-comprehension",
            "scientific-papers",
            "binary-classification",
        ],
        module_path="openbench.evals.qasper",
        function_name="qasper_ll",
    ),
    # Ethics & Social Understanding benchmarks
    "ethics": BenchmarkMetadata(
        name="ETHICS (All Dimensions)",
        description="Aligning AI With Shared Human Values - tests moral reasoning across 5 fundamental dimensions",
        category="ethics-social",
        tags=["multiple-choice", "ethics", "moral-reasoning", "philosophy"],
        module_path="openbench.evals.ethics",
        function_name="ethics",
    ),
    "ethics_justice": BenchmarkMetadata(
        name="ETHICS: Justice",
        description="Tests fairness and impartiality in ethical decision-making",
        category="ethics-social",
        tags=["multiple-choice", "ethics", "moral-reasoning", "justice", "ethics"],
        module_path="openbench.evals.ethics",
        function_name="ethics_justice",
        subtask=True,
    ),
    "ethics_deontology": BenchmarkMetadata(
        name="ETHICS: Deontology",
        description="Tests duty-based ethics and understanding of moral rules",
        category="ethics-social",
        tags=["multiple-choice", "ethics", "moral-reasoning", "deontology", "ethics"],
        module_path="openbench.evals.ethics",
        function_name="ethics_deontology",
        subtask=True,
    ),
    "ethics_virtue": BenchmarkMetadata(
        name="ETHICS: Virtue",
        description="Tests character-based ethics and recognition of virtuous behavior",
        category="ethics-social",
        tags=["multiple-choice", "ethics", "moral-reasoning", "virtue", "ethics"],
        module_path="openbench.evals.ethics",
        function_name="ethics_virtue",
        subtask=True,
    ),
    "ethics_utilitarianism": BenchmarkMetadata(
        name="ETHICS: Utilitarianism",
        description="Tests consequence-based ethics and utility maximization",
        category="ethics-social",
        tags=[
            "multiple-choice",
            "ethics",
            "moral-reasoning",
            "utilitarianism",
            "ethics",
        ],
        module_path="openbench.evals.ethics",
        function_name="ethics_utilitarianism",
        subtask=True,
    ),
    "ethics_commonsense": BenchmarkMetadata(
        name="ETHICS: Commonsense",
        description="Tests everyday moral reasoning and common ethical intuitions",
        category="ethics-social",
        tags=["multiple-choice", "ethics", "moral-reasoning", "commonsense", "ethics"],
        module_path="openbench.evals.ethics",
        function_name="ethics_commonsense",
        subtask=True,
    ),
    "social_iqa": BenchmarkMetadata(
        name="Social IQA",
        description="Social Intelligence Question Answering - tests reasoning about social situations, emotions, and mental states",
        category="ethics-social",
        tags=[
            "multiple-choice",
            "social-intelligence",
            "emotional-reasoning",
            "theory-of-mind",
        ],
        module_path="openbench.evals.social_iqa",
        function_name="social_iqa",
    ),
    "toxigen": BenchmarkMetadata(
        name="ToxiGen",
        description="Toxicity detection benchmark - tests ability to identify toxic and hateful language",
        category="ethics-social",
        tags=["multiple-choice", "toxicity-detection", "hate-speech", "safety"],
        module_path="openbench.evals.toxigen",
        function_name="toxigen",
    ),
    # AGIEval benchmarks
    "agieval": BenchmarkMetadata(
        name="AGIEval (All Subsets)",
        description="Human-centric benchmark with 17 official qualifying exam questions testing general cognitive abilities",
        category="agieval",
        tags=["multiple-choice", "academic-exams", "reasoning", "cognitive-abilities"],
        module_path="openbench.evals.agieval",
        function_name="agieval",
    ),
    "agieval_aqua_rat": BenchmarkMetadata(
        name="AGIEval: AQUA-RAT",
        description="Algebraic question answering and reasoning",
        category="agieval",
        tags=["multiple-choice", "algebra", "reasoning", "math", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_aqua_rat",
        subtask=True,
    ),
    "agieval_logiqa_en": BenchmarkMetadata(
        name="AGIEval: LogiQA (English)",
        description="Logical reasoning questions in English",
        category="agieval",
        tags=["multiple-choice", "logic", "reasoning", "english", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_logiqa_en",
        subtask=True,
    ),
    "agieval_logiqa_zh": BenchmarkMetadata(
        name="AGIEval: LogiQA (Chinese)",
        description="Logical reasoning questions in Chinese",
        category="agieval",
        tags=["multiple-choice", "logic", "reasoning", "chinese", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_logiqa_zh",
        subtask=True,
    ),
    "agieval_lsat_ar": BenchmarkMetadata(
        name="AGIEval: LSAT Analytical Reasoning",
        description="Law School Admission Test - Analytical Reasoning section",
        category="agieval",
        tags=["multiple-choice", "law", "analytical-reasoning", "lsat", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_lsat_ar",
        subtask=True,
    ),
    "agieval_lsat_lr": BenchmarkMetadata(
        name="AGIEval: LSAT Logical Reasoning",
        description="Law School Admission Test - Logical Reasoning section",
        category="agieval",
        tags=["multiple-choice", "law", "logical-reasoning", "lsat", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_lsat_lr",
        subtask=True,
    ),
    "agieval_lsat_rc": BenchmarkMetadata(
        name="AGIEval: LSAT Reading Comprehension",
        description="Law School Admission Test - Reading Comprehension section",
        category="agieval",
        tags=["multiple-choice", "law", "reading-comprehension", "lsat", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_lsat_rc",
        subtask=True,
    ),
    "agieval_sat_en": BenchmarkMetadata(
        name="AGIEval: SAT English",
        description="Scholastic Assessment Test - English section",
        category="agieval",
        tags=["multiple-choice", "sat", "english", "reading", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_sat_en",
        subtask=True,
    ),
    "agieval_sat_en_without_passage": BenchmarkMetadata(
        name="AGIEval: SAT English (No Passage)",
        description="SAT English questions without reading passages",
        category="agieval",
        tags=["multiple-choice", "sat", "english", "grammar", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_sat_en_without_passage",
        subtask=True,
    ),
    "agieval_sat_math": BenchmarkMetadata(
        name="AGIEval: SAT Math",
        description="Scholastic Assessment Test - Math section",
        category="agieval",
        tags=["multiple-choice", "sat", "mathematics", "problem-solving", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_sat_math",
        subtask=True,
    ),
    "agieval_gaokao_biology": BenchmarkMetadata(
        name="AGIEval: Gaokao Biology",
        description="Chinese national college entrance exam - Biology",
        category="agieval",
        tags=["multiple-choice", "gaokao", "biology", "science", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_biology",
        subtask=True,
    ),
    "agieval_gaokao_chemistry": BenchmarkMetadata(
        name="AGIEval: Gaokao Chemistry",
        description="Chinese national college entrance exam - Chemistry",
        category="agieval",
        tags=["multiple-choice", "gaokao", "chemistry", "science", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_chemistry",
        subtask=True,
    ),
    "agieval_gaokao_chinese": BenchmarkMetadata(
        name="AGIEval: Gaokao Chinese",
        description="Chinese national college entrance exam - Chinese language",
        category="agieval",
        tags=["multiple-choice", "gaokao", "chinese", "language", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_chinese",
        subtask=True,
    ),
    "agieval_gaokao_english": BenchmarkMetadata(
        name="AGIEval: Gaokao English",
        description="Chinese national college entrance exam - English",
        category="agieval",
        tags=["multiple-choice", "gaokao", "english", "language", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_english",
        subtask=True,
    ),
    "agieval_gaokao_geography": BenchmarkMetadata(
        name="AGIEval: Gaokao Geography",
        description="Chinese national college entrance exam - Geography",
        category="agieval",
        tags=["multiple-choice", "gaokao", "geography", "social-studies", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_geography",
        subtask=True,
    ),
    "agieval_gaokao_history": BenchmarkMetadata(
        name="AGIEval: Gaokao History",
        description="Chinese national college entrance exam - History",
        category="agieval",
        tags=["multiple-choice", "gaokao", "history", "social-studies", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_history",
        subtask=True,
    ),
    "agieval_gaokao_mathqa": BenchmarkMetadata(
        name="AGIEval: Gaokao Math",
        description="Chinese national college entrance exam - Mathematics",
        category="agieval",
        tags=["multiple-choice", "gaokao", "mathematics", "problem-solving", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_mathqa",
        subtask=True,
    ),
    "agieval_gaokao_physics": BenchmarkMetadata(
        name="AGIEval: Gaokao Physics",
        description="Chinese national college entrance exam - Physics",
        category="agieval",
        tags=["multiple-choice", "gaokao", "physics", "science", "agieval"],
        module_path="openbench.evals.agieval",
        function_name="agieval_gaokao_physics",
        subtask=True,
    ),
    "legalsupport": BenchmarkMetadata(
        name="LegalSupport",
        description="Legal citation support identification - identify which citation provides stronger support for a legal argument",
        category="domain-specific",
        tags=["multiple-choice", "legal", "reasoning", "citation-analysis"],
        module_path="openbench.evals.legalsupport",
        function_name="legalsupport",
    ),
    "arabic_exams": BenchmarkMetadata(
        name="Arabic Exams (40 Subjects)",
        description="Multi-task Arabic language understanding benchmark from school exams across North Africa, the Levant, and the Gulf",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "multilingual", "education", "msa"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams",
    ),
    "arabic_exams_accounting_university": BenchmarkMetadata(
        name="Arabic Exams: Accounting (University)",
        description="Arabic MMLU - Accounting questions from university-level exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "accounting", "university"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_accounting_university",
        subtask=True,
    ),
    "arabic_exams_arabic_language_general": BenchmarkMetadata(
        name="Arabic Exams: Arabic Language (General)",
        description="Arabic MMLU - Arabic language questions from general exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "language", "general"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_arabic_language_general",
        subtask=True,
    ),
    "arabic_exams_computer_science_high_school": BenchmarkMetadata(
        name="Arabic Exams: Computer Science (High School)",
        description="Arabic MMLU - Computer science questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "computer-science", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_computer_science_high_school",
        subtask=True,
    ),
    "arabic_exams_computer_science_university": BenchmarkMetadata(
        name="Arabic Exams: Computer Science (University)",
        description="Arabic MMLU - Computer science questions from university-level exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "computer-science", "university"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_computer_science_university",
        subtask=True,
    ),
    "arabic_exams_islamic_studies_general": BenchmarkMetadata(
        name="Arabic Exams: Islamic Studies (General)",
        description="Arabic MMLU - Islamic studies questions from general exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "islamic-studies", "general"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_islamic_studies_general",
        subtask=True,
    ),
    "arabic_exams_math_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Math (Primary School)",
        description="Arabic MMLU - Math questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "math", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_math_primary_school",
        subtask=True,
    ),
    "arabic_exams_math_high_school": BenchmarkMetadata(
        name="Arabic Exams: Math (High School)",
        description="Arabic MMLU - Math questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "math", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_math_high_school",
        subtask=True,
    ),
    "arabic_exams_physics_high_school": BenchmarkMetadata(
        name="Arabic Exams: Physics (High School)",
        description="Arabic MMLU - Physics questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "physics", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_physics_high_school",
        subtask=True,
    ),
    # Arabic Language (additional subsets)
    "arabic_exams_arabic_language_grammar": BenchmarkMetadata(
        name="Arabic Exams: Arabic Language (Grammar)",
        description="Arabic MMLU - Arabic language grammar questions",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "language", "grammar"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_arabic_language_grammar",
        subtask=True,
    ),
    "arabic_exams_arabic_language_high_school": BenchmarkMetadata(
        name="Arabic Exams: Arabic Language (High School)",
        description="Arabic MMLU - Arabic language questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "language", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_arabic_language_high_school",
        subtask=True,
    ),
    "arabic_exams_arabic_language_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Arabic Language (Middle School)",
        description="Arabic MMLU - Arabic language questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "language", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_arabic_language_middle_school",
        subtask=True,
    ),
    "arabic_exams_arabic_language_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Arabic Language (Primary School)",
        description="Arabic MMLU - Arabic language questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "language", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_arabic_language_primary_school",
        subtask=True,
    ),
    # Biology
    "arabic_exams_biology_high_school": BenchmarkMetadata(
        name="Arabic Exams: Biology (High School)",
        description="Arabic MMLU - Biology questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "biology", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_biology_high_school",
        subtask=True,
    ),
    # Civics
    "arabic_exams_civics_high_school": BenchmarkMetadata(
        name="Arabic Exams: Civics (High School)",
        description="Arabic MMLU - Civics questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "civics", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_civics_high_school",
        subtask=True,
    ),
    "arabic_exams_civics_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Civics (Middle School)",
        description="Arabic MMLU - Civics questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "civics", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_civics_middle_school",
        subtask=True,
    ),
    # Computer Science (additional subsets)
    "arabic_exams_computer_science_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Computer Science (Middle School)",
        description="Arabic MMLU - Computer science questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "computer-science", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_computer_science_middle_school",
        subtask=True,
    ),
    "arabic_exams_computer_science_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Computer Science (Primary School)",
        description="Arabic MMLU - Computer science questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "computer-science", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_computer_science_primary_school",
        subtask=True,
    ),
    # Driving Test
    "arabic_exams_driving_test": BenchmarkMetadata(
        name="Arabic Exams: Driving Test",
        description="Arabic MMLU - Driving test questions",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "driving"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_driving_test",
        subtask=True,
    ),
    # Economics
    "arabic_exams_economics_high_school": BenchmarkMetadata(
        name="Arabic Exams: Economics (High School)",
        description="Arabic MMLU - Economics questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "economics", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_economics_high_school",
        subtask=True,
    ),
    "arabic_exams_economics_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Economics (Middle School)",
        description="Arabic MMLU - Economics questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "economics", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_economics_middle_school",
        subtask=True,
    ),
    "arabic_exams_economics_university": BenchmarkMetadata(
        name="Arabic Exams: Economics (University)",
        description="Arabic MMLU - Economics questions from university-level exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "economics", "university"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_economics_university",
        subtask=True,
    ),
    # General Knowledge
    "arabic_exams_general_knowledge": BenchmarkMetadata(
        name="Arabic Exams: General Knowledge",
        description="Arabic MMLU - General knowledge questions",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "general-knowledge"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_general_knowledge",
        subtask=True,
    ),
    "arabic_exams_general_knowledge_middle_school": BenchmarkMetadata(
        name="Arabic Exams: General Knowledge (Middle School)",
        description="Arabic MMLU - General knowledge questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "general-knowledge", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_general_knowledge_middle_school",
        subtask=True,
    ),
    "arabic_exams_general_knowledge_primary_school": BenchmarkMetadata(
        name="Arabic Exams: General Knowledge (Primary School)",
        description="Arabic MMLU - General knowledge questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "general-knowledge", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_general_knowledge_primary_school",
        subtask=True,
    ),
    # Geography
    "arabic_exams_geography_high_school": BenchmarkMetadata(
        name="Arabic Exams: Geography (High School)",
        description="Arabic MMLU - Geography questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "geography", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_geography_high_school",
        subtask=True,
    ),
    "arabic_exams_geography_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Geography (Middle School)",
        description="Arabic MMLU - Geography questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "geography", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_geography_middle_school",
        subtask=True,
    ),
    "arabic_exams_geography_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Geography (Primary School)",
        description="Arabic MMLU - Geography questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "geography", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_geography_primary_school",
        subtask=True,
    ),
    # History
    "arabic_exams_history_high_school": BenchmarkMetadata(
        name="Arabic Exams: History (High School)",
        description="Arabic MMLU - History questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "history", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_history_high_school",
        subtask=True,
    ),
    "arabic_exams_history_middle_school": BenchmarkMetadata(
        name="Arabic Exams: History (Middle School)",
        description="Arabic MMLU - History questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "history", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_history_middle_school",
        subtask=True,
    ),
    "arabic_exams_history_primary_school": BenchmarkMetadata(
        name="Arabic Exams: History (Primary School)",
        description="Arabic MMLU - History questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "history", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_history_primary_school",
        subtask=True,
    ),
    # Islamic Studies (additional subsets)
    "arabic_exams_islamic_studies_high_school": BenchmarkMetadata(
        name="Arabic Exams: Islamic Studies (High School)",
        description="Arabic MMLU - Islamic studies questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "islamic-studies", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_islamic_studies_high_school",
        subtask=True,
    ),
    "arabic_exams_islamic_studies_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Islamic Studies (Middle School)",
        description="Arabic MMLU - Islamic studies questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "islamic-studies", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_islamic_studies_middle_school",
        subtask=True,
    ),
    "arabic_exams_islamic_studies_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Islamic Studies (Primary School)",
        description="Arabic MMLU - Islamic studies questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "islamic-studies", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_islamic_studies_primary_school",
        subtask=True,
    ),
    # Law
    "arabic_exams_law_professional": BenchmarkMetadata(
        name="Arabic Exams: Law (Professional)",
        description="Arabic MMLU - Law questions from professional exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "law", "professional"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_law_professional",
        subtask=True,
    ),
    # Management
    "arabic_exams_management_university": BenchmarkMetadata(
        name="Arabic Exams: Management (University)",
        description="Arabic MMLU - Management questions from university-level exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "management", "university"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_management_university",
        subtask=True,
    ),
    # Natural Science
    "arabic_exams_natural_science_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Natural Science (Middle School)",
        description="Arabic MMLU - Natural science questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "natural-science", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_natural_science_middle_school",
        subtask=True,
    ),
    "arabic_exams_natural_science_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Natural Science (Primary School)",
        description="Arabic MMLU - Natural science questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "natural-science", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_natural_science_primary_school",
        subtask=True,
    ),
    # Philosophy
    "arabic_exams_philosophy_high_school": BenchmarkMetadata(
        name="Arabic Exams: Philosophy (High School)",
        description="Arabic MMLU - Philosophy questions from high school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "philosophy", "high-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_philosophy_high_school",
        subtask=True,
    ),
    # Political Science
    "arabic_exams_political_science_university": BenchmarkMetadata(
        name="Arabic Exams: Political Science (University)",
        description="Arabic MMLU - Political science questions from university-level exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "political-science", "university"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_political_science_university",
        subtask=True,
    ),
    # Social Science
    "arabic_exams_social_science_middle_school": BenchmarkMetadata(
        name="Arabic Exams: Social Science (Middle School)",
        description="Arabic MMLU - Social science questions from middle school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "social-science", "middle-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_social_science_middle_school",
        subtask=True,
    ),
    "arabic_exams_social_science_primary_school": BenchmarkMetadata(
        name="Arabic Exams: Social Science (Primary School)",
        description="Arabic MMLU - Social science questions from primary school exams",
        category="domain-specific",
        tags=["multiple-choice", "arabic", "social-science", "primary-school"],
        module_path="openbench.evals.arabic_exams",
        function_name="arabic_exams_social_science_primary_school",
        subtask=True,
    ),
    "otis_mock_aime": BenchmarkMetadata(
        name="MockAIME (2024-2025)",
        description="Otis Mock AIME - a benchmark from the OTIS Mock AIME 2024-2025 exams",
        category="math",
        tags=["aime", "problem-solving", "math", "2024-2025"],
        module_path="openbench.evals.mockaime",
        function_name="otis_mock_aime",
    ),
    "otis_mock_aime_2024": BenchmarkMetadata(
        name="MockAIME (2024)",
        description="Otis Mock AIME - a benchmark from the OTIS Mock AIME 2024 exam",
        category="math",
        tags=["aime", "problem-solving", "math", "2024"],
        module_path="openbench.evals.mockaime",
        function_name="otis_mock_aime_2024",
        subtask=True,
    ),
    "otis_mock_aime_2025": BenchmarkMetadata(
        name="MockAIME (2025)",
        description="Otis Mock AIME - a benchmark from the OTIS Mock AIME 2025 exams",
        category="math",
        tags=["aime", "problem-solving", "math", "2025"],
        module_path="openbench.evals.mockaime",
        function_name="otis_mock_aime_2025",
        subtask=True,
    ),
    "multichallenge": BenchmarkMetadata(
        name="MultiChallenge",
        description="Multi-turn conversational tasks requiring reasoning, instruction retention, and coherence.",
        category="core",
        tags=["multi-turn", "reasoning", "chat", "judged"],
        module_path="openbench.evals.multichallenge",
        function_name="multichallenge",
        is_alpha=False,
    ),
}


def _normalize_benchmark_key(name: str) -> str:
    """Normalize benchmark keys so '-' and '_' are treated the same."""

    return name.replace("-", "_")


def _merge_benchmarks_with_normalization(
    builtin: dict[str, BenchmarkMetadata],
    entry_points: dict[str, BenchmarkMetadata],
) -> dict[str, BenchmarkMetadata]:
    """Merge benchmark dicts, treating '-' and '_' as equivalent for overrides.

    Entry point benchmarks can override built-ins even if they differ only in
    '-' vs '_'. The entry point's key format wins.

    When multiple entry points normalize to the same key, the last one wins
    (consistent with dict merge semantics).

    Args:
        builtin: Built-in benchmark metadata
        entry_points: Entry point benchmark metadata

    Returns:
        Merged benchmark dict with entry point overrides applied
    """
    merged = dict(builtin)

    # Build a reverse lookup: normalized_key -> original_key (tracks ALL keys processed)
    normalized_lookup = {_normalize_benchmark_key(k): k for k in builtin.keys()}

    for ep_key, ep_meta in entry_points.items():
        normalized = _normalize_benchmark_key(ep_key)

        # Check if there's an existing key that normalizes to the same value
        if normalized in normalized_lookup:
            old_key = normalized_lookup[normalized]
            # Remove the old key if it's different from the new one
            if old_key != ep_key and old_key in merged:
                del merged[old_key]

        # Update lookup and merged dict
        normalized_lookup[normalized] = ep_key
        merged[ep_key] = ep_meta

    return merged


# Merge built-in benchmarks with those from entry points.
# Entry points are merged last so they can override built-ins. This allows external
# packages to patch/extend benchmarks (e.g., fixing dataset bugs, adding custom splits,
# or swapping implementations). If you want stable behavior, pin your dependencies.
# Keys differing only in '-' vs '_' are treated as referring to the same benchmark,
# with the entry point's key format taking precedence.
BENCHMARKS = _merge_benchmarks_with_normalization(
    _BUILTIN_BENCHMARKS,
    _load_entry_point_benchmarks(),
)


def _build_normalized_lookup(names: Iterable[str]) -> dict[str, str]:
    """Build a lookup mapping normalized benchmark keys to canonical names."""

    lookup: dict[str, str] = {}
    for key in names:
        normalized = _normalize_benchmark_key(key)
        existing = lookup.get(normalized)
        if existing and existing != key:
            raise ValueError(
                "Benchmark names cannot differ only by '-' vs '_' ("
                f"conflict between '{existing}' and '{key}')"
            )
        lookup[normalized] = key
    return lookup


_NORMALIZED_BENCHMARK_NAMES = _build_normalized_lookup(BENCHMARKS.keys())


def get_benchmark_metadata(name: str) -> Optional[BenchmarkMetadata]:
    """Get benchmark metadata by name."""
    metadata = BENCHMARKS.get(name)
    if metadata:
        return metadata

    canonical_name = _NORMALIZED_BENCHMARK_NAMES.get(_normalize_benchmark_key(name))
    if canonical_name:
        return BENCHMARKS[canonical_name]

    return None


def get_all_benchmarks(include_alpha: bool = False) -> dict[str, BenchmarkMetadata]:
    """Get all benchmark metadata.

    Args:
        include_alpha: Whether to include alpha/experimental benchmarks
    """
    if include_alpha:
        return BENCHMARKS
    return {name: meta for name, meta in BENCHMARKS.items() if not meta.is_alpha}


def get_benchmarks_by_category(
    category: str, include_alpha: bool = False
) -> dict[str, BenchmarkMetadata]:
    """Get all benchmarks in a category.

    Args:
        category: Category to filter by
        include_alpha: Whether to include alpha/experimental benchmarks
    """
    results = {
        name: meta for name, meta in BENCHMARKS.items() if meta.category == category
    }
    if not include_alpha:
        results = {name: meta for name, meta in results.items() if not meta.is_alpha}
    return results


def get_categories() -> List[str]:
    """Get all available categories."""
    return sorted(list(set(meta.category for meta in BENCHMARKS.values())))


def search_benchmarks(
    query: str, include_alpha: bool = False
) -> dict[str, BenchmarkMetadata]:
    """Search benchmarks by name, description, or tags.

    Args:
        query: Search query
        include_alpha: Whether to include alpha/experimental benchmarks
    """
    query = query.lower()
    results = {}

    for name, meta in BENCHMARKS.items():
        if not include_alpha and meta.is_alpha:
            continue
        if (
            query in meta.name.lower()
            or query in meta.description.lower()
            or any(query in tag.lower() for tag in meta.tags)
        ):
            results[name] = meta

    return results


# ============================================================================
# Task Loading for CLI
# ============================================================================


def _generate_task_registry(include_alpha: bool = True):
    """Generate task registry from config.

    Args:
        include_alpha: Whether to include alpha/experimental benchmarks
    """
    registry = {}
    for name, metadata in get_all_benchmarks(include_alpha=include_alpha).items():
        registry[name] = f"{metadata.module_path}.{metadata.function_name}"
    return registry


# Full registry including alpha benchmarks for backward compatibility
TASK_REGISTRY = _generate_task_registry(include_alpha=True)

_NORMALIZED_TASK_REGISTRY = _build_normalized_lookup(TASK_REGISTRY.keys())


def _import_module_from_path(path: Path) -> ModuleType:
    """
    Import a .py file or package directory as an anonymous module.
    """
    file_path = path
    if path.is_dir():
        file_path = path / "__init__.py"
        if not file_path.exists():
            raise ValueError(f"{path} is a directory but has no __init__.py")

    mod_name = f"_openbench_dyn_{uuid.uuid4().hex}"
    spec = importlib.util.spec_from_file_location(mod_name, str(file_path))
    if spec is None or spec.loader is None:
        raise ImportError(f"Cannot create import spec for {file_path}")

    module = importlib.util.module_from_spec(spec)

    # For packages, set up proper package structure for relative imports
    if path.is_dir():
        module.__package__ = mod_name
        sys.modules[mod_name] = module

        # Pre-load submodules to support relative imports
        for submodule_file in path.glob("*.py"):
            if submodule_file.name != "__init__.py":
                submodule_name = submodule_file.stem
                submodule_full_name = f"{mod_name}.{submodule_name}"
                submodule_spec = importlib.util.spec_from_file_location(
                    submodule_full_name, str(submodule_file)
                )
                if submodule_spec and submodule_spec.loader:
                    submodule = importlib.util.module_from_spec(submodule_spec)
                    submodule.__package__ = mod_name
                    sys.modules[submodule_full_name] = submodule
                    submodule_spec.loader.exec_module(submodule)
    else:
        sys.modules[mod_name] = module

    spec.loader.exec_module(module)
    return module


@lru_cache()
def load_task(benchmark_name: str, allow_alpha: bool = False) -> Callable:
    """
    Loads a task by benchmark name using the registry or from a local path.

    Args:
        benchmark_name (str): The name of the benchmark or path to a local eval.
        allow_alpha (bool): Whether to allow loading alpha/experimental benchmarks.

    Returns:
        Callable: The imported function object.

    Raises:
        ValueError: If the benchmark is not in the registry and not a valid path.
        ImportError: If the module cannot be imported.
        AttributeError: If the function does not exist in the module.
    """
    # Check if this is an alpha benchmark
    benchmark_meta = get_benchmark_metadata(benchmark_name)
    if benchmark_meta and benchmark_meta.is_alpha and not allow_alpha:
        raise ValueError(
            f"'{benchmark_name}' is an experimental/alpha benchmark. "
            f"Use --alpha flag to run it."
        )

    # Try registry first (registry names take precedence)
    import_path = TASK_REGISTRY.get(benchmark_name)
    if import_path is None:
        canonical_name = _NORMALIZED_TASK_REGISTRY.get(
            _normalize_benchmark_key(benchmark_name)
        )
        if canonical_name:
            import_path = TASK_REGISTRY[canonical_name]
            benchmark_name = canonical_name
    if import_path:
        module_path, func_name = import_path.rsplit(".", 1)
        module = importlib.import_module(module_path)
        return getattr(module, func_name)

    # Fallback to path-based loading
    path = Path(benchmark_name).expanduser()
    if path.exists():
        return _load_task_from_local_path(path)

    # Neither registry nor valid path
    raise ValueError(
        f"Unknown benchmark: '{benchmark_name}'. "
        # return available benchmarks alphabetically
        f"Available benchmarks: {', '.join(sorted(TASK_REGISTRY.keys()))}"
    )


def _load_task_from_local_path(path: Path) -> Callable:
    """
    Load a task from a local path containing __metadata__.

    Args:
        path: Path to a directory or .py file containing an eval

    Returns:
        Callable: The imported function object

    Raises:
        ValueError: If no valid __metadata__ is found
        AttributeError: If the function does not exist in the module
        ImportError: If the module cannot be imported
    """
    root_module = _import_module_from_path(path)
    metadata = getattr(root_module, "__metadata__", None)

    if not isinstance(metadata, BenchmarkMetadata):
        raise ValueError(f"{path} has no valid __metadata__")

    # Resolve module path relative to root module
    # For local evals, module_path is typically relative like "simpleqa.simpleqa"
    # We need to extract just the last part and combine with the root module name
    if metadata.module_path.startswith(root_module.__name__):
        full_module_name = metadata.module_path
    else:
        # For paths like "simpleqa.simpleqa", we want the last component "simpleqa"
        module_components = metadata.module_path.split(".")
        module_name = module_components[-1]  # Take the last component
        full_module_name = f"{root_module.__name__}.{module_name}"

    try:
        module = importlib.import_module(full_module_name)
    except ImportError as e:
        raise ImportError(f"Cannot import module '{full_module_name}': {e}")

    try:
        return getattr(module, metadata.function_name)
    except AttributeError:
        raise AttributeError(
            f"Function '{metadata.function_name}' not found in module '{full_module_name}'"
        )


def get_eval_metadata(path_like: str) -> BenchmarkMetadata | None:
    """
    Best-effort extraction of __metadata__ for path-based evals.
    Returns None for registry-based benchmarks or when no metadata is present.
    """
    p = Path(path_like).expanduser()
    if not p.exists():
        return None

    try:
        module = _import_module_from_path(p)
        meta = getattr(module, "__metadata__", None)
        return meta if isinstance(meta, BenchmarkMetadata) else None
    except Exception:
        return None


# Eval groups - family benchmark collections that can be expanded
EVAL_GROUPS = {
    "bigbench": EvalGroup(
        name="BIG-Bench",
        description="Aggregate of 121 BIG-Bench tasks for comprehensive language model evaluation",
        benchmarks=[
            "bigbench_anachronisms",
            "bigbench_analogical_similarity",
            "bigbench_analytic_entailment",
            "bigbench_arithmetic",
            "bigbench_authorship_verification",
            "bigbench_bbq_lite_json",
            "bigbench_causal_judgment",
            "bigbench_cause_and_effect",
            "bigbench_checkmate_in_one",
            "bigbench_cifar10_classification",
            "bigbench_code_line_description",
            "bigbench_color",
            "bigbench_common_morpheme",
            "bigbench_conceptual_combinations",
            "bigbench_contextual_parametric_knowledge_conflicts",
            "bigbench_crash_blossom",
            "bigbench_crass_ai",
            "bigbench_cryobiology_spanish",
            "bigbench_cs_algorithms",
            "bigbench_dark_humor_detection",
            "bigbench_date_understanding",
            "bigbench_disambiguation_qa",
            "bigbench_discourse_marker_prediction",
            "bigbench_dyck_languages",
            "bigbench_elementary_math_qa",
            "bigbench_emoji_movie",
            "bigbench_emojis_emotion_prediction",
            "bigbench_empirical_judgments",
            "bigbench_english_proverbs",
            "bigbench_english_russian_proverbs",
            "bigbench_entailed_polarity",
            "bigbench_entailed_polarity_hindi",
            "bigbench_epistemic_reasoning",
            "bigbench_evaluating_information_essentiality",
            "bigbench_fact_checker",
            "bigbench_fantasy_reasoning",
            "bigbench_figure_of_speech_detection",
            "bigbench_formal_fallacies_syllogisms_negation",
            "bigbench_general_knowledge",
            "bigbench_geometric_shapes",
            "bigbench_goal_step_wikihow",
            "bigbench_gre_reading_comprehension",
            "bigbench_hhh_alignment",
            "bigbench_hindu_knowledge",
            "bigbench_hinglish_toxicity",
            "bigbench_human_organs_senses",
            "bigbench_hyperbaton",
            "bigbench_identify_math_theorems",
            "bigbench_identify_odd_metaphor",
            "bigbench_implicatures",
            "bigbench_implicit_relations",
            "bigbench_indic_cause_and_effect",
            "bigbench_intent_recognition",
            "bigbench_international_phonetic_alphabet_nli",
            "bigbench_intersect_geometry",
            "bigbench_irony_identification",
            "bigbench_kanji_ascii",
            "bigbench_kannada",
            "bigbench_key_value_maps",
            "bigbench_known_unknowns",
            "bigbench_language_identification",
            "bigbench_logic_grid_puzzle",
            "bigbench_logical_args",
            "bigbench_logical_deduction",
            "bigbench_logical_fallacy_detection",
            "bigbench_logical_sequence",
            "bigbench_mathematical_induction",
            "bigbench_medical_questions_russian",
            "bigbench_metaphor_boolean",
            "bigbench_metaphor_understanding",
            "bigbench_minute_mysteries_qa",
            "bigbench_misconceptions",
            "bigbench_misconceptions_russian",
            "bigbench_mnist_ascii",
            "bigbench_moral_permissibility",
            "bigbench_movie_dialog_same_or_different",
            "bigbench_movie_recommendation",
            "bigbench_navigate",
            "bigbench_nonsense_words_grammar",
            "bigbench_novel_concepts",
            "bigbench_odd_one_out",
            "bigbench_parsinlu_qa",
            "bigbench_penguins_in_a_table",
            "bigbench_periodic_elements",
            "bigbench_persian_idioms",
            "bigbench_phrase_relatedness",
            "bigbench_physical_intuition",
            "bigbench_physics",
            "bigbench_play_dialog_same_or_different",
            "bigbench_presuppositions_as_nli",
            "bigbench_question_selection",
            "bigbench_real_or_fake_text",
            "bigbench_reasoning_about_colored_objects",
            "bigbench_rhyming",
            "bigbench_riddle_sense",
            "bigbench_ruin_names",
            "bigbench_salient_translation_error_detection",
            "bigbench_sentence_ambiguity",
            "bigbench_similarities_abstraction",
            "bigbench_simple_ethical_questions",
            "bigbench_snarks",
            "bigbench_social_iqa",
            "bigbench_social_support",
            "bigbench_sports_understanding",
            "bigbench_strange_stories",
            "bigbench_strategyqa",
            "bigbench_suicide_risk",
            "bigbench_swahili_english_proverbs",
            "bigbench_swedish_to_german_proverbs",
            "bigbench_symbol_interpretation",
            "bigbench_temporal_sequences",
            "bigbench_timedial",
            "bigbench_tracking_shuffled_objects",
            "bigbench_understanding_fables",
            "bigbench_undo_permutation",
            "bigbench_unit_conversion",
            "bigbench_unit_interpretation",
            "bigbench_vitaminc_fact_verification",
            "bigbench_what_is_the_tao",
            "bigbench_which_wiki_edit",
            "bigbench_winowhy",
        ],
    ),
    "bbh": EvalGroup(
        name="BIG-Bench Hard",
        description="Aggregate of 18 challenging BIG-Bench tasks that require multi-step reasoning",
        benchmarks=[
            "bbh_causal_judgment",
            "bbh_date_understanding",
            "bbh_disambiguation_qa",
            "bbh_geometric_shapes",
            "bbh_logical_deduction_five_objects",
            "bbh_logical_deduction_seven_objects",
            "bbh_logical_deduction_three_objects",
            "bbh_movie_recommendation",
            "bbh_navigate",
            "bbh_reasoning_about_colored_objects",
            "bbh_ruin_names",
            "bbh_salient_translation_error_detection",
            "bbh_snarks",
            "bbh_sports_understanding",
            "bbh_temporal_sequences",
            "bbh_tracking_shuffled_objects_five_objects",
            "bbh_tracking_shuffled_objects_seven_objects",
            "bbh_tracking_shuffled_objects_three_objects",
        ],
    ),
    "agieval": EvalGroup(
        name="AGIEval",
        description="Aggregate of 17 AGIEval exam tasks testing human-level reasoning across various domains",
        benchmarks=[
            "agieval_aqua_rat",
            "agieval_gaokao_biology",
            "agieval_gaokao_chemistry",
            "agieval_gaokao_chinese",
            "agieval_gaokao_english",
            "agieval_gaokao_geography",
            "agieval_gaokao_history",
            "agieval_gaokao_mathqa",
            "agieval_gaokao_physics",
            "agieval_logiqa_en",
            "agieval_logiqa_zh",
            "agieval_lsat_ar",
            "agieval_lsat_lr",
            "agieval_lsat_rc",
            "agieval_sat_en",
            "agieval_sat_en_without_passage",
            "agieval_sat_math",
        ],
    ),
    "ethics": EvalGroup(
        name="ETHICS",
        description="Aggregate of 5 ETHICS moral reasoning tasks",
        benchmarks=[
            "ethics_commonsense",
            "ethics_deontology",
            "ethics_justice",
            "ethics_utilitarianism",
            "ethics_virtue",
        ],
    ),
    "blimp": EvalGroup(
        name="BLiMP",
        description="Aggregate of 67 BLiMP linguistic tasks",
        benchmarks=[
            "blimp_adjunct_island",
            "blimp_anaphor_gender_agreement",
            "blimp_anaphor_number_agreement",
            "blimp_animate_subject_passive",
            "blimp_animate_subject_trans",
            "blimp_causative",
            "blimp_complex_NP_island",
            "blimp_coordinate_structure_constraint_complex_left_branch",
            "blimp_coordinate_structure_constraint_object_extraction",
            "blimp_determiner_noun_agreement_1",
            "blimp_determiner_noun_agreement_2",
            "blimp_determiner_noun_agreement_irregular_1",
            "blimp_determiner_noun_agreement_irregular_2",
            "blimp_determiner_noun_agreement_with_adj_2",
            "blimp_determiner_noun_agreement_with_adj_irregular_1",
            "blimp_determiner_noun_agreement_with_adj_irregular_2",
            "blimp_determiner_noun_agreement_with_adjective_1",
            "blimp_distractor_agreement_relational_noun",
            "blimp_distractor_agreement_relative_clause",
            "blimp_drop_argument",
            "blimp_ellipsis_n_bar_1",
            "blimp_ellipsis_n_bar_2",
            "blimp_existential_there_object_raising",
            "blimp_existential_there_quantifiers_1",
            "blimp_existential_there_quantifiers_2",
            "blimp_existential_there_subject_raising",
            "blimp_expletive_it_object_raising",
            "blimp_inchoative",
            "blimp_intransitive",
            "blimp_irregular_past_participle_adjectives",
            "blimp_irregular_past_participle_verbs",
            "blimp_irregular_plural_subject_verb_agreement_1",
            "blimp_irregular_plural_subject_verb_agreement_2",
            "blimp_left_branch_island_echo_question",
            "blimp_left_branch_island_simple_question",
            "blimp_matrix_question_npi_licensor_present",
            "blimp_npi_present_1",
            "blimp_npi_present_2",
            "blimp_only_npi_licensor_present",
            "blimp_only_npi_scope",
            "blimp_passive_1",
            "blimp_passive_2",
            "blimp_principle_A_c_command",
            "blimp_principle_A_case_1",
            "blimp_principle_A_case_2",
            "blimp_principle_A_domain_1",
            "blimp_principle_A_domain_2",
            "blimp_principle_A_domain_3",
            "blimp_principle_A_reconstruction",
            "blimp_regular_plural_subject_verb_agreement_1",
            "blimp_regular_plural_subject_verb_agreement_2",
            "blimp_sentential_negation_npi_licensor_present",
            "blimp_sentential_negation_npi_scope",
            "blimp_sentential_subject_island",
            "blimp_superlative_quantifiers_1",
            "blimp_superlative_quantifiers_2",
            "blimp_tough_vs_raising_1",
            "blimp_tough_vs_raising_2",
            "blimp_transitive",
            "blimp_wh_island",
            "blimp_wh_questions_object_gap",
            "blimp_wh_questions_subject_gap",
            "blimp_wh_questions_subject_gap_long_distance",
            "blimp_wh_vs_that_no_gap",
            "blimp_wh_vs_that_no_gap_long_distance",
            "blimp_wh_vs_that_with_gap",
            "blimp_wh_vs_that_with_gap_long_distance",
        ],
    ),
    "glue": EvalGroup(
        name="GLUE",
        description="Aggregate of 10 GLUE NLU tasks",
        benchmarks=[
            "glue_cola",
            "glue_mnli",
            "glue_mnli_mismatched",
            "glue_mrpc",
            "glue_qnli",
            "glue_qqp",
            "glue_rte",
            "glue_sst2",
            "glue_stsb",
            "glue_wnli",
        ],
    ),
    "global_mmlu": EvalGroup(
        name="Global-MMLU",
        description="Aggregate of 42 Global-MMLU language tasks",
        benchmarks=[
            "global_mmlu_amharic",
            "global_mmlu_arabic",
            "global_mmlu_bengali",
            "global_mmlu_chichewa",
            "global_mmlu_chinese",
            "global_mmlu_czech",
            "global_mmlu_dutch",
            "global_mmlu_english",
            "global_mmlu_filipino",
            "global_mmlu_french",
            "global_mmlu_german",
            "global_mmlu_greek",
            "global_mmlu_hausa",
            "global_mmlu_hebrew",
            "global_mmlu_hindi",
            "global_mmlu_igbo",
            "global_mmlu_indonesian",
            "global_mmlu_italian",
            "global_mmlu_japanese",
            "global_mmlu_korean",
            "global_mmlu_kyrgyz",
            "global_mmlu_lithuanian",
            "global_mmlu_malagasy",
            "global_mmlu_malay",
            "global_mmlu_nepali",
            "global_mmlu_persian",
            "global_mmlu_polish",
            "global_mmlu_portuguese",
            "global_mmlu_romanian",
            "global_mmlu_russian",
            "global_mmlu_serbian",
            "global_mmlu_shona",
            "global_mmlu_sinhala",
            "global_mmlu_somali",
            "global_mmlu_spanish",
            "global_mmlu_swahili",
            "global_mmlu_swedish",
            "global_mmlu_telugu",
            "global_mmlu_turkish",
            "global_mmlu_ukrainian",
            "global_mmlu_vietnamese",
            "global_mmlu_yoruba",
        ],
    ),
    "xcopa": EvalGroup(
        name="XCOPA",
        description="Aggregate of 11 XCOPA language tasks",
        benchmarks=[
            "xcopa_et",
            "xcopa_ht",
            "xcopa_id",
            "xcopa_it",
            "xcopa_qu",
            "xcopa_sw",
            "xcopa_ta",
            "xcopa_th",
            "xcopa_tr",
            "xcopa_vi",
            "xcopa_zh",
        ],
    ),
    "xstorycloze": EvalGroup(
        name="XStoryCloze",
        description="Aggregate of 11 XStoryCloze tasks",
        benchmarks=[
            "xstorycloze_ar",
            "xstorycloze_en",
            "xstorycloze_es",
            "xstorycloze_eu",
            "xstorycloze_hi",
            "xstorycloze_id",
            "xstorycloze_my",
            "xstorycloze_ru",
            "xstorycloze_sw",
            "xstorycloze_te",
            "xstorycloze_zh",
        ],
    ),
    "xwinograd": EvalGroup(
        name="XWinograd",
        description="Aggregate of 6 XWinograd tasks",
        benchmarks=[
            "xwinograd_en",
            "xwinograd_fr",
            "xwinograd_jp",
            "xwinograd_pt",
            "xwinograd_ru",
            "xwinograd_zh",
        ],
    ),
    "mgsm": EvalGroup(
        name="MGSM",
        description="Multilingual Grade School Math across 11 languages",
        benchmarks=[
            "mgsm_en",
            "mgsm_de",
            "mgsm_es",
            "mgsm_fr",
            "mgsm_sw",
            "mgsm_bn",
            "mgsm_ja",
            "mgsm_ru",
            "mgsm_te",
            "mgsm_th",
            "mgsm_zh",
            "mgsm_latin",
            "mgsm_non_latin",
        ],
    ),
    "mmmu": EvalGroup(
        name="MMMU",
        description="Aggregate of 29+ MMMU subject tasks",
        benchmarks=[
            "mmmu_accounting",
            "mmmu_agriculture",
            "mmmu_architecture_and_engineering",
            "mmmu_art",
            "mmmu_art_theory",
            "mmmu_basic_medical_science",
            "mmmu_biology",
            "mmmu_chemistry",
            "mmmu_clinical_medicine",
            "mmmu_design",
            "mmmu_diagnostics_and_laboratory_medicine",
            "mmmu_electronics",
            "mmmu_energy_and_power",
            "mmmu_finance",
            "mmmu_geography",
            "mmmu_history",
            "mmmu_literature",
            "mmmu_manage",
            "mmmu_marketing",
            "mmmu_materials",
            "mmmu_math",
            "mmmu_mcq",
            "mmmu_mechanical_engineering",
            "mmmu_music",
            "mmmu_open",
            "mmmu_pharmacy",
            "mmmu_physics",
            "mmmu_public_health",
            "mmmu_sociology",
        ],
    ),
    "arabic_exams": EvalGroup(
        name="Arabic Exams",
        description="Aggregate of 40+ Arabic exam tasks",
        benchmarks=[
            "arabic_exams_accounting_university",
            "arabic_exams_arabic_language_general",
            "arabic_exams_arabic_language_grammar",
            "arabic_exams_arabic_language_high_school",
            "arabic_exams_arabic_language_middle_school",
            "arabic_exams_arabic_language_primary_school",
            "arabic_exams_biology_high_school",
            "arabic_exams_civics_high_school",
            "arabic_exams_civics_middle_school",
            "arabic_exams_computer_science_high_school",
            "arabic_exams_computer_science_middle_school",
            "arabic_exams_computer_science_primary_school",
            "arabic_exams_computer_science_university",
            "arabic_exams_driving_test",
            "arabic_exams_economics_high_school",
            "arabic_exams_economics_middle_school",
            "arabic_exams_economics_university",
            "arabic_exams_general_knowledge",
            "arabic_exams_general_knowledge_middle_school",
            "arabic_exams_general_knowledge_primary_school",
            "arabic_exams_geography_high_school",
            "arabic_exams_geography_middle_school",
            "arabic_exams_geography_primary_school",
            "arabic_exams_history_high_school",
            "arabic_exams_history_middle_school",
            "arabic_exams_history_primary_school",
            "arabic_exams_islamic_studies_general",
            "arabic_exams_islamic_studies_high_school",
            "arabic_exams_islamic_studies_middle_school",
            "arabic_exams_islamic_studies_primary_school",
            "arabic_exams_law_professional",
            "arabic_exams_management_university",
            "arabic_exams_math_high_school",
            "arabic_exams_math_primary_school",
            "arabic_exams_natural_science_middle_school",
            "arabic_exams_natural_science_primary_school",
            "arabic_exams_philosophy_high_school",
            "arabic_exams_physics_high_school",
            "arabic_exams_political_science_university",
            "arabic_exams_social_science_middle_school",
            "arabic_exams_social_science_primary_school",
        ],
    ),
    "exercism": EvalGroup(
        name="Exercism",
        description="Aggregate of 5 Exercism coding tasks",
        benchmarks=[
            "exercism_go",
            "exercism_java",
            "exercism_javascript",
            "exercism_python",
            "exercism_rust",
        ],
    ),
    "anli": EvalGroup(
        name="ANLI",
        description="Aggregate of 3 ANLI rounds",
        benchmarks=[
            "anli_r1",
            "anli_r2",
            "anli_r3",
        ],
    ),
    "healthbench": EvalGroup(
        name="HealthBench",
        description="Aggregate of HealthBench tasks",
        benchmarks=[
            "healthbench_consensus",
            "healthbench_hard",
        ],
    ),
    "openai_mrcr": EvalGroup(
        name="OpenAI MRCR",
        description="Aggregate of 3 MRCR needle tasks",
        benchmarks=[
            "openai_mrcr_2n",
            "openai_mrcr_4n",
            "openai_mrcr_8n",
        ],
    ),
    "matharena": EvalGroup(
        name="MathArena",
        description="Aggregate of 11 math competition tasks",
        benchmarks=[
            "aime_2023_I",
            "aime_2023_II",
            "aime_2024",
            "aime_2024_I",
            "aime_2024_II",
            "aime_2025",
            "aime_2025_II",
            "brumo_2025",
            "hmmt_feb_2023",
            "hmmt_feb_2024",
            "hmmt_feb_2025",
        ],
    ),
    "qa4mre": EvalGroup(
        name="QA4MRE",
        description="Aggregate of 3 QA4MRE years",
        benchmarks=[
            "qa4mre_2011",
            "qa4mre_2012",
            "qa4mre_2013",
        ],
    ),
    "otis_mock_aime": EvalGroup(
        name="OTIS Mock AIME",
        description="Aggregate of 2 Mock AIME years",
        benchmarks=[
            "otis_mock_aime_2024",
            "otis_mock_aime_2025",
        ],
    ),
    "bbl": EvalGroup(
        name="BIG-Bench Lite",
        description="BIG-Bench Lite - 18 selected BBH tasks",
        benchmarks=[
            "bbh_causal_judgment",
            "bbh_date_understanding",
            "bbh_disambiguation_qa",
            "bbh_geometric_shapes",
            "bbh_logical_deduction_five_objects",
            "bbh_logical_deduction_seven_objects",
            "bbh_logical_deduction_three_objects",
            "bbh_movie_recommendation",
            "bbh_navigate",
            "bbh_reasoning_about_colored_objects",
            "bbh_ruin_names",
            "bbh_salient_translation_error_detection",
            "bbh_snarks",
            "bbh_sports_understanding",
            "bbh_temporal_sequences",
            "bbh_tracking_shuffled_objects_five_objects",
            "bbh_tracking_shuffled_objects_seven_objects",
            "bbh_tracking_shuffled_objects_three_objects",
        ],
    ),
    "math": EvalGroup(
        name="MATH Dataset",
        description="MATH dataset variants for mathematical problem solving",
        benchmarks=[
            "math",
            "math_500",
        ],
    ),
    "mmmlu": EvalGroup(
        name="MMMLU",
        description="MMLU in 14 languages",
        benchmarks=[
            "mmmlu_ar_xy",
            "mmmlu_bn_bd",
            "mmmlu_de_de",
            "mmmlu_es_la",
            "mmmlu_fr_fr",
            "mmmlu_hi_in",
            "mmmlu_id_id",
            "mmmlu_it_it",
            "mmmlu_ja_jp",
            "mmmlu_ko_kr",
            "mmmlu_pt_br",
            "mmmlu_zh_cn",
            "mmmlu_sw_ke",
            "mmmlu_yo_ng",
        ],
    ),
    "superglue": EvalGroup(
        name="SuperGLUE",
        description="SuperGLUE benchmark with 6 NLU tasks",
        benchmarks=[
            "boolq",
            "cb",
            "copa",
            "multirc",
            "rte",
            "wic",
            "wsc",
        ],
    ),
}
