#!/bin/env python3
"""
Created on 6 Jul 2025

@author: ph1jb
"""

from configargparse import Namespace, YAMLConfigFileParser  # type: ignore
from pathlib import Path
import configargparse
import dbcache
import dotenv
import logging
import yaml

LOGFORMAT = "%(asctime)s %(levelname)s [%(module)s:%(lineno)d] %(funcName)s | %(message)s"

logger = logging.getLogger(__name__)


class Config:
    def __init__(self, secrets_file: Path):
        """Initialise the Main class, parse configuration and set up database connection parameters."""
        self.config: Namespace = self._parse_config(secrets_file)
        self._setup_logging(level=self.config.loglevel, format=LOGFORMAT)

    @staticmethod
    def _parse_config(secrets_file: Path) -> Namespace:
        """Get parser with specified arguments (CLI, config, env var, default)"""
        p = configargparse.ArgParser(
            config_file_parser_class=YAMLConfigFileParser,
            default_config_files=[secrets_file],
        )
        p.add_argument(
            "--loglevel",
            choices=["INFO", "WARNING", "DEBUG", "ERROR", "CRITICAL"],
            default="INFO",
            env_var="LOGLEVEL",
            help="Log level (INFO, WARNING, DEBUG etc)",
        )
        p.add("--mysql_database", env_var="MYSQL_DATABASE", help="Database name")
        p.add("--mysql_host", env_var="MYSQL_HOST", help="Database host")
        p.add("--mysql_user", env_var="MYSQL_USER", help="Database user")
        p.add("--mysql_password", help="Database password.")
        p.add(
            "--mysql_options",
            env_var="MYSQL_OPTIONS",
            required=True,
            type=yaml.safe_load,
            help="MySQL options",
        )
        p.add(
            "--table_source",
            env_var="TABLE_SOURCE",
            required=True,
            help="Source database table name",
        )
        logger.debug("Parsing configuration arguments.")
        return p.parse_args()

    @staticmethod
    def _setup_logging(**kwargs) -> None:
        """Set up logging."""
        logging.basicConfig(**kwargs)


class Demo:

    def __init__(self, config: Config):
        """Initialise the Main class, parse configuration and set up database connection parameters."""
        self.config = config.config
        self.main = dbcache.Main(self.config)
        self.cache_manager = self.main.cache_manager

    def run(self):
        """Main entrypoint for cache handling."""
        table_name = self.config.table_source
        logger.info(
            "Starting data retrieval for table '%(table_name)s'.", {"table_name": table_name}
        )
        try:
            df = self.cache_manager.get_dataframe(table_name)
            logger.info(
                "Retrieved DataFrame with %(rows)d rows and %(cols)d columns.",
                {"rows": len(df), "cols": len(df.columns)},
            )
            print(df.head())
        except Exception as e:
            logger.exception(
                "Failed to retrieve DataFrame for '%(table_name)s': %(error)s",
                {"table_name": table_name, "error": str(e)},
            )


if __name__ == "__main__":
    dotenv.load_dotenv()
    SECRETS_FILE = Path(__file__).resolve().parent.parent.parent / "secrets" / "dbcache_secrets.yml"
    print(SECRETS_FILE)
    CONFIG = Config(SECRETS_FILE)
    DEMO = Demo(CONFIG)
    DEMO.run()
