"""
Created on 29 Oct 2025

@author: ph1jb
"""

from dbcache2 import CacheManager, Main
from sqlalchemy import create_engine, text
from types import SimpleNamespace
import pandas as pd
import pytest


# ----------------------------------------------------------------------
# Fixtures
# ----------------------------------------------------------------------


@pytest.fixture(scope="module")
def engine():
    """Provide an in-memory SQLite engine for testing."""
    eng = create_engine("sqlite+pysqlite:///:memory:", future=True)
    with eng.begin() as conn:
        conn.execute(text("CREATE TABLE cache (name TEXT PRIMARY KEY, data BLOB)"))
        conn.execute(text("CREATE TABLE test_table (id INTEGER, name TEXT)"))
        conn.execute(text("INSERT INTO test_table VALUES (1, 'Alice'), (2, 'Bob')"))
    yield eng
    eng.dispose()


@pytest.fixture
def cache_manager(engine):
    """Fixture for CacheManager instance."""
    return CacheManager(engine)


@pytest.fixture
def main_config():
    """Mock configuration object for Main."""
    config = SimpleNamespace(
        mysql_database=":memory:",
        mysql_host="localhost",
        mysql_user="user",
        mysql_password="pass",
        mysql_options={
            "database": ":memory:",
            "host": "localhost",
            "user": "user",
            "password": "pass",
        },
        table_source="test_table",
    )
    return config


@pytest.fixture
def main_instance(main_config):
    """Fixture for Main instance (with SQLite engine)."""
    # Override engine creation to use SQLite instead of MySQL
    main = Main(main_config)
    main.engine = create_engine("sqlite+pysqlite:///:memory:", future=True)
    with main.engine.begin() as conn:
        conn.execute(text("CREATE TABLE cache (name TEXT PRIMARY KEY, data BLOB)"))
        conn.execute(text("CREATE TABLE test_table (id INTEGER, name TEXT)"))
        conn.execute(text("INSERT INTO test_table VALUES (1, 'Alice'), (2, 'Bob')"))
    main.cache_manager = CacheManager(main.engine)
    return main


# ----------------------------------------------------------------------
# Tests for CacheManager
# ----------------------------------------------------------------------


class TestCacheManager:
    def test_serialize_and_deserialize(self, cache_manager):
        """Test DataFrame serialization and deserialization."""
        df = pd.DataFrame({"a": [1, 2, 3], "b": ["x", "y", "z"]})
        data = cache_manager.serialize_df_to_bytes(df)
        assert isinstance(data, bytes)
        df2 = cache_manager.deserialize_df_from_bytes(data)
        pd.testing.assert_frame_equal(df, df2)

    def test_get_dataframe_cache_miss_and_hit(self, cache_manager, engine):
        """Test that cache miss loads from table, then cache hit loads from cache."""
        # Cache miss: load from source table
        df1 = cache_manager.get_dataframe("test_table")
        assert not df1.empty
        assert set(df1.columns) == {"id", "name"}

        # Cache hit: should now load from cache
        df2 = cache_manager.get_dataframe("test_table")
        pd.testing.assert_frame_equal(df1, df2)


# ----------------------------------------------------------------------
# Tests for Main
# ----------------------------------------------------------------------


class TestMain:
    def test_override_mysql_options(self, main_config):
        """Test override_mysql_options updates config correctly."""
        new_opts = {
            "database": "otherdb",
            "host": "otherhost",
            "user": "otheruser",
            "password": "otherpass",
        }
        main_config.mysql_database = "db1"
        main_config.mysql_host = "h1"
        main_config.mysql_user = "u1"
        main_config.mysql_password = "p1"
        main_config.mysql_options = {}
        Main.override_mysql_options(main_config)
        assert main_config.mysql_options == {
            "database": "db1",
            "host": "h1",
            "password": "p1",
            "user": "u1",
        }

    def test_run_fetches_dataframe(self, main_instance, capsys):
        """Test that Main.run prints DataFrame head."""
        main_instance.run()
        captured = capsys.readouterr()
        assert "Alice" in captured.out
        assert "Bob" in captured.out
