from __future__ import annotations

from typing import Annotated

from sayer import Argument, Option, command, error, success

from lilya.cli.exceptions import DirectiveError
from lilya.cli.templates import TemplateDirective


@command(name="createdeployment")
def create_deployment(
    name: Annotated[str, Argument(help="The name of the current project.")],
    verbosity: Annotated[
        int, Option(1, "-v", help="Displays the files generated", show_default=True)
    ],
    deployment_folder_name: Annotated[
        str,
        Option(
            default="deployment",
            help="The name of the folder for the deployment files.",
            show_default=True,
        ),
    ],
) -> None:
    """
    Generates the scaffold for the deployment of a Lilya application.

    The scaffold contains the configurations for docker, nginx, supervisor and gunicorn.

    The configurations should be adapted accordingly.
    The parameter <NAME> corresponds to the name of the
    project where the deployment should be placed.

    How to run: `lilya createdeployment <NAME>`

    Example: `lilya createdeployment myproject`
    """
    options = {
        "verbosity": verbosity,
        "deployment_folder_name": deployment_folder_name,
        "is_simple": False,
    }
    directive = TemplateDirective()

    try:
        directive.handle("deployment", name=name, **options)
        success(f"Deployment for {name} generated successfully!")
    except DirectiveError as e:
        error(str(e))
