import mapel.allocations.core.logs as logs
logger = logs.get_logger(__name__)
import itertools
import numpy as np

import mapel.core.matchings as matchings

def ideal_distance(left_task, right_task, *args, **kwargs):
  logger.debug("Computing single ideal distance")
  _validate(left_task, right_task)

  optimal_matchings = []
  for agents_mapping in itertools.permutations(range(left_task.agents_count)):
      costs = _get_resource_mapping_costs(left_task, right_task, agents_mapping)
      optimal_matchings.append(matchings.solve_matching_vectors(costs))

  return min(optimal_matchings, key = lambda x: x[0])

def _get_resource_mapping_costs(left_task, right_task, agents_mapping):
    """ Return: Cost table """
    cost_table = np.zeros([left_task.resources_count, left_task.resources_count])
    for left_res in range(left_task.resources_count):
      for right_res in range(right_task.resources_count):
        ell_one_dist = 0
        for left_agent in range(left_task.agents_count):
          right_agent = agents_mapping[left_agent]
          ell_one_diff = abs(left_task[left_agent][left_res] - \
          right_task[right_agent][right_res])
          ell_one_dist += ell_one_diff
        cost_table[left_res][right_res] = ell_one_dist
    return cost_table

def _validate(left_task, right_task):
  if left_task.agents_count != right_task.agents_count:
    raise ValueError("Cannot compute distance between two tasks with different "
    "counts of agents")
  if left_task.resources_count != right_task.resources_count:
    raise ValueError("Cannot compute distance between two tasks with different "
    "counts of resources")

def ideal_distance_ilp(left_task, right_task, *args, **kwargs):
  logger.debug("Computing single ideal distance via ILP")
  import gurobipy as gp
  import itertools as it

  def ell_one_of_matchings(k, l, p, q):
    left_val = left_task[k][p]
    right_val = right_task[l][q]
    return abs(left_val - right_val)

  def get_matchings_quadruples():
    return it.product(range(ags_count), range(ags_count), range(res_count),
              range(res_count))

  res_count = left_task.resources_count
  ags_count = left_task.agents_count

  model = gp.Model("ILP distance")

  rmatch = model.addMVar((res_count, res_count), vtype=gp.GRB.BINARY,
  name="rmatch")
  amatch = model.addMVar((ags_count, ags_count), vtype=gp.GRB.BINARY, name="amatch")

  #enforce matching of resources and candidates
  model.addConstrs((gp.quicksum(rmatch[x,:]) == 1 for x in range(res_count)))
  model.addConstrs((gp.quicksum(rmatch[:,x]) == 1 for x in range(res_count)))
  model.addConstrs((gp.quicksum(amatch[x,:]) == 1 for x in range(ags_count)))
  model.addConstrs((gp.quicksum(amatch[:,x]) == 1 for x in range(ags_count)))


  summands = model.addMVar((ags_count, ags_count, res_count, res_count),
                           vtype=gp.GRB.BINARY, name = "summands")

  #enforcing that summand is one only if the respective matchings are one
  for (k, l, p, q) in get_matchings_quadruples():
    model.addGenConstrAnd(summands[k, l, p, q], [amatch[k,l], rmatch[p, q]])

  #objective func
  model.setObjective(gp.quicksum(
    (summands[k, l, p, q] * ell_one_of_matchings(k, l, p, q)
    for (k, l, p, q) in get_matchings_quadruples())),
    gp.GRB.MINIMIZE)

  model.optimize()

#  for var in model.getVars():
#    if (var.X >= 1):
#      print(var.VarName)
#      print(var.X)

  obj = model.getObjective()
  return obj.getValue(), None




