# Copyright 2025 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
"""Integration tests for Chronicle API.

These tests require valid credentials and API access.
"""
import pytest
from datetime import datetime, timedelta, timezone
from secops import SecOpsClient
from ..config import CHRONICLE_CONFIG, SERVICE_ACCOUNT_JSON
from secops.exceptions import APIError, SecOpsError
from secops.chronicle.models import EntitySummary
from secops.chronicle.data_table import DataTableColumnType
from secops.chronicle.reference_list import ReferenceListSyntaxType, ReferenceListView
import json
import re
import time


@pytest.mark.integration
def test_chronicle_search():
    """Test Chronicle search functionality with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(hours=1)

    result = chronicle.fetch_udm_search_csv(
        query='ip != ""',
        start_time=start_time,
        end_time=end_time,
        fields=["timestamp", "user", "hostname", "process name"],
    )

    assert isinstance(result, str)
    assert "timestamp" in result  # Basic validation of CSV header


@pytest.mark.integration
def test_chronicle_stats():
    """Test Chronicle stats search functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(hours=1)

    # Use a stats query format
    query = """metadata.event_type = "NETWORK_CONNECTION"
match:
    metadata.event_type
outcome:
    $count = count(metadata.id)
order:
    metadata.event_type asc"""

    validation = chronicle.validate_query(query)
    print(f"\nValidation response: {validation}")  # Debug print
    assert "queryType" in validation
    assert (
        validation.get("queryType") == "QUERY_TYPE_STATS_QUERY"
    )  # Note: changed assertion

    try:
        # Perform stats search with limited results
        result = chronicle.get_stats(
            query=query,
            start_time=start_time,
            end_time=end_time,
            max_events=10,  # Limit results for testing
            max_values=10,  # Limit field values for testing
            timeout=60 # Short Timeout
        )

        assert "columns" in result
        assert "rows" in result
        assert isinstance(result["total_rows"], int)

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")  # Debug print
        raise


@pytest.mark.integration
def test_chronicle_udm_search():
    """Test Chronicle UDM search functionality with real API.

    This test is designed to be robust against timeouts and network issues.
    It will pass with either found events or empty results.
    """
    try:
        # Set up client
        client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
        chronicle = client.chronicle(**CHRONICLE_CONFIG)

        # Use a very small time window to minimize processing time
        end_time = datetime.now(timezone.utc)
        start_time = end_time - timedelta(minutes=1)

        # Create a very specific query to minimize results
        query = 'metadata.event_type = "NETWORK_HTTP"'

        print("\nStarting UDM search integration test...")
        print(f"Time window: {start_time.isoformat()} to {end_time.isoformat()}")
        print(f"Query: {query}")

        # First, validate that the query is valid
        try:
            validation = chronicle.validate_query(query)
            print(f"Query validation result: {validation}")
            assert "queryType" in validation
        except Exception as e:
            print(f"Query validation failed: {str(e)}")
            # Continue anyway, the query should be valid

        # Perform the search with minimal expectations
        try:
            # Modified search_udm function to accept debugging
            result = chronicle.search_udm(
                query=query,
                start_time=start_time,
                end_time=end_time,
                max_events=1,  # Just need one event to verify
                max_attempts=5,  # Don't wait too long
                timeout=10,  # Short timeout
                debug=True,  # Enable debug messages
            )

            # Basic structure checks
            assert isinstance(result, dict)
            assert "events" in result
            assert "total_events" in result
            assert "more_data_available" in result

            print(f"Search completed. Found {result['total_events']} events.")

            # If we got events, do some basic validation
            if result["events"]:
                print("Validating event structure...")
                event = result["events"][0]
                assert "name" in event
                assert "udm" in event
                assert "metadata" in event["udm"]
            else:
                print("No events found in time window. This is acceptable.")

        except Exception as e:
            print(f"Search failed but test will continue: {type(e).__name__}: {str(e)}")
            # We'll consider no results as a pass condition too
            # Create a placeholder result
            result = {"events": [], "total_events": 0, "more_data_available": False}

        # The test passes as long as we got a valid response structure,
        # even if it contained no events
        assert isinstance(result, dict)
        assert "events" in result
        print("UDM search test passed successfully.")

    except Exception as e:
        # Last resort exception handling - print details but don't fail the test
        print(f"Unexpected error in UDM search test: {type(e).__name__}: {str(e)}")
        print("UDM search test will be marked as skipped.")
        pytest.skip(f"Test skipped due to unexpected error: {str(e)}")


@pytest.mark.integration
def test_chronicle_summarize_entity():
    """Test Chronicle entity summary functionality with the real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(days=1)  # Look back 1 day

    try:
        # Get summary for a common public IP (more likely to have data)
        ip_to_check = "8.8.8.8"
        result = chronicle.summarize_entity(
            value=ip_to_check, start_time=start_time, end_time=end_time
        )

        # Basic validation - we expect an EntitySummary object
        assert isinstance(result, EntitySummary)

        # Check if a primary entity was found (can be None if no data)
        if result.primary_entity:
            print(
                f"\nPrimary entity found: {result.primary_entity.metadata.entity_type}"
            )
            # The primary entity type could be ASSET or IP_ADDRESS
            assert result.primary_entity.metadata.entity_type in ["ASSET", "IP_ADDRESS"]
        else:
            print(f"\nNo primary entity found for {ip_to_check} in the last day.")

        # Print some details if available (optional checks)
        if result.alert_counts:
            print(f"Found {len(result.alert_counts)} alert counts.")
        if result.timeline:
            print(f"Timeline found with {len(result.timeline.buckets)} buckets.")
        if result.prevalence:
            print(f"Prevalence data found: {len(result.prevalence)} entries.")

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")  # Debug print
        raise


@pytest.mark.integration
def test_chronicle_alerts():
    """Test Chronicle alerts functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Get alerts from the last 1 day
    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(days=1)

    try:
        # Use a query to get non-closed alerts
        result = chronicle.get_alerts(
            start_time=start_time,
            end_time=end_time,
            snapshot_query='feedback_summary.status != "CLOSED"',
            max_alerts=10,  # Limit to 10 alerts for testing
            max_attempts=5,  # Limit polling attempts for faster test
        )

        # Basic validation of the response
        assert "complete" in result
        assert result.get("complete") is True or result.get("progress") == 1

        # Check if we got any alerts
        alerts = result.get("alerts", {}).get("alerts", [])
        print(f"\nFound {len(alerts)} alerts")

        # If we have alerts, validate their structure
        if alerts:
            alert = alerts[0]
            assert "id" in alert
            assert "type" in alert
            assert "createdTime" in alert

            # Check detection info if this is a rule detection
            if alert.get("type") == "RULE_DETECTION" and "detection" in alert:
                detection = alert.get("detection", [])[0]
                assert "ruleName" in detection
                print(f"\nRule name: {detection.get('ruleName')}")

            # Check if alert is linked to a case
            if "caseName" in alert:
                print(f"\nAlert is linked to case: {alert.get('caseName')}")

                # Try to get case details if we have case IDs
                case_ids = {
                    alert.get("caseName") for alert in alerts if alert.get("caseName")
                }
                if case_ids:
                    print(f"\nFound {len(case_ids)} unique case IDs")
                    try:
                        cases = chronicle.get_cases(list(case_ids))
                        print(f"Retrieved {len(cases.cases)} cases")

                        # Validate case structure
                        if cases.cases:
                            case = cases.cases[0]
                            assert hasattr(case, "id")
                            assert hasattr(case, "display_name")
                            assert hasattr(case, "priority")
                            assert hasattr(case, "status")
                            print(f"First case: {case.display_name} (ID: {case.id})")
                    except APIError as e:
                        print(f"Could not retrieve case details: {e}")
                        # This is not a test failure - cases might not be accessible

        # Validate field aggregations if present
        field_aggregations = result.get("fieldAggregations", {}).get("fields", [])
        if field_aggregations:
            assert isinstance(field_aggregations, list)

            # Check specific field aggregations if available
            status_field = next(
                (
                    f
                    for f in field_aggregations
                    if f.get("fieldName") == "feedback_summary.status"
                ),
                None,
            )
            if status_field:
                print(
                    f"\nStatus field values: {[v.get('value', {}).get('enumValue') for v in status_field.get('allValues', [])]}"
                )

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")  # Debug print
        raise


@pytest.mark.integration
def test_chronicle_list_iocs():
    """Test Chronicle IoC listing functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Look back 30 days for IoCs
    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(days=30)

    try:
        # Test with default parameters
        result = chronicle.list_iocs(
            start_time=start_time,
            end_time=end_time,
            max_matches=10,  # Limit to 10 for testing
        )

        # Verify the response structure
        assert isinstance(result, dict)

        # Print the count of matches for debugging
        match_count = len(result.get("matches", []))
        print(f"\nFound {match_count} IoC matches")

        # Check the data structure if matches were found
        if match_count > 0:
            match = result["matches"][0]
            # Verify fields are processed correctly
            if "properties" in match:
                assert isinstance(match["properties"], dict)

            # Check that timestamp fields are correctly formatted
            for ts_field in [
                "iocIngestTimestamp",
                "firstSeenTimestamp",
                "lastSeenTimestamp",
            ]:
                if ts_field in match:
                    # Should not end with Z after our processing
                    assert not match[ts_field].endswith("Z")

            # Check the associations if present
            if "associationIdentifier" in match:
                # Verify no duplicates with same name and type
                names_and_types = set()
                for assoc in match["associationIdentifier"]:
                    key = (assoc["name"], assoc["associationType"])
                    # Should not be able to add the same key twice if deduplication worked
                    assert key not in names_and_types
                    names_and_types.add(key)

        # Test with prioritized IoCs only
        prioritized_result = chronicle.list_iocs(
            start_time=start_time,
            end_time=end_time,
            max_matches=10,
            prioritized_only=True,
        )
        assert isinstance(prioritized_result, dict)
        prioritized_count = len(prioritized_result.get("matches", []))
        print(f"\nFound {prioritized_count} prioritized IoC matches")

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")  # Debug print
        # Skip the test rather than fail if no IoCs are available
        if "No IoCs found" in str(e):
            pytest.skip("No IoCs available in this environment")
        raise


@pytest.mark.integration
def test_chronicle_rule_management():
    """Test Chronicle rule management functionality with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Create a simple test rule
    test_rule_text = """
rule test_rule {
    meta:
        description = "Test rule for SDK testing"
        author = "Test Author"
        severity = "Low"
        yara_version = "YL2.0"
        rule_version = "1.0"
    events:
        $e.metadata.event_type = "NETWORK_CONNECTION"
    condition:
        $e
}
"""

    # Create the rule
    try:
        created_rule = chronicle.create_rule(test_rule_text)

        # Extract the rule ID from the response
        rule_name = created_rule.get("name", "")
        rule_id = rule_name.split("/")[-1]

        print(f"Created rule with ID: {rule_id}")

        # Get the rule
        rule = chronicle.get_rule(rule_id)
        assert rule.get("name") == rule_name
        assert "text" in rule

        # List rules and verify our rule is in the list
        rules = chronicle.list_rules()
        rule_names = [r.get("name") for r in rules.get("rules", [])]
        assert rule_name in rule_names

        # Update the rule with a modification
        updated_rule_text = test_rule_text.replace(
            'severity = "Low"', 'severity = "Medium"'
        )
        updated_rule = chronicle.update_rule(rule_id, updated_rule_text)
        assert updated_rule.get("name") == rule_name

        # Enable the rule
        deployment = chronicle.enable_rule(rule_id)
        assert "executionState" in deployment

        # Disable the rule
        deployment = chronicle.enable_rule(rule_id, False)
        assert "executionState" in deployment

        # Finally, delete the rule
        delete_result = chronicle.delete_rule(rule_id, force=True)
        assert delete_result == {}  # Empty response on success

        # Verify the rule is gone
        with pytest.raises(APIError):
            chronicle.get_rule(rule_id)

    except APIError as e:
        pytest.fail(f"API Error during rule management test: {str(e)}")


@pytest.mark.integration
def test_chronicle_rule_pagination():
    """Test pagination when listing rules with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Create multiple test rules to ensure we have enough for pagination
    created_rules = []

    try:
        # List all rules with a small page size to force pagination
        fetched_rules = chronicle.list_rules(page_size=1)

        # Checking if only 1 rule is fetched
        assert len(fetched_rules.get("rules", [])) == 1

        print(
            f"Successfully verified {len(created_rules)} test rules in paginated results"
        )

    except APIError as e:
        pytest.fail(f"API Error during rule pagination test: {str(e)}")

@pytest.mark.integration
def test_chronicle_search_rules():
    """Test Chronicle rule search functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    try:
        # Search for rules containing "Uppercase"
        result = chronicle.search_rules("Uppercase")

        # Basic validation of the response structure
        assert isinstance(result, dict)
        assert "rules" in result

        # Print some debug info about what we found
        print(f"\nFound {len(result['rules'])} rules containing 'Uppercase'")

        # If we found any rules, validate their structure
        if result["rules"]:
            rule = result["rules"][0]
            assert "name" in rule
            assert "text" in rule

            # Print the first rule's name for debugging
            print(f"First matching rule ID: {rule['name'].split('/')[-1]}")

            # Verify the search term appears in the rule text
            assert any(
                "Uppercase" in rule["text"] for rule in result["rules"]
            ), "Search term 'Uppercase' not found in any returned rule's text"
        else:
            print("No rules found containing 'Uppercase' - this is acceptable")

    except Exception as e:
        print(f"\nUnexpected error in rule search test: {type(e).__name__}: {str(e)}")
        pytest.skip(f"Test skipped due to unexpected error: {str(e)}")


@pytest.mark.integration
def test_chronicle_test_rule():
    """Test Chronicle rule testing functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Create a simple test rule that should find common events
    test_rule_text = """
rule test_network_events {
    meta:
        description = "Test rule for finding network connection events"
        author = "SecOps SDK Integration Test"
        severity = "Informational" 
        yara_version = "YL2.0"
        rule_version = "1.0"
    events:
        $e.metadata.event_type = "NETWORK_CONNECTION"
    condition:
        $e
}
"""

    try:
        print("\nStarting rule testing integration test...")

        # Set time range for testing - use a small window to ensure fast response
        end_time = datetime.now(timezone.utc) - timedelta(minutes=60)
        start_time = end_time - timedelta(
            minutes=80
        )  # Just test against 10 minutes of data

        print(f"Testing rule against data from {start_time} to {end_time}")
        print("Rule type: Simple network connection finder")

        # Initialize result tracking variables
        results = []
        progress_updates = []
        detection_count = 0
        error_messages = []

        # Use run_rule_test with streaming response
        for result in chronicle.run_rule_test(
            rule_text=test_rule_text,
            start_time=start_time,
            end_time=end_time,
            max_results=5,
        ):
            # Store results by type for validation
            if result.get("type") == "progress":
                progress_updates.append(result.get("percentDone", 0))
            elif result.get("type") == "detection":
                detection_count += 1
                results.append(result)
            elif result.get("type") == "error":
                error_messages.append(result.get("message", "Unknown error"))

        # Validate that we got progress updates
        assert (
            len(progress_updates) > 0
        ), "Should have received at least one progress update"

        # Print summary of what we found
        print(f"Received {len(progress_updates)} progress updates")
        print(f"Found {detection_count} detections")

        if error_messages:
            print(f"Encountered {len(error_messages)} errors: {error_messages}")

        # Check the progress updates - should have at least one with 100%
        assert any(
            p == 100 for p in progress_updates
        ), "Should have received a 100% progress update"

        # We don't assert on detection_count as it might be 0 in some environments
        # The test passes as long as the API responds properly

        # If we got detections, validate their structure
        if results:
            detection = results[0].get("detection", {})
            assert (
                "id" in detection or "resultEvents" in detection
            ), "Detection should have id or resultEvents field"
            print("Detection structure validation passed")

    except APIError as e:
        print(f"API Error during rule testing: {str(e)}")

        # If we get a "not found" or permission error, skip rather than fail
        if (
            "permission" in str(e).lower()
            or "not found" in str(e).lower()
            or "not enabled" in str(e).lower()
            or "not authorized" in str(e).lower()
            or "outside available data range"
            in str(e).lower()  # Also skip if data not available
        ):
            pytest.skip(
                f"Skipping due to permission/access issues or data range limitations: {str(e)}"
            )
        raise


@pytest.mark.integration
def test_chronicle_retrohunt():
    """Test Chronicle retrohunt functionality with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Create a simple test rule for retrohunting
    test_rule_text = """
rule test_retrohunt_rule {
    meta:
        description = "Test rule for retrohunt SDK testing"
        author = "Test Author"
        severity = "Low"
        yara_version = "YL2.0"
        rule_version = "1.0"
    events:
        $e.metadata.event_type = "NETWORK_CONNECTION"
    condition:
        $e
}
"""

    try:
        # Create the rule
        created_rule = chronicle.create_rule(test_rule_text)
        rule_name = created_rule.get("name", "")
        rule_id = rule_name.split("/")[-1]

        # Set up time range for retrohunt (from 48 hours ago to 24 hours ago)
        end_time = datetime.now(timezone.utc) - timedelta(hours=24)
        start_time = end_time - timedelta(hours=24)

        # Create retrohunt
        retrohunt = chronicle.create_retrohunt(rule_id, start_time, end_time)

        # Get operation ID from the response
        operation_name = retrohunt.get("name", "")
        operation_id = operation_name.split("/")[-1]

        print(f"Created retrohunt with operation ID: {operation_id}")

        # Get retrohunt status
        retrohunt_status = chronicle.get_retrohunt(rule_id, operation_id)
        assert "name" in retrohunt_status

        # Clean up
        chronicle.delete_rule(rule_id, force=True)

    except APIError as e:
        pytest.fail(f"API Error during retrohunt test: {str(e)}")


@pytest.mark.integration
def test_chronicle_rule_detections():
    """Test Chronicle rule detections functionality with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Use the specific rule ID provided
    rule_id = "ru_b2caeac4-c3bd-4b61-9007-bd1e481eff85"

    try:
        # List detections
        detections = chronicle.list_detections(rule_id)
        assert isinstance(detections, dict)
        print(f"Successfully retrieved detections for rule {rule_id}")

        # List errors
        errors = chronicle.list_errors(rule_id)
        assert isinstance(errors, dict)
        print(f"Successfully retrieved errors for rule {rule_id}")

    except APIError as e:
        pytest.fail(f"API Error during rule detections test: {str(e)}")


@pytest.mark.integration
def test_chronicle_rule_validation():
    """Test Chronicle rule validation functionality with real API."""
    client = SecOpsClient()
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Test with a valid rule
    valid_rule = """
rule test_rule {
    meta:
        description = "Test rule for validation"
        author = "Test Author"
        severity = "Low"
        yara_version = "YL2.0"
        rule_version = "1.0"
    events:
        $e.metadata.event_type = "NETWORK_CONNECTION"
    condition:
        $e
}
"""

    try:
        # Validate valid rule
        result = chronicle.validate_rule(valid_rule)
        assert result.success is True
        assert result.message is None
        assert result.position is None

        # Test with an invalid rule (missing condition)
        invalid_rule = """
rule test_rule {
    meta:
        description = "Test rule for validation"
        author = "Test Author"
        severity = "Low"
    events:
        $e.metadata.event_type = "NETWORK_CONNECTION"
}
"""
        result = chronicle.validate_rule(invalid_rule)
        assert result.success is False
        assert result.message is not None

    except APIError as e:
        pytest.fail(f"API Error during rule validation test: {str(e)}")


@pytest.mark.integration
def test_chronicle_nl_search():
    """Test Chronicle natural language search functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Use a smaller time window to minimize processing time
    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(minutes=10)

    try:
        # First, test the translation function only
        udm_query = chronicle.translate_nl_to_udm("ip address is known")

        print(f"\nTranslated query: {udm_query}")
        assert isinstance(udm_query, str)
        assert "ip" in udm_query  # Basic validation that it contains 'ip'

        # Now test the full search function
        # Try a simple query that should return results
        results = chronicle.nl_search(
            text="show me network connections",
            start_time=start_time,
            end_time=end_time,
            max_events=5,
        )

        assert isinstance(results, dict)
        assert "events" in results
        assert "total_events" in results

        print(f"\nFound {results.get('total_events', 0)} events")

        # Sleep for 10 seconds between nl_search calls
        # time.sleep(10) #fixed with 429 handler

        # Try a query that might not have results but should translate properly
        more_specific = chronicle.nl_search(
            text="show me failed login attempts",
            start_time=start_time,
            end_time=end_time,
            max_events=5,
        )

        assert isinstance(more_specific, dict)
        print(f"\nSpecific query found {more_specific.get('total_events', 0)} events")

    except APIError as e:
        if "no valid query could be generated" in str(e):
            # If translation fails, the test still passes as this is a valid API response
            print(f"\nAPI returned expected error for invalid query: {str(e)}")
            pytest.skip("Translation failed with expected error message")
        else:
            # For other API errors, fail the test
            print(f"\nAPI Error details: {str(e)}")
            raise


@pytest.mark.integration
def test_chronicle_data_export():
    """Test Chronicle data export functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Set up time range for testing
    end_time = datetime.now(timezone.utc)
    start_time = end_time - timedelta(days=14)  # Look back 1 day

    try:
        # First, fetch available log types
        log_types_result = chronicle.fetch_available_log_types(
            start_time=start_time,
            end_time=end_time,
            page_size=10,  # Limit to 10 for testing
        )

        print(
            f"\nFound {len(log_types_result['available_log_types'])} available log types for export"
        )

        # If no log types available, skip the test
        if not log_types_result["available_log_types"]:
            pytest.skip("No log types available for export in the specified time range")

        # Show some of the available log types
        for log_type in log_types_result["available_log_types"][:3]:  # Show first 3
            print(f"  {log_type.display_name} ({log_type.log_type.split('/')[-1]})")
            print(f"  Available from {log_type.start_time} to {log_type.end_time}")

        # For the actual export test, we'll create an export but not wait for completion
        # Choose a log type that's likely to be present
        if log_types_result["available_log_types"]:
            selected_log_type = log_types_result["available_log_types"][
                0
            ].log_type.split("/")[-1]

            # Create a data export (this might fail if the GCS bucket isn't properly set up)
            try:
                # This part would require a valid GCS bucket to work properly
                # We'll make the request but catch and report errors without failing the test
                bucket_name = "dk-test-export-bucket"

                export = chronicle.create_data_export(
                    gcs_bucket=f"projects/{CHRONICLE_CONFIG['project_id']}/buckets/{bucket_name}",
                    start_time=start_time,
                    end_time=end_time,
                    log_type=selected_log_type,
                )

                print(f"\nCreated data export for log type: {selected_log_type}")
                print(f"Export ID: {export['name'].split('/')[-1]}")
                print(f"Status: {export['data_export_status']['stage']}")

                # Test the get_data_export function
                export_id = export["name"].split("/")[-1]
                export_status = chronicle.get_data_export(export_id)
                print(
                    f"Retrieved export status: {export_status['data_export_status']['stage']}"
                )

                # Cancel the export
                cancelled = chronicle.cancel_data_export(export_id)
                print(
                    f"Cancelled export status: {cancelled['data_export_status']['stage']}"
                )

            except APIError as e:
                # Don't fail the test if export creation fails due to permissions
                # (GCS bucket access, etc.)
                print(f"\nData export creation failed: {str(e)}")
                print(
                    "This is expected if GCS bucket isn't configured or permissions are missing."
                )

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")  # Debug print
        # If we get "not found" or permission errors, skip rather than fail
        if "permission" in str(e).lower() or "not found" in str(e).lower():
            pytest.skip(f"Skipping due to permission issues: {str(e)}")
        raise


@pytest.mark.integration
def test_chronicle_batch_log_ingestion():
    """Test batch log ingestion with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Get current time for use in logs
    current_time = datetime.now(timezone.utc).isoformat().replace("+00:00", "Z")

    # Create several sample logs with different usernames
    okta_logs = []
    usernames = ["user1@example.com", "user2@example.com", "user3@example.com"]

    for username in usernames:
        okta_log = {
            "actor": {
                "displayName": f"Test User {username.split('@')[0]}",
                "alternateId": username,
            },
            "client": {
                "ipAddress": "192.168.1.100",
                "userAgent": {"os": "Mac OS X", "browser": "SAFARI"},
            },
            "displayMessage": "User login to Okta",
            "eventType": "user.session.start",
            "outcome": {"result": "SUCCESS"},
            "published": current_time,  # Use current time
        }
        okta_logs.append(json.dumps(okta_log))

    try:
        # Ingest multiple logs in a single API call
        print(f"\nIngesting {len(okta_logs)} logs in batch")
        result = chronicle.ingest_log(log_type="OKTA", log_message=okta_logs)

        # Verify response
        assert result is not None
        print(f"Batch ingestion result: {result}")
        if "operation" in result:
            assert result["operation"], "Operation ID should be present"
            print(f"Batch operation ID: {result['operation']}")

        # Test batch ingestion with a different valid log type
        # Create several Windows Defender ATP logs (simplified format for testing)
        defender_logs = [
            json.dumps(
                {
                    "DeviceId": "device1",
                    "Timestamp": current_time,
                    "FileName": "test1.exe",
                    "ActionType": "AntivirusDetection",
                    "SHA1": "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
                }
            ),
            json.dumps(
                {
                    "DeviceId": "device2",
                    "Timestamp": current_time,
                    "FileName": "test2.exe",
                    "ActionType": "SmartScreenUrlWarning",
                    "SHA1": "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb",
                }
            ),
            json.dumps(
                {
                    "DeviceId": "device3",
                    "Timestamp": current_time,
                    "FileName": "test3.exe",
                    "ActionType": "ProcessCreated",
                    "SHA1": "cccccccccccccccccccccccccccccccccccccccc",
                }
            ),
        ]

        # Ingest Windows Defender ATP logs in batch
        print(f"\nIngesting {len(defender_logs)} Windows Defender ATP logs in batch")
        try:
            defender_result = chronicle.ingest_log(
                log_type="WINDOWS_DEFENDER_ATP", log_message=defender_logs
            )

            # Verify response
            assert defender_result is not None
            print(f"Windows Defender ATP batch ingestion result: {defender_result}")
            if "operation" in defender_result:
                assert defender_result["operation"], "Operation ID should be present"
                print(
                    f"Windows Defender ATP batch operation ID: {defender_result['operation']}"
                )
        except APIError as e:
            # This might fail in some environments
            print(f"Windows Defender ATP ingestion reported API error: {e}")

    except APIError as e:
        print(f"\nAPI Error details: {str(e)}")
        # Skip the test rather than fail if permissions are not available
        if "permission" in str(e).lower():
            pytest.skip("Insufficient permissions to ingest logs")
        elif "invalid" in str(e).lower():
            pytest.skip("Invalid log format or API error")
        else:
            raise


@pytest.mark.integration
def test_chronicle_gemini():
    """Test Chronicle Gemini conversational AI functionality with real API.

    This test is designed to interact with the Gemini API and verify the response structure.
    """
    try:
        # Set up client
        client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
        chronicle = client.chronicle(**CHRONICLE_CONFIG)

        print("\nStarting Gemini integration test...")

        # Test with a simple, factual query that should have consistent responses
        query = "What is Windows event ID 4625?"
        print(f"Querying Gemini with: {query}")

        try:
            response = chronicle.gemini(query=query)

            # Basic structure validation
            print("Checking response structure...")
            assert hasattr(response, "blocks"), "Response should have blocks attribute"
            assert hasattr(response, "name"), "Response should have a name"
            assert hasattr(
                response, "create_time"
            ), "Response should have a creation time"
            assert (
                response.input_query == query
            ), "Response should contain the original query"

            # Check if we got some content
            assert (
                len(response.blocks) > 0
            ), "Response should have at least one content block"

            # Print some information about the response
            print(f"Received {len(response.blocks)} content blocks")

            # Check block types
            block_types = [block.block_type for block in response.blocks]
            print(f"Block types: {block_types}")

            # Check if we have text content
            text_content = response.get_text_content()
            if text_content:
                print(f"Text content (truncated): {text_content[:100]}...")

            # Check for code blocks (may or may not be present)
            code_blocks = response.get_code_blocks()
            if code_blocks:
                print(f"Found {len(code_blocks)} code blocks")
                for i, block in enumerate(code_blocks):
                    print(f"Code block {i+1} title: {block.title}")

            # Check for references (may or may not be present)
            if response.references:
                print(f"Found {len(response.references)} references")

            # Check for suggested actions (may or may not be present)
            if response.suggested_actions:
                print(f"Found {len(response.suggested_actions)} suggested actions")
                for i, action in enumerate(response.suggested_actions):
                    print(
                        f"Action {i+1}: {action.display_text} (type: {action.action_type})"
                    )

            print("Gemini integration test passed successfully.")

        except APIError as e:
            if "users must opt-in before using Gemini" in str(e):
                pytest.skip(
                    "Test skipped: User account has not been opted-in to Gemini. Please enable Gemini in Chronicle settings."
                )
            else:
                raise  # Re-raise if it's a different API error

    except Exception as e:
        print(f"Unexpected error in Gemini test: {type(e).__name__}: {str(e)}")
        pytest.skip(f"Test skipped due to unexpected error: {str(e)}")


@pytest.mark.integration
def test_chronicle_gemini_text_content():
    """Test that GeminiResponse.get_text_content() properly strips HTML.

    Uses a query known to return HTML blocks and verifies that the text
    content includes the information from HTML blocks without the tags.
    """
    try:
        # Set up client
        client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
        chronicle = client.chronicle(**CHRONICLE_CONFIG)

        print("\nStarting Gemini get_text_content() integration test...")

        # Query known to return HTML blocks
        query = "What is Windows event ID 4625?"
        print(f"Querying Gemini with: {query}")

        try:
            response = chronicle.gemini(query=query)

            # Basic structure validation
            assert hasattr(response, "blocks"), "Response should have blocks attribute"
            assert (
                len(response.blocks) > 0
            ), "Response should have at least one content block"

            # Find an HTML block in the response
            html_block_content = None
            for block in response.blocks:
                if block.block_type == "HTML":
                    html_block_content = block.content
                    print(
                        f"Found HTML block content (raw): {html_block_content[:200]}..."
                    )
                    break

            assert (
                html_block_content is not None
            ), "Response should contain at least one HTML block for this test"

            # Get the combined text content
            text_content = response.get_text_content()
            print(f"Combined text content (stripped): {text_content[:200]}...")

            assert text_content, "get_text_content() should return non-empty string"

            # Check that HTML tags are stripped
            assert "<p>" not in text_content, "HTML <p> tags should be stripped"
            assert "<li>" not in text_content, "HTML <li> tags should be stripped"
            assert "<a>" not in text_content, "HTML <a> tags should be stripped"
            assert (
                "<strong>" not in text_content
            ), "HTML <strong> tags should be stripped"

            # Check that the *content* from the HTML block is present (approximate check)
            # We strip tags from the original HTML and check if a snippet exists in the combined text
            stripped_html_for_check = re.sub(
                r"<[^>]+>", " ", html_block_content
            ).strip()
            # Take a small snippet from the stripped HTML to verify its presence
            snippet_to_find = (
                stripped_html_for_check[:50].split()[-1]
                if stripped_html_for_check
                else None
            )  # Get last word of first 50 chars
            if snippet_to_find:
                print(f"Verifying presence of snippet: '{snippet_to_find}'")
                assert (
                    snippet_to_find in text_content
                ), f"Text content should include content from HTML block (missing snippet: {snippet_to_find})"

            print("Gemini get_text_content() HTML stripping test passed successfully.")

        except APIError as e:
            if "users must opt-in before using Gemini" in str(e):
                pytest.skip(
                    "Test skipped: User account has not been opted-in to Gemini. Please enable Gemini in Chronicle settings."
                )
            else:
                raise  # Re-raise if it's a different API error

    except Exception as e:
        print(
            f"Unexpected error in Gemini get_text_content test: {type(e).__name__}: {str(e)}"
        )
        pytest.skip(f"Test skipped due to unexpected error: {str(e)}")


@pytest.mark.integration
def test_chronicle_gemini_rule_generation():
    """Test Chronicle Gemini's ability to generate security rules.

    This test asks Gemini to generate a detection rule and verifies the response structure.
    """
    try:
        # Set up client
        client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
        chronicle = client.chronicle(**CHRONICLE_CONFIG)

        print("\nStarting Gemini rule generation test...")

        # Ask Gemini to generate a detection rule
        query = "Write a rule to detect powershell downloading a file called gdp.zip"
        print(f"Querying Gemini with: {query}")

        try:
            response = chronicle.gemini(query=query)

            # Basic structure validation
            assert (
                len(response.blocks) > 0
            ), "Response should have at least one content block"

            # We should have at least one code block for the rule
            code_blocks = response.get_code_blocks()
            assert (
                len(code_blocks) > 0
            ), "Response should contain at least one code block with the rule"

            # Verify the code block contains a YARA-L rule
            rule_block = code_blocks[0]
            assert (
                "rule " in rule_block.content
            ), "Code block should contain a YARA-L rule"
            assert "meta:" in rule_block.content, "Rule should have a meta section"
            assert "events:" in rule_block.content, "Rule should have an events section"
            assert (
                "condition:" in rule_block.content
            ), "Rule should have a condition section"

            # Check for powershell and gdp.zip in the rule
            assert (
                "powershell" in rule_block.content.lower()
            ), "Rule should reference powershell"
            assert (
                "gdp.zip" in rule_block.content.lower()
            ), "Rule should reference gdp.zip"

            # Check for suggested actions (typically rule editor)
            if response.suggested_actions:
                rule_editor_action = [
                    action
                    for action in response.suggested_actions
                    if "rule" in action.display_text.lower()
                    and action.action_type == "NAVIGATION"
                ]
                if rule_editor_action:
                    print(
                        f"Found rule editor action: {rule_editor_action[0].display_text}"
                    )
                    assert (
                        rule_editor_action[0].navigation is not None
                    ), "Navigation action should have a target URI"

            print("Gemini rule generation test passed successfully.")

        except APIError as e:
            if "users must opt-in before using Gemini" in str(e):
                pytest.skip(
                    "Test skipped: User account has not been opted-in to Gemini. Please enable Gemini in Chronicle settings."
                )
            else:
                raise  # Re-raise if it's a different API error

    except Exception as e:
        print(
            f"Unexpected error in Gemini rule generation test: {type(e).__name__}: {str(e)}"
        )
        pytest.skip(f"Test skipped due to unexpected error: {str(e)}")


@pytest.mark.integration
def test_chronicle_data_tables():
    """Test Chronicle data table functionality with API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Use timestamp for unique names to avoid conflicts
    timestamp = datetime.now(timezone.utc).strftime("%Y%m%d%H%M%S")
    dt_name = f"sdktest_dt_{timestamp}"

    try:
        print("\n>>> Testing data table operations")

        # List existing data tables (to verify API access)
        try:
            initial_tables = chronicle.list_data_tables(order_by="createTime asc")
            print(f"Found {len(initial_tables)} existing data tables")
        except APIError as e:
            if "invalid order by field" in str(e):
                # Handle the specific error we know about
                print("Note: API only supports 'createTime asc' for ordering")
                initial_tables = chronicle.list_data_tables()  # Try without order_by
            else:
                raise

        # Create a data table with string columns
        print(f"Creating data table: {dt_name}")
        created_dt = chronicle.create_data_table(
            name=dt_name,
            description="SDK Integration Test Data Table",
            header={
                "hostname": DataTableColumnType.STRING,
                "ip_address": DataTableColumnType.STRING,
                "description": DataTableColumnType.STRING,
            },
            rows=[
                ["host1.example.com", "192.168.1.10", "Primary server"],
                ["host2.example.com", "192.168.1.11", "Backup server"],
            ],
        )

        print(f"Created data table: {created_dt.get('name')}")
        assert created_dt.get("name").endswith(dt_name)
        assert created_dt.get("description") == "SDK Integration Test Data Table"

        # Get the data table
        retrieved_dt = chronicle.get_data_table(dt_name)
        assert retrieved_dt.get("name") == created_dt.get("name")
        assert len(retrieved_dt.get("columnInfo", [])) == 3

        # List rows
        rows = chronicle.list_data_table_rows(dt_name)
        print(f"Found {len(rows)} rows in data table")
        assert len(rows) == 2  # We added 2 rows during creation

        # Store row IDs for deletion testing
        row_ids = [
            row.get("name", "").split("/")[-1] for row in rows if row.get("name")
        ]

        if row_ids:
            # Delete one row
            row_to_delete = row_ids[0]
            print(f"Deleting row: {row_to_delete}")
            delete_result = chronicle.delete_data_table_rows(dt_name, [row_to_delete])

            # Check rows after deletion
            updated_rows = chronicle.list_data_table_rows(dt_name)
            assert len(updated_rows) == 1  # Should be one less than before

        # Add more rows
        new_rows = [
            ["host3.example.com", "192.168.1.12", "Development server"],
            ["host4.example.com", "192.168.1.13", "Test server"],
        ]
        print("Adding more rows")
        add_rows_result = chronicle.create_data_table_rows(dt_name, new_rows)

        # Check rows after addition
        final_rows = chronicle.list_data_table_rows(dt_name)
        assert len(final_rows) == 3  # 1 remaining + 2 new ones

    except Exception as e:
        print(f"Error during data table test: {e}")
        raise
    finally:
        # Clean up - delete the test data table
        try:
            print(f"Cleaning up - deleting data table: {dt_name}")
            chronicle.delete_data_table(dt_name, force=True)
            print("Data table deleted successfully")
        except Exception as cleanup_error:
            print(f"Warning: Failed to clean up data table {dt_name}: {cleanup_error}")


@pytest.mark.integration
def test_chronicle_data_tables_cidr():
    """Test Chronicle data table functionality with CIDR columns."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    timestamp = datetime.now(timezone.utc).strftime("%Y%m%d%H%M%S")
    dt_name = f"sdktest_dt_cidr_{timestamp}"

    try:
        print("\n>>> Testing data table with CIDR column")

        # Create a data table with CIDR column
        created_dt = chronicle.create_data_table(
            name=dt_name,
            description="SDK Integration Test Data Table with CIDR",
            header={
                "network": DataTableColumnType.CIDR,
                "location": DataTableColumnType.STRING,
            },
            rows=[["10.0.0.0/8", "Corporate HQ"], ["192.168.0.0/16", "Branch offices"]],
        )

        print(f"Created CIDR data table: {created_dt.get('name')}")
        assert created_dt.get("name").endswith(dt_name)

        # Get the data table to verify CIDR column type
        retrieved_dt = chronicle.get_data_table(dt_name)
        column_info = retrieved_dt.get("columnInfo", [])

        # Find the CIDR column
        cidr_column = next(
            (col for col in column_info if col.get("originalColumn") == "network"), None
        )
        assert cidr_column is not None
        assert cidr_column.get("columnType") == "CIDR"

        # List rows to verify CIDR values were stored correctly
        rows = chronicle.list_data_table_rows(dt_name)
        assert len(rows) == 2

        # Try to add an invalid CIDR to test validation
        try:
            chronicle.create_data_table_rows(
                dt_name, [["not-a-cidr", "Invalid Network"]]
            )
            pytest.fail("Should have raised an error for invalid CIDR")
        except APIError as e:
            print(f"Expected error for invalid CIDR: {e}")
            assert "not a valid CIDR" in str(e) or "Invalid Row Value" in str(e)

    except Exception as e:
        print(f"Error during CIDR data table test: {e}")
        raise
    finally:
        # Clean up
        try:
            chronicle.delete_data_table(dt_name, force=True)
        except Exception as cleanup_error:
            print(
                f"Warning: Failed to clean up CIDR data table {dt_name}: {cleanup_error}"
            )


@pytest.mark.integration
def test_replace_data_table_rows():
    """Test replace_data_table_rows functionality with real API.
    
    This test creates a data table with initial rows, replaces all rows with new ones,
    verifies the replacement, and then cleans up by deleting the data table.
    """
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)
    
    # Create a unique data table name for this test using timestamp
    dt_name = f"replace_rows_test_{int(time.time())}"
    
    try:
        print(f"\n>>> Testing replace_data_table_rows for: {dt_name}")
        
        # Create a data table with initial rows
        created_dt = chronicle.create_data_table(
            name=dt_name,
            description="Test Data Table for Replace Rows",
            header={
                "hostname": DataTableColumnType.STRING,
                "ip_address": DataTableColumnType.STRING,
                "description": DataTableColumnType.STRING,
            },
            rows=[
                ["initial1.example.com", "10.0.0.1", "Initial host 1"],
                ["initial2.example.com", "10.0.0.2", "Initial host 2"],
                ["initial3.example.com", "10.0.0.3", "Initial host 3"],
            ],
        )
        
        print(f"Created data table: {dt_name} with 3 initial rows")
        assert created_dt.get("name").endswith(dt_name)
        
        # Verify initial rows
        initial_rows = chronicle.list_data_table_rows(dt_name)
        assert len(initial_rows) == 3, f"Expected 3 initial rows, got {len(initial_rows)}"
        
        # Prepare new rows to replace existing ones
        replacement_rows = [
            ["replaced1.example.com", "192.168.1.1", "Replaced host 1"],
            ["replaced2.example.com", "192.168.1.2", "Replaced host 2"],
        ]
        
        print(f"Replacing all rows in data table: {dt_name}")
        # Replace all rows with new ones
        replace_results = chronicle.replace_data_table_rows(dt_name, replacement_rows)
        
        # Verify replacement response
        assert len(replace_results) == 1, "Expected single response for batch replacement"
        assert "dataTableRows" in replace_results[0], "Response should contain dataTableRows"
        
        # Verify rows after replacement
        updated_rows = chronicle.list_data_table_rows(dt_name)
        assert len(updated_rows) == 2, f"Expected 2 rows after replacement, got {len(updated_rows)}"
        
    except Exception as e:
        print(f"Error in test_replace_data_table_rows: {str(e)}")
        pytest.fail(f"Replace data table rows test failed: {e}")
        
    finally:
        # Clean up - delete the test data table
        try:
            print(f"Cleaning up - deleting data table: {dt_name}")
            chronicle.delete_data_table(dt_name, force=True)
            print("Data table deleted successfully")
        except Exception as cleanup_error:
            print(f"Warning: Failed to clean up data table {dt_name}: {cleanup_error}")


@pytest.mark.integration
def test_update_data_table():
    """Test update_data_table functionality with real API.
    
    This test creates a data table, updates its properties, verifies the
    changes, and then cleans up by deleting the data table.
    """
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)
    
    # Create a unique data table name for this test using timestamp
    dt_name = f"update_dt_test_{int(time.time())}"
    
    try:
        print(f"\n>>> Testing update_data_table for: {dt_name}")
        
        # Create a data table
        initial_description = "Initial data table description"
        
        created_dt = chronicle.create_data_table(
            name=dt_name,
            description=initial_description,
            header={
                "hostname": DataTableColumnType.STRING,
                "ip_address": DataTableColumnType.STRING,
                "description": DataTableColumnType.STRING,
            },
            rows=[
                ["host1.example.com", "192.168.1.10", "Test host"],
            ],
        )
        
        print(f"Created data table: {dt_name}")
        assert created_dt.get("name").endswith(dt_name)
        assert created_dt.get("description") == initial_description
        
        # Update the data table - change description and row TTL
        updated_description = "Updated data table description"
        updated_ttl = "48h"
        
        print(f"Updating data table with new description and TTL")
        update_result = chronicle.update_data_table(
            name=dt_name,
            description=updated_description,
            row_time_to_live=updated_ttl
        )
        
        # Verify the update was successful
        assert update_result.get("name").endswith(dt_name)
        
        # Retrieve the data table to double-check changes persisted
        retrieved_dt = chronicle.get_data_table(dt_name)
        assert retrieved_dt.get("description") == updated_description
        assert retrieved_dt.get("rowTimeToLive") == updated_ttl
        
        # Test partial update - only update description
        final_description = "Final description"
        
        print(f"Updating only the description")
        partial_update = chronicle.update_data_table(
            name=dt_name,
            description=final_description
        )
        
        assert dt_name in partial_update.get("name")
        
        # Test update with explicit update_mask
        masked_ttl = "72h"
        masked_desc = "This description should not be applied"
        
        print(f"Testing update with update_mask")
        chronicle.update_data_table(
            name=dt_name,
            description=masked_desc,
            row_time_to_live=masked_ttl,
            update_mask=["row_time_to_live"]  # Only update TTL
        )
    

        # Retrieve the data table to check changes persisted
        retrieved_updated_dt = chronicle.get_data_table(dt_name)
        assert retrieved_updated_dt.get("description") == final_description
        assert retrieved_updated_dt.get("rowTimeToLive") == masked_ttl
        
        print("All update_data_table tests passed successfully")
        
    except Exception as e:
        print(f"Error in test_update_data_table: {str(e)}")
        pytest.fail(f"Update data table test failed: {e}")
        
    finally:
        # Clean up - delete the test data table
        try:
            print(f"Cleaning up - deleting data table: {dt_name}")
            chronicle.delete_data_table(dt_name, force=True)
            print("Data table deleted successfully")
        except Exception as cleanup_error:
            print(f"Warning: Failed to clean up data table {dt_name}: {cleanup_error}")


@pytest.mark.integration
def test_chronicle_reference_lists():
    """Test Chronicle reference list functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    timestamp = datetime.now(timezone.utc).strftime("%Y%m%d%H%M%S")
    rl_name = f"sdktest_rl_{timestamp}"

    try:
        print("\n>>> Testing reference list operations")

        # List existing reference lists
        initial_lists = chronicle.list_reference_lists()
        print(f"Found {len(initial_lists)} existing reference lists")

        # Create a reference list
        print(f"Creating reference list: {rl_name}")
        created_rl = chronicle.create_reference_list(
            name=rl_name,
            description="SDK Integration Test Reference List",
            entries=[
                "malicious.example.com",
                "suspicious.example.org",
                "evil.example.net",
            ],
            syntax_type=ReferenceListSyntaxType.STRING,
        )

        print(f"Created reference list: {created_rl.get('name')}")
        assert created_rl.get("name").endswith(rl_name)
        assert created_rl.get("description") == "SDK Integration Test Reference List"

        # Get the reference list with FULL view
        retrieved_rl_full = chronicle.get_reference_list(
            rl_name, view=ReferenceListView.FULL
        )
        assert retrieved_rl_full.get("name").endswith(rl_name)
        assert len(retrieved_rl_full.get("entries", [])) == 3

        # Get the reference list with BASIC view
        retrieved_rl_basic = chronicle.get_reference_list(
            rl_name, view=ReferenceListView.BASIC
        )
        assert retrieved_rl_basic.get("name").endswith(rl_name)

        # Update the reference list
        updated_description = "Updated SDK Test Reference List"
        updated_entries = ["updated.example.com", "new.example.org"]

        updated_rl = chronicle.update_reference_list(
            name=rl_name, description=updated_description, entries=updated_entries
        )

        assert updated_rl.get("description") == updated_description
        assert len(updated_rl.get("entries", [])) == 2

        # Verify update with a get
        final_rl = chronicle.get_reference_list(rl_name)
        assert final_rl.get("description") == updated_description
        assert len(final_rl.get("entries", [])) == 2

    except Exception as e:
        print(f"Error during reference list test: {e}")
        raise
    finally:
        # Note: Reference list deletion is not supported by the API
        print(
            f"Note: Reference list {rl_name} remains since reference list deletion is not supported by the API"
        )


@pytest.mark.integration
def test_chronicle_reference_lists_cidr():
    """Test Chronicle reference list functionality with CIDR syntax type."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    timestamp = datetime.now(timezone.utc).strftime("%Y%m%d%H%M%S")
    rl_name = f"sdktest_rl_cidr_{timestamp}"

    try:
        print("\n>>> Testing CIDR reference list operations")

        # Create a CIDR reference list
        created_rl = chronicle.create_reference_list(
            name=rl_name,
            description="SDK Integration Test CIDR Reference List",
            entries=["10.0.0.0/8", "172.16.0.0/12", "192.168.0.0/16"],
            syntax_type=ReferenceListSyntaxType.CIDR,
        )

        print(f"Created CIDR reference list: {created_rl.get('name')}")
        assert created_rl.get("name").endswith(rl_name)
        assert created_rl.get("syntaxType") == ReferenceListSyntaxType.CIDR.value

        # Get the reference list to verify CIDR entries
        retrieved_rl = chronicle.get_reference_list(rl_name)
        assert retrieved_rl.get("syntaxType") == ReferenceListSyntaxType.CIDR.value
        assert len(retrieved_rl.get("entries", [])) == 3

        # Try to update with an invalid CIDR to test validation
        try:
            chronicle.update_reference_list(
                name=rl_name, entries=["not-a-cidr", "192.168.1.0/24"]
            )
            pytest.fail("Should have raised an error for invalid CIDR")
        except SecOpsError as e:
            print(f"Expected error for invalid CIDR: {e}")
            assert "Invalid CIDR entry" in str(e)

    except Exception as e:
        print(f"Error during CIDR reference list test: {e}")
        raise
    finally:
        # Note: Reference list deletion is not supported by the API
        print(
            f"Note: CIDR reference list {rl_name} remains since reference list deletion is not supported by the API"
        )

@pytest.mark.integration
def test_chronicle_generate_udm_key_value_mapping():
    """Test UDM key/value mapping functionality with real API."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    # Sample Row JSON log
    json_log = """
    {
        "events": [
            {
                "id": "123",
                "user": "test_user",
                "source_ip": "192.168.1.10",
                "destination_url": "www.example.com",
                "action_taken": "allowed",
                "timestamp": 1588059648129
            },
            {
                "id": "231",
                "user": "test_user2",
                "source_ip": "192.168.1.9",
                "destination_url": "www.example2.com",
                "action_taken": "allowed",
                "timestamp": 1588059649129
            }
        ]
    }
    """

    try:
        print("\n>>> Testing Generating UDM key/value mapping")

        json_mappings = chronicle.generate_udm_key_value_mappings(
            log_format="JSON",
            log=json_log,
            use_array_bracket_notation=True,
            compress_array_fields=False,
        )
        print(f"JSON mappings retrieved: {len(json_mappings)} fields")

        # Verify we got expected mapping fields
        assert isinstance(json_mappings, dict)
        assert len(json_mappings) > 0

        # Check for specific expected fields
        expected_fields = [
            "events[0].id",
            "events[0].user",
            "events[1].id",
            "events[1].user",
        ]

        # Check for expected fields
        assert all(field in json_mappings for field in expected_fields)

        print("UDM key/value mapping successful")

    except Exception as e:
        print(f"Error during generate UDM key/value mapping test: {e}")
        raise


@pytest.mark.integration
def test_find_udm_field_values():
    """Test find_udm_field_values functionality."""
    client = SecOpsClient(service_account_info=SERVICE_ACCOUNT_JSON)
    chronicle = client.chronicle(**CHRONICLE_CONFIG)

    try:
        # Test with page_size parameter
        page_size = 1  # Small page size to ensure we get a next page token
        query = "source"  # Common term that should exist in most environments
        paged_results = chronicle.find_udm_field_values(
            query=query, page_size=page_size
        )

        assert paged_results
        assert "valueMatches" in paged_results
        assert "fieldMatches" in paged_results
        assert "fieldMatchRegex" in paged_results
        assert paged_results["fieldMatchRegex"]
        # Verify pagination works
        assert (
            len(paged_results["valueMatches"]) <= page_size
        ), f"Should return at most {page_size} value matches"

        print(
            f"Pagination test: got {len(paged_results['valueMatches'])} "
            f"value matches with page_size={page_size}"
        )

        # Check if we have a next page token
        if "nextPageToken" in paged_results:
            # Check next page token
            assert paged_results["nextPageToken"]
        else:
            print("No next page token found")

    except APIError as e:
        print(f"API Error during find_udm_field_values test: {str(e)}")
        pytest.fail(f"Test failed due to API error: {str(e)}")
    except Exception as e:
        print(f"Unexpected error in find_udm_field_values test: {str(e)}")
        raise
