# django-ipc: Production-Ready WebSocket RPC for Django

[![Python 3.10+](https://img.shields.io/badge/python-3.10+-blue.svg?style=flat-square&logo=python)](https://www.python.org/downloads/)
[![Django Compatible](https://img.shields.io/badge/django-compatible-green.svg?style=flat-square&logo=django)](https://www.djangoproject.com/)
[![PyPI](https://img.shields.io/pypi/v/django-ipc.svg?style=flat-square&logo=pypi)](https://pypi.org/project/django-ipc/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg?style=flat-square)](https://opensource.org/licenses/MIT)
[![Tests](https://img.shields.io/badge/tests-259%20passed-brightgreen.svg?style=flat-square)](docs/reports/test-report.md)
[![Downloads](https://img.shields.io/pypi/dm/django-ipc.svg?style=flat-square)](https://pypi.org/project/django-ipc/)

<div align="center">

### 🚀 WebSocket RPC for Django - From Zero to Production in 5 Minutes

**Auto-generated clients** • **100% type-safe** • **Production-ready** • **Zero configuration**

**Part of the [django-cfg](https://djangocfg.com) ecosystem**

**[📚 Full Documentation](https://djangocfg.com/docs/features/integrations/websocket-ipc)** • **[🎯 Live Demo](http://demo.djangocfg.com)** • **[⚡ Quick Start](#-quick-start)**

</div>

---

## 🎯 What is django-ipc?

**django-ipc** is a production-ready WebSocket RPC framework that brings **real-time communication to Django** in 5 minutes. Auto-generate TypeScript & Python clients, deploy with Redis & WebSocket, and scale to 10,000+ connections.

### Why django-ipc?

**Traditional Django real-time wastes 28,800 requests/day with polling. django-ipc delivers instant updates with 1 persistent connection.**

- ✅ **5-minute setup** - No complex configuration like Django Channels
- ✅ **Auto-generated clients** - TypeScript & Python generated automatically
- ✅ **100% type-safe** - Full Pydantic v2 validation (Python ↔ TypeScript)
- ✅ **Zero boilerplate** - 19 files generated, 0 lines of manual code
- ✅ **Production-ready** - Horizontal scaling, load balancing, monitoring
- ✅ **Django-CFG integration** - Works standalone or with django-cfg ecosystem

**[📚 Why django-ipc? See comparison →](https://djangocfg.com/docs/features/integrations/websocket-ipc/why-websocket-rpc)**

---

## 🚀 Quick Start

### Installation

```bash
pip install django-ipc
```

### 1. Start WebSocket Server

```python
# rpc_server.py
import asyncio
from django_ipc.server import WebSocketServer
from django_ipc.server.config import ServerConfig, WSServerConfig, AuthMode

config = ServerConfig(
    server=WSServerConfig(
        host="0.0.0.0",
        port=8765,
        redis_url="redis://localhost:6379/2",
        auth_mode=AuthMode.NONE,  # Development only!
    )
)

async def main():
    server = WebSocketServer(config)
    await server.start()

if __name__ == "__main__":
    print("🚀 Starting WebSocket RPC Server...")
    print("📡 WebSocket: ws://localhost:8765")
    asyncio.run(main())
```

### 2. Generate Clients (One Command!)

```bash
python -m django_ipc.codegen.cli generate-clients \
    --output ./clients \
    --redis-url redis://localhost:6379/2
```

**Result**: 19 production-ready files! ✨

```
clients/
├── typescript/          # 10 files - TypeScript client + configs
│   ├── client.ts       # Type-safe RPC client
│   ├── types.ts        # TypeScript interfaces
│   ├── tsconfig.json   # ✅ Auto-generated
│   ├── package.json    # ✅ Auto-generated (8 npm scripts!)
│   ├── .eslintrc.json  # ✅ Auto-generated
│   └── README.md       # ✅ Auto-generated docs
└── python/              # 9 files - Python client + configs
    ├── client.py        # Type-safe RPC client
    ├── models.py        # Pydantic models
    ├── pyproject.toml   # ✅ Auto-generated
    └── README.md        # ✅ Auto-generated docs
```

### 3. Send Real-Time Notifications from Django

```python
# Django view
from django_ipc.client import RPCClient

def notify_user(request):
    rpc = RPCClient(redis_url="redis://localhost:6379/2")

    # Send notification - arrives INSTANTLY on frontend! ⚡
    rpc.send_notification(
        user_id=request.user.id,
        message="Your order has been shipped!",
        data={"order_id": 123, "tracking": "ABC123"}
    )

    return JsonResponse({"status": "sent"})
```

### 4. Receive Notifications on Frontend

```typescript
// TypeScript client - auto-generated
import { RPCClient } from './clients/typescript';

const client = new RPCClient('ws://localhost:8765');
await client.connect();

// Listen for real-time notifications
client.on('notification', (notification) => {
    console.log('📬 Notification:', notification.message);
    showToast(notification);  // Show to user instantly!
});
```

**[📚 Full 5-minute tutorial →](https://djangocfg.com/docs/features/integrations/websocket-ipc/quick-start)**

---

## ⭐ Key Features

### 🤖 Auto-Generated Clients (Zero Manual Code)

**One command generates production-ready TypeScript + Python clients:**

- ✅ **TypeScript client** - 100% type-safe interfaces
- ✅ **Python client** - Full Pydantic validation
- ✅ **Complete tooling** - ESLint, Prettier, mypy, all configured
- ✅ **Ready to deploy** - package.json, pyproject.toml, README.md included

### 🌍 Environment-Aware Configuration

**Auto-detect development/staging/production environments:**

```python
# Python client
client = RPCClient.from_env()  # Auto-detects DJANGO_ENV

# TypeScript client
const client = RPCClient.fromEnv();  # Auto-detects NODE_ENV
```

**Supported environments**: `development`, `staging`, `production`, `testing`

### 📡 Production-Ready WebSocket Server

**Built-in features for production scale:**

- ✅ **10,000+ concurrent connections** per server
- ✅ **Horizontal scaling** - Multiple WebSocket servers
- ✅ **Load balancing** - Nginx WebSocket configuration
- ✅ **JWT authentication** - Secure WebSocket connections
- ✅ **Health checks** - HTTP health endpoint
- ✅ **Monitoring** - Built-in metrics

**[📚 Production deployment guide →](https://djangocfg.com/docs/features/integrations/websocket-ipc/deployment)**

### 🔄 Redis IPC Bridge

**Async bridge for Django ↔ WebSocket communication:**

- ✅ **Type-safe messages** - Pydantic v2 validation
- ✅ **Request/response** - RPC-style communication
- ✅ **Pub/sub patterns** - Notifications, broadcasts, room messaging
- ✅ **Stream processing** - Redis Streams for reliable delivery

---

## 📚 Complete Documentation

### 🚀 Getting Started (15 minutes)

**Start here if you're new to django-ipc:**

- **[Quick Start Guide](https://djangocfg.com/docs/features/integrations/websocket-ipc/quick-start)** ⚡ **(5 min)** - Get it working
- **[Why django-ipc?](https://djangocfg.com/docs/features/integrations/websocket-ipc/why-websocket-rpc)** 💡 **(3 min)** - Understand the value
- **[Real-Time Notifications](https://djangocfg.com/docs/features/integrations/websocket-ipc/real-time-notifications)** 📬 **(15 min)** - 4 notification patterns

### 🏗 Integration & Production (1 hour)

**Integrate into your Django project:**

- **[Django Integration Guide](https://djangocfg.com/docs/features/integrations/websocket-ipc/integration)** 🔗 **(30 min)** - Step-by-step setup
- **[Production Deployment](https://djangocfg.com/docs/features/integrations/websocket-ipc/deployment)** 🚢 **(45 min)** - Docker + scaling
- **[Architecture Overview](https://djangocfg.com/docs/features/integrations/websocket-ipc/architecture)** 🏛️ **(15 min)** - System design

### 💡 Real-World Examples

**Production-ready use cases with code:**

- **[Use Cases & Examples](https://djangocfg.com/docs/features/integrations/websocket-ipc/use-cases)** 🌍 **(20 min)** - 5 complete examples
  - E-commerce order tracking (99% API reduction)
  - Live chat application (<50ms latency)
  - Dashboard metrics (real-time updates)
  - Multiplayer game lobby
  - Stock price alerts

### 📊 Understanding the System

**Deep dives and technical details:**

- **[How It Works](https://djangocfg.com/docs/features/integrations/websocket-ipc/how-it-works)** 🔄 **(10 min)** - Visual message flow
- **[Business Value & ROI](https://djangocfg.com/docs/features/integrations/websocket-ipc/business-value)** 💰 **(10 min)** - $68K savings calculator

---

## 🤝 Django-CFG Integration

**django-ipc is part of the django-cfg ecosystem:**

### Standalone Usage

```python
from django_ipc.client import RPCClient

rpc = RPCClient(redis_url="redis://localhost:6379/2")
rpc.send_notification(user_id="123", message="Hello!")
```

### With django-cfg (Type-Safe Django Configuration)

```python
from django_cfg import DjangoConfig
from django_cfg.modules.django_ipc import get_rpc_client

class MyConfig(DjangoConfig):
    project_name: str = "My SaaS App"
    # django-ipc auto-configured

# Use in Django views
rpc = get_rpc_client()
rpc.send_notification(user_id="123", message="Hello!")
```

**[📚 Learn more about django-cfg →](https://github.com/markolofsen/django-cfg)**

---

## 🏗️ Architecture

```
┌─────────────┐         ┌─────────┐         ┌──────────────────┐
│   Django    │         │  Redis  │         │ WebSocket Server │
│     App     │         │   IPC   │         │   (django-ipc)   │
└──────┬──────┘         └────┬────┘         └────────┬─────────┘
       │                     │                       │
       │──RPC Request────────▶                       │
       │   (XADD stream)     │──XREADGROUP──────────▶
       │                     │                       │
       │                     │                  [RPC Bridge]
       │                     │                  [Handlers]
       │                     │                       │
       │                     │                       │───▶ Users (WebSocket)
       │                     │◀──Response (LPUSH)────│
       │◀─RPC Response───────│                       │
       │                     │                       │

┌─────────────────────────────────────────────────────────────┐
│              Auto-Generated Clients (TypeScript/Python)      │
│                          │                                   │
│          WebSocket ──────┴────────── WebSocket Server       │
└─────────────────────────────────────────────────────────────┘
```

---

## 💼 Production-Ready

**259 tests, 100% pass rate** ✅

```bash
pytest tests/ -v
# 259 passed, 65 warnings in 0.75s
```

**Features for production:**

- ✅ **JWT Authentication** - Secure WebSocket connections
- ✅ **Health Checks** - HTTP endpoint for monitoring
- ✅ **Horizontal Scaling** - Multiple servers with load balancing
- ✅ **Error Handling** - Graceful degradation
- ✅ **Type Safety** - 100% Pydantic validation
- ✅ **Logging** - Rich console output with loguru

**[📊 Test Report →](docs/reports/test-report.md)**

---

## 📋 Requirements

- Python 3.10+
- pydantic >= 2.11.0
- redis >= 6.4.0
- websockets >= 15.0
- jinja2 >= 3.1.0 (for code generation)
- rich >= 14.1.0 (for pretty output)

**Optional**: Django 5.0+ (for django-cfg integration)

---

## 🌟 Success Metrics

**After using django-ipc, you should be able to:**

✅ **Beginner Level** (After Quick Start - 5 min):
- Start django-ipc WebSocket server
- Generate TypeScript & Python clients
- Send real-time notifications from Django
- Receive instant updates on frontend

✅ **Intermediate Level** (After Integration - 30 min):
- Integrate django-ipc into Django project
- Use Django signals for automatic notifications
- Implement 4 notification patterns (user, room, broadcast, multi-user)
- Deploy with Docker

✅ **Advanced Level** (After Production - 1 hour):
- Deploy multiple django-ipc servers with load balancing
- Configure JWT authentication
- Set up monitoring and health checks
- Scale to 10,000+ concurrent users

---

## 📊 Comparison

**django-ipc vs Traditional Real-Time Django:**

| Feature | Polling (Traditional) | Django Channels | **django-ipc** |
|---------|----------------------|-----------------|----------------|
| **Setup Time** | 🟡 2 days | ⚠️ 3 weeks | ✅ **5 minutes** |
| **Client Code** | ⚠️ Manual | ⚠️ Manual | ✅ **Auto-generated** |
| **Type Safety** | ❌ None | ⚠️ Partial | ✅ **100% Pydantic v2** |
| **Requests/Day** | ❌ 28,800 | ✅ 1 connection | ✅ **1 connection** |
| **Latency** | ⚠️ 5-60s | ✅ <100ms | ✅ **<50ms** |
| **Django Integration** | ✅ Easy | 🟡 Complex | ✅ **3 lines of code** |
| **Scaling** | ❌ Server load | 🟡 Complex | ✅ **Horizontal** |
| **Production Ready** | ⚠️ Manual | 🟡 Requires work | ✅ **Out of the box** |

**[📚 Full comparison guide →](https://djangocfg.com/docs/features/integrations/websocket-ipc/why-websocket-rpc)**

---

## 🤝 Community & Support

### Resources

- 🌐 **[djangocfg.com](https://djangocfg.com/)** - Official website & docs
- 📚 **[WebSocket RPC Docs](https://djangocfg.com/docs/features/integrations/websocket-ipc)** - Complete documentation
- 🐙 **[GitHub](https://github.com/markolofsen/django-ipc)** - Source code & issues
- 💬 **[Discussions](https://github.com/markolofsen/django-ipc/discussions)** - Community support

### Links

- **[🎯 Live Demo](http://demo.djangocfg.com)** - See django-ipc in action
- **[📦 PyPI](https://pypi.org/project/django-ipc/)** - Package repository
- **[🚀 django-cfg](https://github.com/markolofsen/django-cfg)** - Parent framework

---

## 📄 License

**MIT License** - Free for commercial use

---

**Built with ❤️ for the django-cfg ecosystem**

---

<div align="center">

**Django WebSocket RPC** • **Real-Time Django** • **Type-Safe IPC** • **Auto-Generated Clients**

django-ipc is the production-ready WebSocket RPC framework for Django. Replace polling with real-time WebSocket connections, auto-generate type-safe clients, and scale to 10,000+ users. Perfect for Django real-time notifications, live chat, dashboard updates, and any Django WebSocket use case.

**Keywords**: django websocket rpc, django real-time, websocket server python, django ipc, type-safe websocket, django notifications, real-time django framework, websocket auto-generate client, django redis websocket, pydantic websocket

---

**Get Started:** **[5-Min Quick Start](https://djangocfg.com/docs/features/integrations/websocket-ipc/quick-start)** • **[Full Documentation](https://djangocfg.com/docs/features/integrations/websocket-ipc)** • **[Live Demo](http://demo.djangocfg.com)**

</div>
