"""Module to bridge this package and the abcvoting package.

This module defines functions to convert profiles from this package to the abcvoting package, allowing to parse
preference files generated by the abcvoting package.
"""

from abcvoting.fileio import read_abcvoting_yaml_file
from abcvoting.preferences import Profile

from trivoting.election.alternative import Alternative
from trivoting.election.trichotomous_ballot import TrichotomousBallot
from trivoting.election.trichotomous_profile import TrichotomousProfile


def abcvoting_to_trichotomous_profile(abc_profile: Profile) -> TrichotomousProfile:
    """
    Converts a profile from the abcvoting library into a trichotomous profile.

    Parameters
    ----------
    abc_profile : Profile
        The input profile using abcvoting's approval-based format.

    Returns
    -------
    TrichotomousProfile
        A trichotomous profile where each ballot has only approved alternatives and the rest are left unspecified.
    """
    alternatives_map = {i: Alternative(a) for i, a in enumerate(abc_profile.cand_names)}
    profile = TrichotomousProfile(alternatives=alternatives_map.values())
    for abc_ballot in abc_profile:
        ballot = TrichotomousBallot(
            approved=[alternatives_map[a] for a in abc_ballot.approved]
        )
        profile.add_ballot(ballot)
    return profile


def parse_abcvoting_yaml(file_path: str) -> TrichotomousProfile:
    """
    Parses a YAML file formatted for the abcvoting library and returns the corresponding trichotomous profile.

    Parameters
    ----------
    file_path : str
        Path to the YAML file containing the approval-based voting profile.

    Returns
    -------
    TrichotomousProfile
        The parsed trichotomous profile including its maximum selection size.
    """
    abc_profile, max_size_selection, _, _ = read_abcvoting_yaml_file(file_path)
    profile = abcvoting_to_trichotomous_profile(abc_profile)
    profile.max_size_selection = max_size_selection
    return profile
