import { PerformanceMode } from 'aws-cdk-lib/aws-efs';
import { EventBus, Rule } from 'aws-cdk-lib/aws-events';
import { ArnPrincipal, PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { IDestination } from 'aws-cdk-lib/aws-lambda';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { Queue } from 'aws-cdk-lib/aws-sqs';
import { Construct } from 'constructs';
/**
 * Interface for ServerlessClamscan Virus Definitions S3 Bucket Logging.
 */
export interface ServerlessClamscanLoggingProps {
    /**
     * Destination bucket for the server access logs (Default: Creates a new S3 Bucket for access logs).
     */
    readonly logsBucket?: boolean | IBucket;
    /**
     * Optional log file prefix to use for the bucket's access logs, option is ignored if logs_bucket is set to false.
     */
    readonly logsPrefix?: string;
}
/**
 * Interface for creating a ServerlessClamscan.
 */
export interface ServerlessClamscanProps {
    /**
     * An optional list of S3 buckets to configure for ClamAV Virus Scanning; buckets can be added later by calling addSourceBucket.
     */
    readonly buckets?: IBucket[];
    /**
     * Optionally set a reserved concurrency for the virus scanning Lambda.
     * @see https://docs.aws.amazon.com/lambda/latest/operatorguide/reserved-concurrency.html
     */
    readonly reservedConcurrency?: number;
    /**
     * Optionally set the memory allocation for the scan function. Note that low memory allocations may cause errors. (Default: 10240).
     * @see https://docs.aws.amazon.com/lambda/latest/operatorguide/computing-power.html
     */
    readonly scanFunctionMemorySize?: number;
    /**
     * The Lambda Destination for files marked 'CLEAN' or 'INFECTED' based on the ClamAV Virus scan or 'N/A' for scans triggered by S3 folder creation events marked (Default: Creates and publishes to a new Event Bridge Bus if unspecified).
     */
    readonly onResult?: IDestination;
    /**
     * The Lambda Destination for files that fail to scan and are marked 'ERROR' or stuck 'IN PROGRESS' due to a Lambda timeout (Default: Creates and publishes to a new SQS queue if unspecified).
     */
    readonly onError?: IDestination;
    /**
     * Whether or not to enable encryption on EFS filesystem (Default: enabled).
     */
    readonly efsEncryption?: boolean;
    /**
     * Set the performance mode of the EFS file system (Default: GENERAL_PURPOSE).
     */
    readonly efsPerformanceMode?: PerformanceMode;
    /**
     * Whether or not to enable Access Logging for the Virus Definitions bucket, you can specify an existing bucket and prefix (Default: Creates a new S3 Bucket for access logs).
     */
    readonly defsBucketAccessLogsConfig?: ServerlessClamscanLoggingProps;
    /**
     * Allows the use of imported buckets. When using imported buckets the user is responsible for adding the required policy statement to the bucket policy: `getPolicyStatementForBucket()` can be used to retrieve the policy statement required by the solution.
     */
    readonly acceptResponsibilityForUsingImportedBucket?: boolean;
}
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
export declare class ServerlessClamscan extends Construct {
    /**
      The Lambda Destination for failed on erred scans [ERROR, IN PROGRESS (If error is due to Lambda timeout)].
     */
    readonly errorDest: IDestination;
    /**
      The Lambda Destination for completed ClamAV scans [CLEAN, INFECTED].
     */
    readonly resultDest: IDestination;
    /**
      Conditional: The SQS Queue for erred scans if a failure (onError) destination was not specified.
     */
    readonly errorQueue?: Queue;
    /**
      Conditional: The SQS Dead Letter Queue for the errorQueue if a failure (onError) destination was not specified.
     */
    readonly errorDeadLetterQueue?: Queue;
    /**
      Conditional: The Event Bridge Bus for completed ClamAV scans if a success (onResult) destination was not specified.
     */
    readonly resultBus?: EventBus;
    /**
      Conditional: An Event Bridge Rule for files that are marked 'CLEAN' by ClamAV if a success destination was not specified.
     */
    readonly cleanRule?: Rule;
    /**
      Conditional: An Event Bridge Rule for files that are marked 'INFECTED' by ClamAV if a success destination was not specified.
     */
    readonly infectedRule?: Rule;
    /**
      Conditional: The Bucket for access logs for the virus definitions bucket if logging is enabled (defsBucketAccessLogsConfig).
     */
    readonly defsAccessLogsBucket?: IBucket;
    /**
      Conditional: When true, the user accepted the responsibility for using imported buckets.
     */
    readonly useImportedBuckets?: boolean;
    private _scanFunction;
    private _s3Gw;
    private _efsRootPath;
    private _efsMountPath;
    private _efsDefsPath;
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope: Construct, id: string, props: ServerlessClamscanProps);
    /**
     * @returns ArnPrincipal the ARN of the assumed role principal for the scan function
     */
    get scanAssumedPrincipal(): ArnPrincipal;
    /**
     * Returns the statement that should be added to the bucket policy
       in order to prevent objects to be accessed when they are not clean
       or there have been scanning errors: this policy should be added
       manually if external buckets are passed to addSourceBucket()
     * @param bucket The bucket which you need to protect with the policy
     * @returns PolicyStatement the policy statement if available
     */
    getPolicyStatementForBucket(bucket: IBucket): PolicyStatement;
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket: IBucket): void;
}
