"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_lambda_destinations_1 = require("aws-cdk-lib/aws-lambda-destinations");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_notifications_1 = require("aws-cdk-lib/aws-s3-notifications");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const constructs_1 = require("constructs");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends constructs_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        this.useImportedBuckets = props.acceptResponsibilityForUsingImportedBucket;
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: props.efsPerformanceMode ?? aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = props.defsBucketAccessLogsConfig?.logsBucket;
        const logs_bucket_prefix = props.defsBucketAccessLogsConfig?.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            memorySize: props.scanFunctionMemorySize ?? 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        new aws_cdk_lib_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * @returns ArnPrincipal the ARN of the assumed role principal for the scan function
     */
    get scanAssumedPrincipal() {
        if (this._scanFunction.role) {
            const stack = aws_cdk_lib_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            return new aws_iam_1.ArnPrincipal(scan_assumed_role);
        }
        else {
            throw new Error('The scan function role is undefined');
        }
    }
    /**
     * Returns the statement that should be added to the bucket policy
       in order to prevent objects to be accessed when they are not clean
       or there have been scanning errors: this policy should be added
       manually if external buckets are passed to addSourceBucket()
     * @param bucket The bucket which you need to protect with the policy
     * @returns PolicyStatement the policy statement if available
     */
    getPolicyStatementForBucket(bucket) {
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            return new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                    ArnNotEquals: {
                        'aws:PrincipalArn': [this._scanFunction.role.roleArn, scan_assumed_principal.arn],
                    },
                },
            });
        }
        else {
            throw new Error("Can't generate a valid S3 bucket policy, the scan function role is undefined");
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        bucket.addEventNotification(aws_s3_1.EventType.OBJECT_CREATED, new aws_s3_notifications_1.LambdaDestination(this._scanFunction));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const scan_assumed_principal = this.scanAssumedPrincipal;
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            const result = bucket.addToResourcePolicy(this.getPolicyStatementForBucket(bucket));
            if (!result.statementAdded && !this.useImportedBuckets) {
                throw new Error('acceptResponsibilityForUsingImportedBucket must be set when adding an imported bucket. When using imported buckets the user is responsible for adding the required policy statement to the bucket policy: `getPolicyStatementForBucket()` can be used to retrieve the policy statement required by the solution');
            }
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "cdk-serverless-clamscan.ServerlessClamscan", version: "2.4.145" };
//# sourceMappingURL=data:application/json;base64,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