from tequila.utils import TequilaException
from tequila.apps.unary_state_prep import UnaryStatePrep
from tequila.apps import adapt
from tequila.apps import krylov

def gem(circuits:list, H, variables:dict=None, *args, **kwargs):
    """
    Solve the generalized eigenvalue equation: HC = ESC
    with matries Hij = <i|H|j>, Sij = <i|j> and states |i> = U[i]|00...0> generated by the circuits given
    returns the energies E and vectors C (same format as numpy.linalg.eigh)
    the total wavefunction for energy i is then:
    |E(i)> = \sum_k C[k,i] U[k] |0..0>

    Parameters
    ----------
    circuits: list
        a list of tq.QCircuit that define the effective basis of the GEM
    H: tq.QubitHamiltonian
        a qubit Hamiltonian defining the operator H in the matrix elements Hij
    variables: dict
        a dictionary of variables and their corresponding values (in case the circuits are parametrized)

    Returns
    ----------
    a tuple of eigenenergies E and eigenvectors C in the same format as numpy.linalg.eigh
    """
    
    varnames = sum([U.extract_variables() for U in circuits], [])
    if len(varnames) > 0:
        if variables is None:
            raise TequilaException("GEM method: no variables dictionary given, but circuits have {} variables:\n{}".format(len(varnames),str(varnames)))
        for k in varnames:
            if k not in variables:
                raise TequilaException("GEM method: you gave a parametrized circuit, but the variable {} was not set in the variables dictionary".format(k))

    return krylov.krylov_method(krylov_circs=circuits,H=H,variables=variables) 
