"""Environment file management utilities for credential workflows."""

from __future__ import annotations

from pathlib import Path
from urllib.parse import urlparse

from lib.logging import logger


class EnvFileManager:
    """Handle Automagik Hive environment file access and synchronization."""

    DEFAULT_ENV_TEMPLATE = """# =========================================================================
# ⚡ AUTOMAGIK HIVE - MAIN CONFIGURATION
# =========================================================================
HIVE_ENVIRONMENT=development
HIVE_LOG_LEVEL=INFO
AGNO_LOG_LEVEL=INFO

HIVE_API_HOST=0.0.0.0
HIVE_API_PORT=8886
HIVE_API_WORKERS=1

# Generated by Credential Service
HIVE_DATABASE_URL=postgresql+psycopg://user:pass@localhost:5532/hive
HIVE_API_KEY=hive_generated_key

HIVE_CORS_ORIGINS=http://localhost:3000,http://localhost:8886
HIVE_AUTH_DISABLED=true
HIVE_DEV_MODE=true
HIVE_DEFAULT_MODEL=gpt-4.1-mini
"""

    def __init__(
        self,
        project_root: Path | None = None,
        env_file: Path | None = None,
        alias_file: Path | None = None,
    ) -> None:
        resolved_root = self._resolve_project_root(project_root, env_file)
        self.project_root = resolved_root

        self.primary_env_path = self._resolve_env_path(env_file, default_name=".env")
        alias_path = alias_file or self.primary_env_path.parent / ".env.master"
        self.alias_env_path = alias_path if alias_path.is_absolute() else self.primary_env_path.parent / alias_path

    def _resolve_project_root(self, project_root: Path | None, env_file: Path | None) -> Path:
        if project_root is not None:
            return Path(project_root)
        if env_file is not None and env_file.parent != Path("."):
            return env_file.parent
        return Path.cwd()

    def _resolve_env_path(self, env_file: Path | None, default_name: str) -> Path:
        if env_file is None:
            return self.project_root / default_name
        if env_file.is_absolute():
            return env_file
        return self.project_root / env_file

    @property
    def master_env_path(self) -> Path:
        """Return preferred env path, alias first when available."""
        if self.alias_env_path.exists():
            return self.alias_env_path
        return self.primary_env_path

    def refresh_primary(self, create_if_missing: bool = True) -> Path | None:
        """Ensure primary env file exists and return its path."""
        if self.primary_env_path.exists():
            return self.primary_env_path

        if self.alias_env_path.exists():
            try:
                self.primary_env_path.write_text(self.alias_env_path.read_text())
                logger.debug("Hydrated primary env from alias", file=str(self.primary_env_path))
                return self.primary_env_path
            except OSError as error:
                logger.error(
                    "Failed to hydrate primary env from alias",
                    file=str(self.primary_env_path),
                    alias=str(self.alias_env_path),
                    error=str(error),
                )
                return None

        if not create_if_missing:
            return None

        self._hydrate_from_template()
        return self.primary_env_path if self.primary_env_path.exists() else None

    def _hydrate_from_template(self) -> None:
        template_path = self.project_root / ".env.example"
        try:
            if template_path.exists():
                template_content = template_path.read_text()
                logger.info("Hydrating env from template", template=str(template_path))
            else:
                template_content = self.DEFAULT_ENV_TEMPLATE
                logger.warning(
                    "Template .env.example not found; using default minimal template",
                    file=str(self.primary_env_path),
                )

            self.primary_env_path.write_text(template_content)
            self.sync_alias()
        except OSError as error:
            logger.error(
                "Failed to hydrate env file",
                file=str(self.primary_env_path),
                error=str(error),
            )
            raise

    def sync_alias(self) -> None:
        """Keep alias and primary env files in sync."""
        try:
            if self.primary_env_path.exists():
                content = self.primary_env_path.read_text()
                self.alias_env_path.write_text(content)
                logger.debug("Synchronized alias env", alias=str(self.alias_env_path))
            elif self.alias_env_path.exists():
                content = self.alias_env_path.read_text()
                self.primary_env_path.write_text(content)
                logger.debug("Restored primary env from alias", file=str(self.primary_env_path))
        except OSError as error:
            logger.error(
                "Failed to synchronize env alias",
                primary=str(self.primary_env_path),
                alias=str(self.alias_env_path),
                error=str(error),
            )

    def read_master_lines(self) -> list[str]:
        path = self.master_env_path
        if not path.exists():
            return []
        try:
            return path.read_text().splitlines()
        except OSError as error:
            logger.error("Failed to read env file", file=str(path), error=str(error))
            return []

    def update_values(self, updates: dict[str, str], create_if_missing: bool = True) -> bool:
        """Update key/value pairs in env files, creating from template if needed."""
        target_path = self.refresh_primary(create_if_missing=create_if_missing)
        if target_path is None:
            logger.error(
                "Environment file does not exist and creation disabled",
                file=str(self.primary_env_path),
            )
            return False

        try:
            lines = target_path.read_text().splitlines() if target_path.exists() else []
        except OSError as error:
            logger.error(
                "Failed to read env file before update",
                file=str(target_path),
                error=str(error),
            )
            raise

        updated_keys = set()
        new_lines: list[str] = []
        for line in lines:
            if "=" not in line:
                new_lines.append(line)
                continue
            key, _ = line.split("=", 1)
            if key in updates:
                new_lines.append(f"{key}={updates[key]}")
                updated_keys.add(key)
            else:
                new_lines.append(line)

        for key, value in updates.items():
            if key not in updated_keys:
                new_lines.append(f"{key}={value}")

        content = "\n".join(new_lines)
        if content:
            content += "\n"

        try:
            target_path.write_text(content)
            self.sync_alias()
            return True
        except OSError as error:
            logger.error(
                "Failed to write env file",
                file=str(target_path),
                error=str(error),
            )
            raise

    def extract_postgres_credentials(self, database_url_var: str) -> dict[str, str | None]:
        credentials: dict[str, str | None] = {
            "user": None,
            "password": None,
            "database": None,
            "host": None,
            "port": None,
            "url": None,
        }

        path = self.master_env_path
        if not path.exists():
            logger.warning("Environment file not found", env_file=str(path))
            return credentials

        try:
            for raw_line in path.read_text().splitlines():
                line = raw_line.strip()
                if not line or line.startswith("#"):
                    continue
                if line.startswith(f"{database_url_var}="):
                    url = line.split("=", 1)[1].strip()
                    credentials["url"] = url
                    parsed = urlparse(url)
                    credentials["user"] = parsed.username
                    credentials["password"] = parsed.password
                    credentials["host"] = parsed.hostname
                    credentials["port"] = str(parsed.port) if parsed.port else None
                    if parsed.path and len(parsed.path) > 1:
                        credentials["database"] = parsed.path[1:]
                    logger.info("PostgreSQL credentials extracted from env")
                    break
        except Exception as error:  # noqa: BLE001 - logging unexpected errors
            logger.error(
                "Failed to extract PostgreSQL credentials",
                file=str(path),
                error=str(error),
            )

        return credentials

    def extract_api_key(self, api_key_var: str) -> str | None:
        path = self.master_env_path
        if not path.exists():
            logger.warning("Environment file not found", env_file=str(path))
            return None

        try:
            for raw_line in path.read_text().splitlines():
                line = raw_line.strip()
                if not line or line.startswith("#"):
                    continue
                if line.startswith(f"{api_key_var}="):
                    api_key = line.split("=", 1)[1].strip()
                    if api_key:
                        logger.info("Hive API key extracted from env")
                        return api_key
        except Exception as error:  # noqa: BLE001 - logging unexpected errors
            logger.error("Failed to extract Hive API key", file=str(path), error=str(error))

        return None

    def extract_base_ports(
        self,
        defaults: dict[str, int],
        database_url_var: str,
        api_port_var: str,
    ) -> dict[str, int]:
        base_ports = defaults.copy()
        path = self.master_env_path
        if not path.exists():
            logger.debug("No env file found, using default base ports", defaults=base_ports)
            return base_ports

        try:
            for raw_line in path.read_text().splitlines():
                line = raw_line.strip()
                if not line or line.startswith("#"):
                    continue
                if line.startswith(f"{database_url_var}="):
                    url = line.split("=", 1)[1].strip()
                    if "postgresql+psycopg://" in url:
                        parsed = urlparse(url)
                        if parsed.port:
                            base_ports["db"] = parsed.port
                            logger.debug("Found custom database port in env", port=parsed.port)
                elif line.startswith(f"{api_port_var}="):
                    port_value = line.split("=", 1)[1].strip()
                    try:
                        base_ports["api"] = int(port_value)
                        logger.debug("Found custom API port in env", port=port_value)
                    except ValueError:
                        logger.warning(
                            "Invalid API port in env, using default",
                            invalid_port=port_value,
                        )
        except Exception as error:  # noqa: BLE001 - logging unexpected errors
            logger.error(
                "Failed to extract base ports from env",
                file=str(path),
                error=str(error),
            )

        return base_ports
