"""
Graceful shutdown progress display utility
Adapted from Langflow's cli/progress.py with automagik-hive integration
"""

import platform
import sys
import threading
import time
from collections.abc import Generator
from contextlib import contextmanager
from typing import Any

import click

MIN_DURATION_THRESHOLD = 0.1  # Minimum duration to show in seconds (100ms)


class ShutdownProgress:
    """A CLI progress indicator that shows user-friendly step-by-step shutdown progress.

    Shows animated loading indicators (□ → ■) for each step of the shutdown process.
    """

    def __init__(self, *, verbose: bool = False):
        self.verbose = verbose
        self.steps: list[dict[str, Any]] = []
        self.current_step = 0
        self.running = False
        self._stop_animation = False
        self._animation_thread: threading.Thread | None = None

        # Use Windows-safe characters on Windows to prevent encoding issues
        if platform.system() == "Windows":
            self._animation_chars = ["-", "\\", "|", "/"]  # ASCII spinner
            self._success_icon = "+"  # ASCII plus sign
            self._failure_icon = "x"  # ASCII x
            self._farewell_emoji = ":)"  # ASCII smiley
        else:
            self._animation_chars = ["□", "▢", "▣", "■"]  # Unicode squares
            self._success_icon = "✓"  # Unicode checkmark
            self._failure_icon = "✗"  # Unicode cross
            self._farewell_emoji = "👋"  # Unicode wave

        self._animation_index = 0

    def add_step(self, title: str, description: str = "") -> None:
        """Add a step to the progress indicator."""
        self.steps.append(
            {
                "title": title,
                "description": description,
                "status": "pending",  # pending, running, completed, failed
                "start_time": None,
                "end_time": None,
            }
        )

    def _animate_step(self, step_index: int) -> None:
        """Animate the current step with rotating square characters."""
        if step_index >= len(self.steps):
            return

        step = self.steps[step_index]

        while self.running and step["status"] == "running" and not self._stop_animation:
            # Clear the current line and move cursor to beginning
            sys.stdout.write("\r")

            # Show the animated character
            animation_char = self._animation_chars[self._animation_index]

            # Print the step with animation
            line = f"{animation_char} {step['title']}..."
            sys.stdout.write(line)
            sys.stdout.flush()

            # Update animation
            self._animation_index = (self._animation_index + 1) % len(self._animation_chars)

            time.sleep(0.15)  # Animation speed

    def start_step(self, step_index: int) -> None:
        """Start a specific step and begin animation."""
        if step_index >= len(self.steps):
            return

        self.current_step = step_index
        step = self.steps[step_index]
        step["status"] = "running"
        step["start_time"] = time.time()

        self.running = True
        self._stop_animation = False

        # Start animation in a separate thread
        self._animation_thread = threading.Thread(target=self._animate_step, args=(step_index,))
        self._animation_thread.daemon = True
        self._animation_thread.start()

    def complete_step(self, step_index: int, *, success: bool = True) -> None:
        """Complete a step and stop its animation."""
        if step_index >= len(self.steps):
            return

        step = self.steps[step_index]
        step["status"] = "completed" if success else "failed"
        step["end_time"] = time.time()

        # Stop animation
        self._stop_animation = True
        if self._animation_thread and self._animation_thread.is_alive():
            # Use a shorter timeout to prevent hanging during shutdown
            self._animation_thread.join(timeout=0.2)

            # If thread is still alive, it's a daemon thread and will be cleaned up automatically
            if self._animation_thread.is_alive():
                pass  # Let daemon thread terminate naturally during shutdown

        self.running = False

        # Clear the current line and print final result
        sys.stdout.write("\r")

        if success:
            icon = click.style(self._success_icon, fg="green", bold=True)
            title = click.style(step["title"], fg="green")
        else:
            icon = click.style(self._failure_icon, fg="red", bold=True)
            title = click.style(step["title"], fg="red")

        duration = ""
        if step["start_time"] and step["end_time"]:
            elapsed = step["end_time"] - step["start_time"]
            if self.verbose and elapsed > MIN_DURATION_THRESHOLD:  # Only show duration if verbose and > 100ms
                duration = click.style(f" ({elapsed:.2f}s)", fg="bright_black")

        line = f"{icon} {title}{duration}"
        click.echo(line)

    def fail_step(self, step_index: int, error_msg: str = "") -> None:
        """Mark a step as failed."""
        self.complete_step(step_index, success=False)
        if error_msg and self.verbose:
            click.echo(click.style(f"   Error: {error_msg}", fg="red"))

    @contextmanager
    def step(self, step_index: int) -> Generator[None, None, None]:
        """Context manager for running a step with automatic completion."""
        try:
            self.start_step(step_index)
            yield
            self.complete_step(step_index, success=True)
        except Exception as e:
            error_msg = str(e) if self.verbose else ""
            self.fail_step(step_index, error_msg)
            raise

    def print_shutdown_summary(self) -> None:
        """Print a summary of all completed shutdown steps."""
        if not self.verbose:
            return

        completed_steps = [s for s in self.steps if s["status"] in ["completed", "failed"]]
        if not completed_steps:
            return

        total_time = sum(
            (s["end_time"] - s["start_time"]) for s in completed_steps if s["start_time"] and s["end_time"]
        )

        click.echo()
        click.echo(click.style(f"Total shutdown time: {total_time:.2f}s", fg="bright_black"))

    def print_farewell_message(self) -> None:
        """Print a nice farewell message after shutdown is complete."""
        # Clear any progress indicator output that might be on the current line
        sys.stdout.write("\r")  # Move cursor to beginning of line
        sys.stdout.write(" " * 80)  # Clear the line with spaces
        sys.stdout.write("\r")  # Move cursor back to beginning

        click.echo()
        farewell = click.style(f"{self._farewell_emoji} Server stopped by user", fg="bright_blue", bold=True)
        click.echo(farewell)


def create_automagik_shutdown_progress(*, verbose: bool = False) -> ShutdownProgress:
    """Create a progress indicator with predefined Automagik Hive shutdown steps."""
    progress = ShutdownProgress(verbose=verbose)

    # Define the shutdown steps matching our system architecture
    steps = [
        ("Stopping Server", "Gracefully stopping the web server"),
        ("Cancelling Background Tasks", "Stopping notifications and background jobs"),
        ("Cleaning Up Services", "Closing database connections and services"),
        ("Clearing Temporary Files", "Removing temporary files and cache"),
        ("Finalizing Shutdown", "Completing cleanup and logging"),
    ]

    for title, description in steps:
        progress.add_step(title, description)

    return progress
