"""
Comprehensive tests for V1 router integration and endpoint routing.

Tests router composition, authentication, sub-router integration,
and error handling patterns for TDD development workflow.
"""

from unittest.mock import patch

import pytest
from fastapi import status
from httpx import AsyncClient


class TestV1RouterComposition:
    """Test suite for V1 router composition and structure."""

    def test_v1_router_module_imports(self):
        """Test V1 router module imports successfully."""
        from api.routes.v1_router import v1_router

        assert v1_router is not None
        assert v1_router.prefix == "/api/v1"

    def test_v1_router_includes_health_router(self, test_client):
        """Test V1 router includes health check endpoints."""
        # Health endpoint should be available at /api/v1/health
        response = test_client.get("/api/v1/health")

        assert response.status_code == status.HTTP_200_OK
        data = response.json()
        assert data["status"] == "success"
        assert data["service"] == "Automagik Hive Multi-Agent System"

    def test_v1_router_includes_version_router(self, test_client):
        """Test V1 router includes version management endpoints."""
        # Version endpoints should be available under /api/v1/version
        response = test_client.get("/api/v1/version/components")

        # Should return version data (may be empty but endpoint exists)
        assert response.status_code in [status.HTTP_200_OK, status.HTTP_500_INTERNAL_SERVER_ERROR]

    def test_v1_router_includes_mcp_router(self, test_client, mock_mcp_tools):
        """Test V1 router includes MCP management endpoints."""
        # MCP status endpoint should be available
        response = test_client.get("/api/v1/mcp/status")

        assert response.status_code == status.HTTP_200_OK
        data = response.json()
        assert "status" in data
        assert "available_servers" in data

    def test_v1_router_prefix_applied_correctly(self, test_client):
        """Test all sub-routes have /api/v1 prefix applied."""
        # Test health endpoint at correct prefix
        response = test_client.get("/api/v1/health")
        assert response.status_code == status.HTTP_200_OK

        # Test dual endpoint configuration - we intentionally have health at both levels
        response = test_client.get("/health")
        assert response.status_code == status.HTTP_200_OK  # Both root and v1 health exist for compatibility

        # Test that other v1-specific endpoints don't exist at root level
        response = test_client.get("/version/components")
        assert response.status_code == status.HTTP_404_NOT_FOUND

        response = test_client.get("/mcp/status")
        assert response.status_code == status.HTTP_404_NOT_FOUND

    def test_v1_router_sub_router_isolation(self, test_client):
        """Test sub-routers are properly isolated and composed."""
        from api.routes.v1_router import v1_router

        # Check router composition
        assert len(v1_router.routes) >= 3  # Health, version, MCP routers

        # Verify routes include expected patterns
        route_paths = [route.path for route in v1_router.routes if hasattr(route, "path")]
        health_routes = [path for path in route_paths if "/health" in path]
        version_routes = [path for path in route_paths if "/version" in path]
        mcp_routes = [path for path in route_paths if "/mcp" in path]

        assert len(health_routes) >= 1
        assert len(version_routes) >= 1
        assert len(mcp_routes) >= 1


class TestV1RouterAuthentication:
    """Test suite for V1 router authentication integration."""

    def test_public_endpoints_no_auth_required(self, test_client, mock_auth_service):
        """Test public endpoints work without authentication."""
        # Health check should always be public
        response = test_client.get("/api/v1/health")
        assert response.status_code == status.HTTP_200_OK

        # MCP status should be public for monitoring
        response = test_client.get("/api/v1/mcp/status")
        assert response.status_code == status.HTTP_200_OK

    def test_authenticated_endpoints_with_valid_api_key(self, test_client, api_headers):
        """Test authenticated endpoints accept valid API keys."""
        # Version endpoints might require authentication
        response = test_client.get("/api/v1/version/components", headers=api_headers)

        # Should succeed or fail for business reasons, not auth
        assert response.status_code != status.HTTP_401_UNAUTHORIZED
        assert response.status_code != status.HTTP_403_FORBIDDEN

    def test_auth_service_integration(self, test_client, mock_auth_service):
        """Test router integrates with authentication service."""
        # Configure auth service to require authentication
        mock_auth_service.is_auth_enabled.return_value = True
        mock_auth_service.validate_api_key.return_value = False

        # Requests without valid auth should be handled appropriately
        response = test_client.post("/api/v1/version/components/test/versions")

        # Exact status depends on auth middleware implementation
        assert response.status_code in [
            status.HTTP_401_UNAUTHORIZED,
            status.HTTP_403_FORBIDDEN,
            status.HTTP_400_BAD_REQUEST,  # Missing required data
            status.HTTP_422_UNPROCESSABLE_ENTITY,  # Validation error
        ]

    def test_cors_headers_present(self, test_client):
        """Test CORS headers are properly configured."""
        response = test_client.options("/api/v1/health")

        # Check for CORS headers (may vary based on middleware config)
        cors_headers = [
            "access-control-allow-origin",
            "access-control-allow-methods",
            "access-control-allow-headers",
        ]

        # At least some CORS configuration should be present
        present_headers = [header for header in cors_headers if header in [h.lower() for h in response.headers.keys()]]
        # Either CORS is configured or OPTIONS is handled differently
        assert len(present_headers) >= 0  # Flexible assertion


class TestV1RouterErrorHandling:
    """Test suite for V1 router error handling patterns."""

    def test_invalid_route_returns_404(self, test_client):
        """Test invalid routes return proper 404 errors."""
        invalid_routes = [
            "/api/v1/nonexistent",
            "/api/v1/health/invalid",
            "/api/v1/version/invalid",
            "/api/v1/mcp/invalid",
        ]

        for route in invalid_routes:
            response = test_client.get(route)
            assert response.status_code == status.HTTP_404_NOT_FOUND

    def test_method_not_allowed_returns_405(self, test_client):
        """Test wrong HTTP methods return 405 errors."""
        # Health endpoint only supports GET
        response = test_client.post("/api/v1/health")
        assert response.status_code == status.HTTP_405_METHOD_NOT_ALLOWED

        response = test_client.put("/api/v1/health")
        assert response.status_code == status.HTTP_405_METHOD_NOT_ALLOWED

        response = test_client.delete("/api/v1/health")
        assert response.status_code == status.HTTP_405_METHOD_NOT_ALLOWED

    def test_malformed_request_returns_400(self, test_client):
        """Test malformed requests return appropriate 400 errors."""
        # Send invalid JSON to version endpoint
        response = test_client.post(
            "/api/v1/version/components/test/versions",
            data="invalid json",
            headers={"content-type": "application/json"},
        )

        assert response.status_code in [
            status.HTTP_400_BAD_REQUEST,
            status.HTTP_422_UNPROCESSABLE_ENTITY,
        ]

    def test_server_error_handling(self, test_client):
        """Test server errors are handled gracefully."""
        with patch("api.routes.mcp_router.MCPCatalog") as mock_catalog:
            # Simulate server error in MCP catalog
            mock_catalog.side_effect = Exception("Simulated server error")

            response = test_client.get("/api/v1/mcp/status")

            assert response.status_code == status.HTTP_500_INTERNAL_SERVER_ERROR
            data = response.json()
            assert "Error getting MCP status" in data["detail"]

    def test_error_response_format_consistency(self, test_client):
        """Test error responses follow consistent format."""
        # Test 404 error format
        response = test_client.get("/api/v1/nonexistent")
        assert response.status_code == status.HTTP_404_NOT_FOUND

        # Should return JSON error format
        try:
            error_data = response.json()
            assert "detail" in error_data
        except ValueError:
            # Some 404s might return HTML - that's also acceptable
            pass


class TestV1RouterIntegrationPatterns:
    """Test suite for V1 router integration patterns."""

    @pytest.mark.asyncio
    async def test_async_endpoint_integration(self, async_client: AsyncClient):
        """Test async endpoints work properly through router."""
        # MCP endpoints are async
        response = await async_client.get("/api/v1/mcp/status")

        assert response.status_code == status.HTTP_200_OK
        data = response.json()
        assert "status" in data

    def test_content_type_handling(self, test_client):
        """Test router handles different content types properly."""
        # Test JSON content type
        response = test_client.post(
            "/api/v1/version/components/test/versions",
            json={"component_type": "agent", "version": 1, "config": {}},
            headers={"content-type": "application/json"},
        )

        # Should not fail due to content type issues
        assert response.status_code != status.HTTP_415_UNSUPPORTED_MEDIA_TYPE

    def test_request_id_propagation(self, test_client):
        """Test request IDs are properly propagated through router."""
        # Send request with custom request ID
        headers = {"x-request-id": "test-request-123"}
        response = test_client.get("/api/v1/health", headers=headers)

        assert response.status_code == status.HTTP_200_OK
        # Request ID might be echoed back in headers or logs

    def test_large_request_handling(self, test_client):
        """Test router handles large requests appropriately."""
        # Create a large but valid request
        large_config = {"data": "x" * 1000}  # 1KB config

        response = test_client.post(
            "/api/v1/version/components/test/versions",
            json={"component_type": "agent", "version": 1, "config": large_config, "description": "Large config test"},
        )

        # Should handle large requests gracefully
        assert response.status_code != status.HTTP_413_REQUEST_ENTITY_TOO_LARGE

    def test_concurrent_requests_through_router(self, test_client):
        """Test router handles concurrent requests properly."""
        import concurrent.futures

        def make_health_request():
            return test_client.get("/api/v1/health")

        # Make 5 concurrent requests through router
        with concurrent.futures.ThreadPoolExecutor(max_workers=5) as executor:
            futures = [executor.submit(make_health_request) for _ in range(5)]
            responses = [future.result() for future in futures]

        # All requests should succeed
        for response in responses:
            assert response.status_code == status.HTTP_200_OK

    def test_router_middleware_integration(self, test_client):
        """Test router integrates properly with middleware stack."""
        response = test_client.get("/api/v1/health")

        assert response.status_code == status.HTTP_200_OK

        # Check for common middleware headers
        possible_headers = [
            "content-type",
            "server",
            "date",
            "content-length",
        ]

        [header for header in possible_headers if header in [h.lower() for h in response.headers.keys()]]

        # At least content-type should be present
        assert "content-type" in [h.lower() for h in response.headers.keys()]


class TestV1RouterPerformance:
    """Test suite for V1 router performance characteristics."""

    def test_health_endpoint_response_time(self, test_client):
        """Test health endpoint responds quickly through router."""
        import time

        start_time = time.time()
        response = test_client.get("/api/v1/health")
        end_time = time.time()

        assert response.status_code == status.HTTP_200_OK

        # Health check through router should be fast (under 2 seconds)
        response_time = end_time - start_time
        assert response_time < 2.0, f"Router health check too slow: {response_time}s"

    def test_router_overhead_minimal(self, test_client):
        """Test router adds minimal overhead to requests."""
        # Make multiple requests to check consistency
        response_times = []

        for _ in range(3):
            import time

            start_time = time.time()
            response = test_client.get("/api/v1/health")
            end_time = time.time()

            assert response.status_code == status.HTTP_200_OK
            response_times.append(end_time - start_time)

        # Response times should be consistent (no major variations)
        avg_time = sum(response_times) / len(response_times)
        for time_val in response_times:
            # Each request should be within 200% of average (realistic tolerance for system variance)
            assert abs(time_val - avg_time) < avg_time * 2.0

    def test_router_memory_efficiency(self, test_client):
        """Test router doesn't leak memory with multiple requests."""
        # Make many requests through router
        for _i in range(10):
            response = test_client.get("/api/v1/health")
            assert response.status_code == status.HTTP_200_OK

            # Response should be consistent regardless of request number
            data = response.json()
            assert data["status"] == "success"


class TestV1RouterConfiguration:
    """Test suite for V1 router configuration and setup."""

    def test_router_prefix_configuration(self):
        """Test router prefix is configured correctly."""
        from api.routes.v1_router import v1_router

        assert v1_router.prefix == "/api/v1"
        assert v1_router.prefix.startswith("/api")
        assert "v1" in v1_router.prefix

    def test_sub_router_registration(self):
        """Test all required sub-routers are registered."""
        from api.routes.v1_router import v1_router

        # Check that router has routes (sub-routers add routes)
        assert len(v1_router.routes) >= 3

        # Verify we can access the router object properties
        assert hasattr(v1_router, "routes")
        assert hasattr(v1_router, "prefix")

    def test_router_tags_and_metadata(self):
        """Test router tags and metadata are configured."""
        from api.routes.v1_router import v1_router

        # Router should have basic configuration
        assert hasattr(v1_router, "prefix")
        assert v1_router.prefix == "/api/v1"

    def test_router_route_discovery(self, test_client):
        """Test router exposes expected routes for discovery."""
        # Test route discovery via OpenAPI/docs endpoints
        try:
            response = test_client.get("/docs")
            # Docs endpoint may or may not be enabled
            assert response.status_code in [
                status.HTTP_200_OK,
                status.HTTP_404_NOT_FOUND,
                status.HTTP_401_UNAUTHORIZED,
            ]
        except Exception:  # noqa: S110 - Silent exception handling is intentional
            # Docs endpoint might not be configured - that's acceptable
            pass

    def test_router_health_check_always_available(self, test_client):
        """Test health check is always available through router."""
        response = test_client.get("/api/v1/health")

        assert response.status_code == status.HTTP_200_OK
        data = response.json()

        # Health check should provide router identification
        assert data["status"] == "success"
        assert data["router"] == "health"
        assert data["path"] == "/health"
