"""
Comprehensive test suite for lib/config/yaml_parser.py - targeting 18% to 60%+ coverage.

This test suite covers:
- YAML configuration parsing with MCP support
- Agent and team configuration validation
- MCP tool validation and catalog integration
- Error handling and edge cases
- Configuration updates and validation
- Tool separation (regular vs MCP)
- File operations and error scenarios
"""

import tempfile
from pathlib import Path
from unittest.mock import Mock, patch

import pytest
import yaml

from lib.config.schemas import MCPToolConfig
from lib.config.yaml_parser import YAMLConfigParser
from lib.mcp.catalog import MCPCatalog


@pytest.fixture(autouse=True)
def mock_mcp_catalog_init():
    """Mock MCPCatalog initialization to avoid .mcp.json dependency."""
    with patch("lib.config.yaml_parser.MCPCatalog") as mock_catalog_class:
        mock_instance = Mock(spec=MCPCatalog)
        mock_instance.has_server.return_value = True
        mock_instance.list_servers.return_value = []
        mock_instance.reload_catalog.return_value = None
        mock_catalog_class.return_value = mock_instance
        yield mock_catalog_class


class TestYAMLConfigParserInitialization:
    """Test YAMLConfigParser initialization and setup."""

    def test_init_with_default_mcp_catalog(self, mock_mcp_catalog_init):
        """Test initialization with default MCP catalog."""
        parser = YAMLConfigParser()

        assert parser.mcp_catalog is not None
        # Verify MCPCatalog was instantiated via the mocked class
        mock_mcp_catalog_init.assert_called_once()

    def test_init_with_custom_mcp_catalog(self):
        """Test initialization with custom MCP catalog."""
        custom_catalog = Mock(spec=MCPCatalog)
        parser = YAMLConfigParser(custom_catalog)

        assert parser.mcp_catalog is custom_catalog

    def test_init_with_none_mcp_catalog(self, mock_mcp_catalog_init):
        """Test initialization when None is passed for MCP catalog."""
        parser = YAMLConfigParser(None)

        assert parser.mcp_catalog is not None
        # Verify MCPCatalog was instantiated via the mocked class
        mock_mcp_catalog_init.assert_called()


class TestAgentConfigParsing:
    """Test agent configuration parsing functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    @pytest.fixture
    def valid_agent_config(self):
        """Valid agent configuration for testing."""
        return {
            "agent": {"agent_id": "test-agent", "name": "Test Agent", "version": 1},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "You are a helpful assistant.",
            "tools": ["bash", "python", "mcp.search-repo-docs", "mcp.postgres"],
        }

    @pytest.fixture
    def temp_config_file(self, valid_agent_config):
        """Create temporary config file."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(valid_agent_config, f)
            return f.name

    def test_parse_agent_config_success(self, parser, temp_config_file):
        """Test successful agent configuration parsing."""
        # Setup mock MCP catalog
        parser.mcp_catalog.has_server.return_value = True

        result = parser.parse_agent_config(temp_config_file)

        assert result is not None
        assert hasattr(result, "config")
        assert hasattr(result, "regular_tools")
        assert hasattr(result, "mcp_tools")

        # Verify tool separation
        assert "bash" in result.regular_tools
        assert "python" in result.regular_tools
        assert len(result.mcp_tools) == 2  # search-repo-docs and postgres

    def test_parse_agent_config_file_not_found(self, parser):
        """Test parsing non-existent configuration file."""
        with pytest.raises(FileNotFoundError, match="Configuration file not found"):
            parser.parse_agent_config("/non/existent/config.yaml")

    def test_parse_agent_config_invalid_yaml(self, parser):
        """Test parsing invalid YAML content."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            f.write("invalid: yaml: content: [[[")
            f.flush()

            with pytest.raises(ValueError, match="Invalid YAML"):
                parser.parse_agent_config(f.name)

    def test_parse_agent_config_non_dict_content(self, parser):
        """Test parsing YAML that doesn't contain a dictionary."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(["list", "instead", "of", "dict"], f)
            f.flush()

            with pytest.raises(ValueError, match="Configuration file must contain a YAML object"):
                parser.parse_agent_config(f.name)

    def test_parse_agent_config_invalid_tools_type(self, parser):
        """Test parsing configuration with invalid tools type."""
        config = {
            "agent": {"agent_id": "test", "name": "Test"},
            "tools": "not_a_list",  # Should be a list
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            f.flush()

            with pytest.raises(ValueError, match="'tools' must be a list"):
                parser.parse_agent_config(f.name)

    def test_parse_agent_config_no_tools_section(self, parser):
        """Test parsing configuration without tools section."""
        config = {
            "agent": {"agent_id": "test", "name": "Test"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "Test instructions",
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            f.flush()

            result = parser.parse_agent_config(f.name)

            assert result.regular_tools == []
            assert result.mcp_tools == []

    def test_parse_agent_config_empty_tools_list(self, parser):
        """Test parsing configuration with empty tools list."""
        config = {
            "agent": {"agent_id": "test", "name": "Test"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "Test instructions",
            "tools": [],
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            f.flush()

            result = parser.parse_agent_config(f.name)

            assert result.regular_tools == []
            assert result.mcp_tools == []


class TestTeamConfigParsing:
    """Test team configuration parsing functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser instance with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    @pytest.fixture
    def valid_team_config(self):
        """Valid team configuration for testing."""
        return {
            "team_id": "test-team",
            "name": "Test Team",
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "You are a helpful team.",
        }

    @pytest.fixture
    def temp_team_config_file(self, valid_team_config):
        """Create temporary team config file."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(valid_team_config, f)
            return f.name

    def test_parse_team_config_success(self, parser, temp_team_config_file):
        """Test successful team configuration parsing."""
        result = parser.parse_team_config(temp_team_config_file)

        assert result is not None
        assert hasattr(result, "team_id")
        assert hasattr(result, "name")
        assert hasattr(result, "model")
        assert hasattr(result, "instructions")
        assert result.team_id == "test-team"
        assert result.name == "Test Team"

    def test_parse_team_config_file_not_found(self, parser):
        """Test parsing non-existent team configuration file."""
        with pytest.raises(FileNotFoundError, match="Configuration file not found"):
            parser.parse_team_config("/non/existent/team.yaml")

    def test_parse_team_config_invalid_yaml(self, parser):
        """Test parsing invalid YAML content for team."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            f.write("invalid: yaml: [[[")
            f.flush()

            with pytest.raises(ValueError, match="Invalid YAML"):
                parser.parse_team_config(f.name)

    def test_parse_team_config_non_dict_content(self, parser):
        """Test parsing team YAML that doesn't contain a dictionary."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump("string_instead_of_dict", f)
            f.flush()

            with pytest.raises(ValueError, match="Configuration file must contain a YAML object"):
                parser.parse_team_config(f.name)


class TestToolsParsing:
    """Test tools parsing and separation functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    def test_parse_tools_empty_list(self, parser):
        """Test parsing empty tools list."""
        regular, mcp = parser._parse_tools([])

        assert regular == []
        assert mcp == []

    def test_parse_tools_regular_tools_only(self, parser):
        """Test parsing list with only regular tools."""
        tools = ["bash", "python", "curl", "git"]
        regular, mcp = parser._parse_tools(tools)

        assert regular == ["bash", "python", "curl", "git"]
        assert mcp == []

    def test_parse_tools_mcp_tools_only(self, parser):
        """Test parsing list with only MCP tools."""
        tools = ["mcp.postgres", "mcp.search-repo-docs", "mcp.automagik-forge"]
        regular, mcp = parser._parse_tools(tools)

        assert regular == []
        assert mcp == ["postgres", "search-repo-docs", "automagik-forge"]

    def test_parse_tools_mixed_tools(self, parser):
        """Test parsing list with both regular and MCP tools."""
        tools = ["bash", "mcp.postgres", "python", "mcp.search-repo-docs", "curl"]
        regular, mcp = parser._parse_tools(tools)

        assert regular == ["bash", "python", "curl"]
        assert mcp == ["postgres", "search-repo-docs"]

    def test_parse_tools_invalid_tool_type(self, parser):
        """Test parsing tools list with invalid tool entries."""
        tools = ["bash", 123, "python", None, "mcp.postgres"]

        # Should log warnings but continue processing valid tools
        regular, mcp = parser._parse_tools(tools)

        assert regular == ["bash", "python"]
        assert mcp == ["postgres"]

    def test_parse_tools_empty_mcp_tool_name(self, parser):
        """Test parsing MCP tool with empty server name."""
        tools = ["bash", "mcp.", "python"]

        # Should log warning for empty MCP server name
        regular, mcp = parser._parse_tools(tools)

        assert regular == ["bash", "python"]
        assert mcp == []  # Empty MCP tool name should be ignored

    def test_parse_tools_whitespace_handling(self, parser):
        """Test parsing tools with whitespace."""
        tools = ["  bash  ", "  mcp.postgres  ", "python"]
        regular, mcp = parser._parse_tools(tools)

        assert regular == ["bash", "python"]
        assert mcp == ["postgres"]


class TestMCPToolValidation:
    """Test MCP tool validation functionality."""

    @pytest.fixture
    def parser_with_mock_catalog(self):
        """Create parser with fully mocked MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog), mock_catalog

    def test_validate_mcp_tools_all_valid(self, parser_with_mock_catalog):
        """Test validation when all MCP tools are valid."""
        parser, mock_catalog = parser_with_mock_catalog
        mock_catalog.has_server.return_value = True

        mcp_tools = ["postgres", "search-repo-docs", "automagik-forge"]
        result = parser._validate_mcp_tools(mcp_tools)

        assert len(result) == 3
        for tool_config in result:
            assert isinstance(tool_config, MCPToolConfig)
            assert tool_config.enabled is True

    def test_validate_mcp_tools_some_invalid(self, parser_with_mock_catalog):
        """Test validation when some MCP tools are invalid."""
        parser, mock_catalog = parser_with_mock_catalog

        def mock_has_server(server_name):
            return server_name in ["postgres", "search-repo-docs"]

        mock_catalog.has_server.side_effect = mock_has_server

        mcp_tools = ["postgres", "non-existent-server", "search-repo-docs"]
        result = parser._validate_mcp_tools(mcp_tools)

        assert len(result) == 3
        assert result[0].enabled is True  # postgres
        assert result[1].enabled is False  # non-existent-server
        assert result[2].enabled is True  # search-repo-docs

    def test_validate_mcp_tools_empty_list(self, parser_with_mock_catalog):
        """Test validation with empty MCP tools list."""
        parser, mock_catalog = parser_with_mock_catalog

        result = parser._validate_mcp_tools([])

        assert result == []

    def test_validate_mcp_tools_catalog_error(self, parser_with_mock_catalog):
        """Test validation when MCP catalog raises exceptions."""
        parser, mock_catalog = parser_with_mock_catalog
        mock_catalog.has_server.side_effect = Exception("Catalog error")

        mcp_tools = ["postgres", "search-repo-docs"]
        result = parser._validate_mcp_tools(mcp_tools)

        # Should continue processing despite errors
        assert result == []  # All tools skipped due to errors


class TestConfigurationUpdates:
    """Test configuration update functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser instance with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    @pytest.fixture
    def temp_config_file(self):
        """Create temporary config file for update tests."""
        config = {
            "agent": {"agent_id": "test", "name": "Test Agent"},
            "tools": ["bash", "python"],
            "memory": {"enabled": True},
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            return f.name

    def test_update_agent_config_success(self, parser, temp_config_file):
        """Test successful configuration update."""
        updates = {"memory": {"enabled": False, "max_size": 1000}, "new_section": {"new_param": "new_value"}}

        # Should not raise exception
        parser.update_agent_config(temp_config_file, updates)

        # Verify updates were applied
        with open(temp_config_file) as f:
            updated_config = yaml.safe_load(f)

        assert updated_config["memory"]["enabled"] is False
        assert updated_config["memory"]["max_size"] == 1000
        assert updated_config["new_section"]["new_param"] == "new_value"

    def test_update_agent_config_file_not_found(self, parser):
        """Test updating non-existent configuration file."""
        with pytest.raises(FileNotFoundError, match="Configuration file not found"):
            parser.update_agent_config("/non/existent.yaml", {"test": "value"})

    def test_update_agent_config_read_error(self, parser):
        """Test update when config file cannot be read."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            f.write("invalid: yaml: [[[")
            f.flush()

            with pytest.raises(ValueError, match="Error updating configuration file"):
                parser.update_agent_config(f.name, {"test": "value"})

    def test_update_agent_config_write_error(self, parser):
        """Test update when config file cannot be written."""
        # Create read-only file
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump({"test": "config"}, f)
            f.flush()

            # Make file read-only
            Path(f.name).chmod(0o444)

            try:
                with pytest.raises(ValueError, match="Error updating configuration file"):
                    parser.update_agent_config(f.name, {"test": "new_value"})
            finally:
                # Restore permissions for cleanup
                Path(f.name).chmod(0o644)


class TestMCPToolsSummary:
    """Test MCP tools summary functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser instance with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    @pytest.fixture
    def mock_agent_config_mcp(self):
        """Create mock AgentConfigMCP object."""
        mock_config = Mock()
        mock_config.all_tools = ["bash", "python", "mcp_tool_1", "mcp_tool_2"]
        mock_config.regular_tools = ["bash", "python"]

        # Create mock MCP tools
        mcp_tool_1 = Mock()
        mcp_tool_1.server_name = "postgres"
        mcp_tool_1.enabled = True

        mcp_tool_2 = Mock()
        mcp_tool_2.server_name = "search-repo-docs"
        mcp_tool_2.enabled = False

        mock_config.mcp_tools = [mcp_tool_1, mcp_tool_2]
        mock_config.mcp_server_names = ["postgres", "search-repo-docs"]

        return mock_config

    def test_get_mcp_tools_summary(self, parser, mock_agent_config_mcp):
        """Test MCP tools summary generation."""
        summary = parser.get_mcp_tools_summary(mock_agent_config_mcp)

        assert summary["total_tools"] == 4
        assert summary["regular_tools"] == 2
        assert summary["mcp_tools"] == 2
        assert summary["mcp_servers"] == ["postgres", "search-repo-docs"]
        assert summary["enabled_mcp_tools"] == ["postgres"]
        assert summary["disabled_mcp_tools"] == ["search-repo-docs"]


class TestConfigValidation:
    """Test configuration validation functionality."""

    @pytest.fixture
    def parser(self):
        """Create parser with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        mock_catalog.has_server.return_value = True
        return YAMLConfigParser(mock_catalog)

    @pytest.fixture
    def valid_config_file(self):
        """Create valid configuration file."""
        config = {
            "agent": {"agent_id": "test-agent", "name": "Test Agent", "version": 1},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "You are helpful.",
            "tools": ["bash", "mcp.postgres"],
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            return f.name

    @pytest.mark.skip(
        reason="Blocked by task-8cd5f0e0-1211-4b95-822c-047102ce9dec - source code bug: yaml_parser.py accessing wrong agent attributes"
    )
    def test_validate_config_file_success(self, parser, valid_config_file):
        """Test successful configuration validation."""
        result = parser.validate_config_file(valid_config_file)

        # Debug: print result if test fails
        if not result["valid"]:
            pass

        assert result["valid"] is True
        assert result["config_path"] == valid_config_file
        assert result["agent_id"] == "test-agent"
        assert result["version"] == 1  # Version is integer in schema
        assert "tools_summary" in result
        assert result["errors"] == []
        assert result["warnings"] == []

    def test_validate_config_file_invalid(self, parser):
        """Test validation of invalid configuration file."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            f.write("invalid: yaml: [[[")
            f.flush()

            result = parser.validate_config_file(f.name)

            assert result["valid"] is False
            assert result["config_path"] == f.name
            assert result["agent_id"] is None
            assert result["version"] is None
            assert result["tools_summary"] is None
            assert len(result["errors"]) > 0
            assert result["warnings"] == []

    def test_validate_config_file_not_found(self, parser):
        """Test validation of non-existent file."""
        result = parser.validate_config_file("/non/existent.yaml")

        assert result["valid"] is False
        assert "Configuration file not found" in str(result["errors"])


class TestMCPCatalogOperations:
    """Test MCP catalog integration operations."""

    @pytest.fixture
    def parser_with_mock_catalog(self):
        """Create parser with mock catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog), mock_catalog

    def test_reload_mcp_catalog(self, parser_with_mock_catalog):
        """Test MCP catalog reload functionality."""
        parser, mock_catalog = parser_with_mock_catalog

        parser.reload_mcp_catalog()

        mock_catalog.reload_catalog.assert_called_once()

    def test_string_representation(self, parser_with_mock_catalog):
        """Test string representation of parser."""
        parser, mock_catalog = parser_with_mock_catalog
        mock_catalog.list_servers.return_value = ["server1", "server2", "server3"]

        str_repr = str(parser)

        assert "YAMLConfigParser" in str_repr
        assert "mcp_servers=3" in str_repr


class TestErrorHandlingEdgeCases:
    """Test comprehensive error handling and edge cases."""

    @pytest.fixture
    def parser(self):
        """Create parser instance with mock MCP catalog."""
        mock_catalog = Mock(spec=MCPCatalog)
        return YAMLConfigParser(mock_catalog)

    def test_parse_agent_config_with_unicode_content(self, parser):
        """Test parsing configuration with unicode content."""
        config = {
            "agent": {"agent_id": "test-😀", "name": "Test Agent 🚀"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "You are helpful 👋",
            "tools": ["bash"],
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False, encoding="utf-8") as f:
            yaml.dump(config, f, allow_unicode=True)
            f.flush()

            result = parser.parse_agent_config(f.name)

            assert result.config.agent.agent_id == "test-😀"
            assert result.config.agent.name == "Test Agent 🚀"

    def test_parse_agent_config_large_file(self, parser):
        """Test parsing large configuration file."""
        # Create config with many tools
        tools = [f"tool_{i}" for i in range(1000)]  # Large tools list
        config = {
            "agent": {"agent_id": "large-config", "name": "Large Config"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "Test instructions",
            "tools": tools,
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            f.flush()

            result = parser.parse_agent_config(f.name)

            assert len(result.regular_tools) == 1000

    def test_parse_tools_with_malformed_mcp_entries(self, parser):
        """Test parsing tools with various malformed MCP entries."""
        tools = [
            "bash",
            "mcp.",  # Empty server name
            "mcp.valid-server",
            "mcp..invalid",  # Double dot
            "mcp.",  # Another empty server name
            "python",
        ]

        regular, mcp = parser._parse_tools(tools)

        assert "bash" in regular
        assert "python" in regular
        assert "valid-server" in mcp
        # Malformed entries should be handled gracefully

    def test_concurrent_config_parsing(self, parser):
        """Test parser behavior under concurrent access."""
        import threading

        config = {
            "agent": {"agent_id": "concurrent", "name": "Concurrent Test"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "Test instructions",
            "tools": ["bash", "python"],
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(config, f)
            f.flush()

            results = []
            errors = []

            def parse_config():
                try:
                    result = parser.parse_agent_config(f.name)
                    results.append(result)
                except Exception as e:
                    errors.append(e)

            # Create multiple threads
            threads = [threading.Thread(target=parse_config) for _ in range(5)]

            # Start all threads
            for thread in threads:
                thread.start()

            # Wait for completion
            for thread in threads:
                thread.join()

            # Should have successful results and no errors
            assert len(results) == 5
            assert len(errors) == 0

    def test_memory_usage_large_config(self, parser):
        """Test memory usage with very large configurations."""
        # Create configuration with nested structures
        large_config = {
            "agent": {"agent_id": "memory-test", "name": "Memory Test"},
            "model": {"provider": "openai", "id": "gpt-4"},
            "instructions": "Test instructions",
            "large_section": {
                f"param_{i}": {"nested_param": f"value_{i}", "list_param": [f"item_{j}" for j in range(10)]}
                for i in range(100)
            },
            "tools": [f"tool_{i}" for i in range(100)],
        }

        with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
            yaml.dump(large_config, f)
            f.flush()

            # Should handle large configs without issues
            result = parser.parse_agent_config(f.name)

            assert result is not None
            assert len(result.regular_tools) == 100
