import os
from copy import deepcopy
from textwrap import dedent

import pytest
from funcy import lsplit

from dvc.cli import main
from dvc.dvcfile import LOCK_FILE, PROJECT_FILE
from dvc.exceptions import CyclicGraphError, ReproductionError
from dvc.stage import PipelineStage
from dvc.stage.exceptions import StageNotFound


def test_non_existing_stage_name(tmp_dir, dvc, run_copy):
    tmp_dir.gen("file1", "file1")
    run_copy("file1", "file2", name="copy-file1-file2")

    with pytest.raises(StageNotFound):
        dvc.freeze(":copy-file1-file3")

    assert main(["freeze", ":copy-file1-file3"]) != 0


def test_repro_frozen(tmp_dir, dvc, run_copy):
    (data_stage,) = tmp_dir.dvc_gen("data", "foo")
    stage0 = run_copy("data", "stage0", name="copy-data-stage0")
    run_copy("stage0", "stage1", name="copy-data-stage1")
    run_copy("stage1", "stage2", name="copy-data-stage2")

    dvc.freeze("copy-data-stage1")

    tmp_dir.gen("data", "bar")
    stages = dvc.reproduce()
    assert stages == [data_stage, stage0]


def test_downstream(tmp_dir, dvc):
    # The dependency graph should look like this:
    #
    #       E
    #      / \
    #     D   F
    #    / \   \
    #   B   C   G
    #    \ /
    #     A
    #
    assert main(["run", "-n", "A-gen", "-o", "A", "echo A>A"]) == 0
    assert main(["run", "-n", "B-gen", "-d", "A", "-o", "B", "echo B>B"]) == 0
    assert main(["run", "--single-stage", "-d", "A", "-o", "C", "echo C>C"]) == 0
    assert (
        main(["run", "-n", "D-gen", "-d", "B", "-d", "C", "-o", "D", "echo D>D"]) == 0
    )
    assert main(["run", "--single-stage", "-o", "G", "echo G>G"]) == 0
    assert main(["run", "-n", "F-gen", "-d", "G", "-o", "F", "echo F>F"]) == 0
    assert (
        main(
            [
                "run",
                "--single-stage",
                "-d",
                "D",
                "-d",
                "F",
                "-o",
                "E",
                "echo E>E",
            ]
        )
        == 0
    )

    # We want the evaluation to move from B to E
    #
    #       E
    #      /
    #     D
    #    /
    #   B
    #
    evaluation = dvc.reproduce(PROJECT_FILE + ":B-gen", downstream=True, force=True)

    assert len(evaluation) == 3
    assert isinstance(evaluation[0], PipelineStage)
    assert evaluation[0].relpath == PROJECT_FILE
    assert evaluation[0].name == "B-gen"

    assert isinstance(evaluation[1], PipelineStage)
    assert evaluation[1].relpath == PROJECT_FILE
    assert evaluation[1].name == "D-gen"

    assert not isinstance(evaluation[2], PipelineStage)
    assert evaluation[2].relpath == "E.dvc"

    # B, C should be run (in any order) before D
    # See https://github.com/iterative/dvc/issues/3602
    evaluation = dvc.reproduce(PROJECT_FILE + ":A-gen", downstream=True, force=True)

    assert len(evaluation) == 5
    assert isinstance(evaluation[0], PipelineStage)
    assert evaluation[0].relpath == PROJECT_FILE
    assert evaluation[0].name == "A-gen"

    names = set()
    for stage in evaluation[1:3]:
        if isinstance(stage, PipelineStage):
            assert stage.relpath == PROJECT_FILE
            names.add(stage.name)
        else:
            names.add(stage.relpath)
    assert names == {"B-gen", "C.dvc"}

    assert isinstance(evaluation[3], PipelineStage)
    assert evaluation[3].relpath == PROJECT_FILE
    assert evaluation[3].name == "D-gen"

    assert not isinstance(evaluation[4], PipelineStage)
    assert evaluation[4].relpath == "E.dvc"


def test_repro_when_cmd_changes(tmp_dir, dvc, run_copy, mocker):
    from dvc.dvcfile import ProjectFile

    tmp_dir.gen("foo", "foo")
    stage = run_copy("foo", "bar", name="copy-file")
    target = "copy-file"
    assert not dvc.reproduce(target)

    from dvc.stage.run import cmd_run

    m = mocker.patch("dvc.stage.run.cmd_run", wraps=cmd_run)
    stage.cmd = "  ".join(stage.cmd.split())  # change cmd spacing by two
    ProjectFile(dvc, PROJECT_FILE)._dump_pipeline_file(stage)

    assert dvc.status([target]) == {target: ["changed command"]}
    assert dvc.reproduce(target)[0] == stage
    m.assert_called_once_with(stage, checkpoint_func=None, dry=False, run_env=None)


def test_repro_when_new_deps_is_added_in_dvcfile(tmp_dir, dvc, run_copy, copy_script):
    from dvc.dvcfile import load_file

    tmp_dir.gen({"foo": "foo", "bar": "bar"})
    stage = dvc.run(
        cmd="python copy.py {} {}".format("foo", "foobar"),
        outs=["foobar"],
        deps=["foo"],
        name="copy-file",
    )
    target = PROJECT_FILE + ":copy-file"
    assert not dvc.reproduce(target)

    dvcfile = load_file(dvc, stage.path)
    data, _ = dvcfile._load()
    data["stages"]["copy-file"]["deps"] += ["copy.py"]
    (tmp_dir / stage.path).dump(data)

    assert dvc.reproduce(target)[0] == stage


def test_repro_when_new_outs_is_added_in_dvcfile(tmp_dir, dvc, copy_script):
    from dvc.dvcfile import load_file

    tmp_dir.gen({"foo": "foo", "bar": "bar"})
    stage = dvc.run(
        cmd="python copy.py {} {}".format("foo", "foobar"),
        outs=[],  # scenario where user forgot to add
        deps=["foo"],
        name="copy-file",
    )
    target = ":copy-file"
    assert not dvc.reproduce(target)

    dvcfile = load_file(dvc, stage.path)
    data, _ = dvcfile._load()
    data["stages"]["copy-file"]["outs"] = ["foobar"]
    (tmp_dir / stage.path).dump(data)

    assert dvc.reproduce(target)[0] == stage


def test_repro_when_new_deps_is_moved(tmp_dir, dvc, copy_script):
    from dvc.dvcfile import load_file

    tmp_dir.gen({"foo": "foo", "bar": "foo"})
    stage = dvc.run(
        cmd="python copy.py {} {}".format("foo", "foobar"),
        outs=["foobar"],
        deps=["foo"],
        name="copy-file",
    )
    target = ":copy-file"
    assert not dvc.reproduce(target)

    # hardcode values in source code, ignore sys.argv
    tmp_dir.gen(
        "copy.py",
        """
import shutil

shutil.copyfile('bar', 'foobar')
""",
    )
    from shutil import move

    move("foo", "bar")

    dvcfile = load_file(dvc, stage.path)
    data, _ = dvcfile._load()
    data["stages"]["copy-file"]["deps"] = ["bar"]
    (tmp_dir / stage.path).dump(data)

    assert dvc.reproduce(target)[0] == stage


def test_repro_when_new_out_overlaps_others_stage_outs(tmp_dir, dvc):
    from dvc.exceptions import OverlappingOutputPathsError

    tmp_dir.gen({"dir": {"file1": "file1"}, "foo": "foo"})
    dvc.add("dir")
    (tmp_dir / PROJECT_FILE).dump(
        {
            "stages": {
                "run-copy": {
                    "cmd": "python copy {} {}".format("foo", "dir/foo"),
                    "deps": ["foo"],
                    "outs": ["dir/foo"],
                }
            }
        },
    )
    with pytest.raises(OverlappingOutputPathsError):
        dvc.reproduce(":run-copy")


def test_repro_when_new_deps_added_does_not_exist(tmp_dir, dvc, copy_script):
    tmp_dir.gen("foo", "foo")
    (tmp_dir / PROJECT_FILE).dump(
        {
            "stages": {
                "run-copy": {
                    "cmd": "python copy.py {} {}".format("foo", "foobar"),
                    "deps": ["foo", "bar"],
                    "outs": ["foobar"],
                }
            }
        },
    )
    with pytest.raises(ReproductionError):
        dvc.reproduce(":run-copy")


def test_repro_when_new_outs_added_does_not_exist(tmp_dir, dvc, copy_script):
    tmp_dir.gen("foo", "foo")
    (tmp_dir / PROJECT_FILE).dump(
        {
            "stages": {
                "run-copy": {
                    "cmd": "python copy.py {} {}".format("foo", "foobar"),
                    "deps": ["foo"],
                    "outs": ["foobar", "bar"],
                }
            }
        },
    )
    with pytest.raises(ReproductionError):
        dvc.reproduce(":run-copy")


def test_repro_when_lockfile_gets_deleted(tmp_dir, dvc, copy_script):
    tmp_dir.gen("foo", "foo")
    (tmp_dir / PROJECT_FILE).dump(
        {
            "stages": {
                "run-copy": {
                    "cmd": "python copy.py {} {}".format("foo", "foobar"),
                    "deps": ["foo"],
                    "outs": ["foobar"],
                }
            }
        },
    )
    assert dvc.reproduce(":run-copy")
    assert os.path.exists(LOCK_FILE)

    assert not dvc.reproduce(":run-copy")
    os.unlink(LOCK_FILE)
    stages = dvc.reproduce(":run-copy")
    assert stages
    assert stages[0].relpath == PROJECT_FILE
    assert stages[0].name == "run-copy"


def test_cyclic_graph_error(tmp_dir, dvc, run_copy):
    tmp_dir.gen("foo", "foo")
    run_copy("foo", "bar", name="copy-foo-bar")
    run_copy("bar", "baz", name="copy-bar-baz")
    run_copy("baz", "foobar", name="copy-baz-foobar")

    data = (tmp_dir / PROJECT_FILE).parse()
    data["stages"]["copy-baz-foo"] = {
        "cmd": "echo baz > foo",
        "deps": ["baz"],
        "outs": ["foo"],
    }
    (tmp_dir / PROJECT_FILE).dump(data)
    with pytest.raises(CyclicGraphError):
        dvc.reproduce(":copy-baz-foo")


def test_repro_multiple_params(tmp_dir, dvc):
    from dvc.stage.utils import split_params_deps
    from tests.func.test_run_multistage import supported_params

    (tmp_dir / "params2.yaml").dump(supported_params)
    (tmp_dir / "params.yaml").dump(supported_params)

    (tmp_dir / "foo").write_text("foo")
    stage = dvc.run(
        name="read_params",
        deps=["foo"],
        outs=["bar"],
        params=[
            "params2.yaml:lists,floats,name",
            "answer,floats,nested.nested1",
        ],
        cmd="cat params2.yaml params.yaml > bar",
    )

    params, deps = split_params_deps(stage)
    assert len(params) == 2
    assert len(deps) == 1
    assert len(stage.outs) == 1

    lockfile = stage.dvcfile._lockfile
    assert lockfile.load()["stages"]["read_params"]["params"] == {
        "params2.yaml": {
            "lists": [42, 42.0, "42"],
            "floats": 42.0,
            "name": "Answer",
        },
        "params.yaml": {
            "answer": 42,
            "floats": 42.0,
            "nested.nested1": {"nested2": "42", "nested2-2": 41.99999},
        },
    }
    data, _ = stage.dvcfile._load()
    params = data["stages"]["read_params"]["params"]

    custom, defaults = lsplit(lambda v: isinstance(v, dict), params)
    assert set(custom[0]["params2.yaml"]) == {"name", "lists", "floats"}
    assert set(defaults) == {"answer", "floats", "nested.nested1"}

    assert not dvc.reproduce(stage.addressing)
    params = deepcopy(supported_params)
    params["answer"] = 43
    (tmp_dir / "params.yaml").dump(params)

    assert dvc.reproduce(stage.addressing) == [stage]


@pytest.mark.parametrize("multiline", [True, False])
def test_repro_list_of_commands_in_order(tmp_dir, dvc, multiline):
    cmd = ["echo foo>foo", "echo bar>bar"]
    if multiline:
        cmd = "\n".join(cmd)

    (tmp_dir / "dvc.yaml").dump({"stages": {"multi": {"cmd": cmd}}})

    (tmp_dir / "dvc.yaml").write_text(
        dedent(
            """\
            stages:
              multi:
                cmd:
                - echo foo>foo
                - echo bar>bar
        """
        )
    )
    dvc.reproduce(targets=["multi"])
    assert (tmp_dir / "foo").read_text() == "foo\n"
    assert (tmp_dir / "bar").read_text() == "bar\n"


@pytest.mark.parametrize("multiline", [True, False])
def test_repro_list_of_commands_raise_and_stops_after_failure(tmp_dir, dvc, multiline):
    cmd = ["echo foo>foo", "failed_command", "echo baz>bar"]
    if multiline:
        cmd = "\n".join(cmd)

    (tmp_dir / "dvc.yaml").dump({"stages": {"multi": {"cmd": cmd}}})

    with pytest.raises(ReproductionError):
        dvc.reproduce(targets=["multi"])
    assert (tmp_dir / "foo").read_text() == "foo\n"
    assert not (tmp_dir / "bar").exists()
